/* tkBargraph.c --
 *
 *	This module implements a bargraph widget for the Tk toolkit.
 *
 * Copyright 1992 Victoria University of Wellington, NZ.
 * This code is derived from the tkScale widget.
 * Copyright 1990 Regents of the University of California.
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.	 The University of California
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

/* for now, until we figure out how to config this puppy */
#define HAVE_NR 1

#include "pkg.h"
#include "pfit.h"

#define MAX_LAB_ADJ 200 /* max number of label positions we'll adjust */
                        /* so as to make them fit w/o overlap. */


/*
 *  Default bargraph configuration values
 */

#define DEF_BAR_MODE			"bargraph"
#define DEF_BAR_SLIDER_LENGTH		"10"

#define DEF_BAR_BG_COLOR		NORMAL_BG
#define DEF_BAR_BAR_BG_COLOR		NORMAL_BG
#define DEF_BAR_BAR_FG_COLOR		"red"
#define DEF_BAR_TEXT_COLOR		"black"
#define DEF_BAR_TICK_COLOR		"blue"

/* for monochrome displays */

#define DEF_BAR_BG_MONO			"white"
#define DEF_BAR_BAR_MONO		"black"
#define DEF_BAR_TEXT_MONO		"black"
#define DEF_BAR_TICK_MONO		"black"

#define DEF_BAR_BORDER_WIDTH		"0"
#define DEF_BAR_CURSOR			""
#define DEF_BAR_HIGHLIGHT_WIDTH		"0"
#define DEF_BAR_RELIEF			"raised"

#define DEF_BAR_FONT			DEF_FONT
#define DEF_BAR_DIGITS			"6"
#define DEF_BAR_FROM			"0.0"
#define DEF_BAR_TO			"100.0"
#define DEF_BAR_BASEVALUE		"0.0"
#define DEF_BAR_LENGTH			"100"
#define DEF_BAR_ORIENT			"vertical"
#define DEF_BAR_SHOW_VALUE		"1"
#define DEF_BAR_SHOW_BASE               "true"

#define DEF_BAR_SHOW_WINGS              "false"
#define DEF_BAR_WING_TRIGGER            "0.66"
#define DEF_BAR_SHOW_WINGOVERLAY        "true"
#define DEF_BAR_WING_ALGORITHM		"powell"
#define DEF_BAR_SHOW_MINMAX             "true"
#define DEF_BAR_SHOW_RANGE		"1"
#define DEF_BAR_TICK_INTERVAL		"20.0"
#define DEF_BAR_TICK_LENGTH		"4"
#define DEF_BAR_TICK_WIDTH              "1"
#define DEF_BAR_WING_WIDTH              "3"

#define DEF_BAR_LABEL			""
#define DEF_BAR_WIDTH			"20"
#define DEF_BAR_BAR_BORDER_WIDTH	"2"
#define DEF_BAR_BAR_RELIEF		"groove"
#define DEF_BAR_PADX			"2"
#define DEF_BAR_PADY			"2"

/*
 * A data structure of the following type is kept for each barchart
 * widget managed by this file:
 */

typedef struct {
  Tk_Window tkwin;		/* Window that embodies the entry. NULL
				 * means that the window has been destroyed
				 * but the data structures haven't yet been
				 * cleaned up.*/
  Display *display;		/* Display containing widget.  Used, among
				 * other things, so that resources can be
				 * freed even after tkwin has gone away. */
  Tcl_Interp *interp;		/* Interpreter associated with entry. */
  Tcl_Command widgetCmd;	/* Token for entry's widget command. */

  /* Border stuff */
  Tk_3DBorder border;		/* Structure used to draw 3-D border */
  int borderWidth;		/* Width of 3-D border around window */
  Tk_Cursor cursor;		/* Current cursor for window. */
  int relief;			/* 3-d effect: TK_RELIEF_RAISED, etc. */
  XColor *highlightBgColorPtr;	/* Color for drawing traversal highlight
				 * area when highlight is off. */
  XColor *highlightColorPtr;	/* Color for drawing traversal highlight. */
  int highlightWidth;		/* Width in pixels of highlight to draw
				 * around widget when it has the focus.
				 * <= 0 means don't draw a highlight. */

  /* bargraph stuff */
  int orient;			/* Orientation for window ("vertical" or
				 * "horizontal"). */
  double value;			/* Current value of Bargraph. */
  double baseVal;		/* Value that is the root of the bar (ignored
				 * if making a slider).
				 */
  double toVal_u;		/* as entered by user, before validation */
  double toVal;		/* Value corresponding to right or top of
				 * Bargraph. */
  double fromVal_u;		/* as entered by user, before validation */
  double fromVal;		/* Value corresponding to left or bottom
				 * of Bargraph. */
  double tickInterval_u;	/* as entered by user, before validation */
  double tickInterval;		/* Distance between tick marks;	 0 means
				 * don't display any tick marks. */
  int mode;			/* MODE_BARGRAPH or MODE_SLIDER */
  int sliderlength;		/* length of bar, in pixels: if mode is slider,
				 * we draw a slider of this size.
				 */
  char *label;			/* Label to display above or to right of
				 * Bargraph;  NULL means don't display a
				 * label.  Malloc'ed. */

  /* Information used when displaying widget: */
  int barBorderWidth;		/* Width of bar border */
  int barRelief;		/* Bar relief, e.g., TK_RELIEF_RAISED */
  Tk_3DBorder barBorder;	/* Used for drawing bar background. */
  Tk_3DBorder barColor;		/* Used for drawing bar. */

  Tk_Font tkfont;		/* Information about text font, or NULL. */
  int digits;			/* Precision for formatting values */
  XColor *textColorPtr;		/* Color for drawing text. */
  GC textGC;			/* GC for drawing text. */

  XColor *tickColorPtr;		/* Color for drawing ticks and their labels */
  GC tickGC;			/* GC for drawing ticks */

  GC wingGC;                    /* GC for drawing "wing" (indicator mark) */
  XColor *wingColorPtr;         /* Color for drawing wings */
  int wingWidth;                /* Width of wings, in pixels */
  GC wingoverlayGC;             /* GC for wingline overlaying the main line */

  int width;			/* Desired narrow dimension of Bargraph,
				 * in pixels. */
  int max_height;		/* Desired long dimension of Bargraph,
				 * in pixels. */
  int tickLength;		/* Size of ticks */
  int tickWidth;                /* Width of ticks, in pixels */

  int padX;			/* Horiz padding around strings */
  int padY;			/* Vertical padding around strings */
  int showValue;		/* 1 means to display the Bargraph value */
  int showRange;		/* 1 means to display min and max values */

  int showBase;                 /* !=0 means to display the "base" value. */

  int showWing;                 /* !=0 means to display "wings" on vert
                                 * bargraph when labels are forced to move
				 * to avoid overlaps.
                                 */
  int wingOverlay;          	/* !=0 means that when showWing !0, then we
                                 * also extend wingline through bar.
                                 */
  int wing_Aside;               /* bool: is Aside wing drawn? */
  int wing_Amindist;            /* dist w/in which Aside is drawn */
  XPoint Awing[3];              /* for saving a-side wing locations */
  int wing_Bside;               /* bool: is Bside wing drawn? */
  int wing_Bmindist;            /* dist w/in which Bside is drawn */
  XPoint Bwing[3];              /* for saving b-side wing locations */
  double wingTrigger;           /* trigger wings, in units of font linespace */
  int wingAlgorithm;		/* Algorithm for figuring out where labels
  				 * should be placed; at this writing, "uniform"
				 * and "powell" are known.  NULL means default
				 * = "powell".  Malloc'ed. */

  int minmaxPixels;		/* Number of pixels required for widest text
				 * labelling tickmark.	0 means don't display
				 * ticks. */
  int numTicks;			/* The number of tick marks. */
  XSegment *ticks;		/* The tick marks.  Malloc'd. */

  char *alabelstr_u;		/* The "a" complex labels, as a string,
				 * as entered by user */
  Tk_Font Atkfont;		/* alabel text font, or NULL. */
  GC alabGC;			/* GC for drawing alabels. */
  char *alabelstr;		/* The "a" complex labels, as a string */
  int n_alabels;		/* number of "a" complex labels */
  char **alabels;		/* The "a" complex labels, as a list. */
  int alabelPixels;		/* Number of pixels required for
				 * widest alabel  */

  char *blabelstr_u;		/* The "b" complex labels, as a string, as
				 * entered by user  */
  Tk_Font Btkfont;		/* blabel text font, or NULL. */
  GC blabGC;			/* GC for drawing blabels. */
  char *blabelstr;		/* The "b" complex labels, as a string */
  int n_blabels;		/* number of "b" complex labels */
  char **blabels;		/* The "b" complex labels, as a list. */
  int blabelPixels;		/* Number of pixels required for
				 * widest blabel */

  int valuePixels;		/* Number of pixels required for value text */

  struct {			/* Pixel offsets of various components */
    int bot_edge;		/* left/bottom edge of bar */
    int top_edge;		/* top/right edge of bar */
    int base;			/* base value in bar */
    int label_y;
    int minmax0_x, minmax0_y;
    int minmax1_x, minmax1_y;
    int value_x, value_y;
    int lab_a, lab_b;
    int tick_a, tick_b;
    int bar_x, bar_y;
  } offset;
  struct splineinfo sp_a;       /* for a-labels */
  struct splineinfo sp_b;       /* for b-labels */

  /* Miscellaneous information: */

  double pixval;                /* The last ValueToPixel value, untruncated */



  /* Miscellaneous information: */
  long int flags;		/* Various flags; see below for definitions */

  unsigned int awing_showing : 1;       /* are the A-wings presently visible? */
  unsigned int bwing_showing : 1;       /* are the B-wings presently visible? */

} Bargraph;

/* Handy shorthands */
#define SHOW_TICKS(bgPtr)	((bgPtr)->tickInterval != 0.)
#define SHOW_RANGE(bgPtr)	(SHOW_TICKS(bgPtr) && (bgPtr)->showRange)
#define SHOW_RANGE_OR_BASEV(bgPtr) \
     ((((bgPtr)->baseVal != (bgPtr)->fromVal) && ((bgPtr)->baseVal != (bgPtr)->toVal)) || SHOW_RANGE(bgPtr))
#define SHOW_BASE(BarGrPtr)     ((BarGrPtr)->showBase != 0)
#define SHOW_VALUE(BarGrPtr)    ((BarGrPtr)->showValue != 0)
#define SHOW_WINGS(BarGrPtr)    ((BarGrPtr)->showWing != 0)
#define HAS_ALAB(bgPtr)		((bgPtr)->n_alabels != 0)
#define HAS_BLAB(bgPtr)		((bgPtr)->n_blabels != 0)


/*
 *  flags:
 *  DISPLAY_TICKS - draw the tick points and min/max vals
 *  DISPLAY_BAR - draw the bar
 *  DISPLAY_BORDER - draw the border
 *  DISPLAY_LABELS - draw the alabels and blabels
 *  DISPLAY_LABEL - draw the label
 *  DISPLAY_VALUE - draw the textual value
 */

#define	DISPLAY_TICKS		01
#define	DISPLAY_BAR		02
#define	DISPLAY_BORDER		04
#define	DISPLAY_LABELS		010
#define	DISPLAY_VALUE		020
#define	DISPLAY_LABEL		040
#define	REDRAW_PENDING		0100
#define	CLEAR_NEEDED		0200
#define GOT_FOCUS		0400

#define	DISPLAY_ALL	(DISPLAY_TICKS | DISPLAY_BAR | DISPLAY_BORDER | \
			 DISPLAY_LABELS | DISPLAY_VALUE | DISPLAY_LABEL)

/*
 * Information used for argv parsing.
 */

#define	MODE_BARGRAPH	1
#define	MODE_SLIDER	2
static Cmd_Struct modeVals[]= {
  {"bargraph",	MODE_BARGRAPH},
  {"slider",	MODE_SLIDER},
  {"",		0 }
};

static Tk_CustomOption modeOption = { Cmd_OptionSet, Cmd_OptionGet,
				      (ClientData)(&modeVals) };

static Cmd_Struct algorithmVals[]= {
  {"powell",	ALGORITHM_POWELL},
  {"uniform",	ALGORITHM_UNIFORM},
  {"stupid",	ALGORITHM_UNIFORM},
  {"",		0 }
};

static Tk_CustomOption algorithmOption = { Cmd_OptionSet, Cmd_OptionGet,
				      (ClientData)(&algorithmVals) };

#define	ORIENT_VERTICAL		1
#define	ORIENT_HORIZONTAL	2
static Cmd_Struct orientVals[]= {
  {"vertical",		ORIENT_VERTICAL},
  {"horizontal",	ORIENT_HORIZONTAL},
  {"",		0 }
};

static Tk_CustomOption orientOption = { Cmd_OptionSet, Cmd_OptionGet,
					(ClientData)(&orientVals) };

static Tk_ConfigSpec configSpecs[] = {
  {TK_CONFIG_STRING, "-alabels", "alabels", "Alabels",
   DEF_BAR_LABEL, Tk_Offset(Bargraph, alabelstr_u), 0},
  {TK_CONFIG_FONT, "-alabfont", "Alabfont", "AlabFont",
   DEF_BAR_FONT, Tk_Offset(Bargraph, Atkfont), 0},

  {TK_CONFIG_BORDER, "-background", "background", "Background",
   DEF_BAR_BG_COLOR, Tk_Offset(Bargraph, border),
   TK_CONFIG_COLOR_ONLY},
  {TK_CONFIG_BORDER, "-background", "background", "Background",
   DEF_BAR_BG_MONO, Tk_Offset(Bargraph, border),
   TK_CONFIG_MONO_ONLY},
  {TK_CONFIG_BORDER, "-barbackground", "barbackground", "Background",
   DEF_BAR_BAR_BG_COLOR, Tk_Offset(Bargraph, barBorder),
   TK_CONFIG_COLOR_ONLY},
  {TK_CONFIG_BORDER, "-barbackground", "barbackground", "Background",
   DEF_BAR_BG_MONO, Tk_Offset(Bargraph, barBorder),
   TK_CONFIG_MONO_ONLY},
  {TK_CONFIG_INT, "-barBorderWidth", "barBorderWidth", "barBorderWidth",
   DEF_BAR_BAR_BORDER_WIDTH, Tk_Offset(Bargraph, barBorderWidth), 0},
  {TK_CONFIG_BORDER, "-barcolor", "barcolor", "Foreground",
   DEF_BAR_BAR_FG_COLOR, Tk_Offset(Bargraph, barColor),
   TK_CONFIG_COLOR_ONLY},
  {TK_CONFIG_BORDER, "-barcolor", "barcolor", "Foreground",
   DEF_BAR_BAR_MONO, Tk_Offset(Bargraph, barColor),
   TK_CONFIG_MONO_ONLY},
  {TK_CONFIG_RELIEF, "-barRelief", "barRelief", "BarRelief",
   DEF_BAR_BAR_RELIEF, Tk_Offset(Bargraph, barRelief), 0},
  {TK_CONFIG_DOUBLE, "-base", "base", "Base",
   DEF_BAR_BASEVALUE, Tk_Offset(Bargraph, baseVal), 0},
  {TK_CONFIG_SYNONYM, "-bd", "borderwidth", (char *) NULL, (char *) NULL, 0, 0},
  {TK_CONFIG_SYNONYM, "-bg", "background", (char *) NULL, (char *) NULL, 0, 0},
  {TK_CONFIG_STRING, "-blabels", "blabels", "Blabels",
   DEF_BAR_LABEL, Tk_Offset(Bargraph, blabelstr_u), 0},
  {TK_CONFIG_FONT, "-blabfont", "Blabfont", "BlabFont",
   DEF_BAR_FONT, Tk_Offset(Bargraph, Btkfont), 0},
  {TK_CONFIG_INT, "-borderwidth", "borderwidth", "Borderwidth",
   DEF_BAR_BORDER_WIDTH, Tk_Offset(Bargraph, borderWidth), 0},
  {TK_CONFIG_ACTIVE_CURSOR, "-cursor", "cursor", "Cursor",
   DEF_BAR_CURSOR, Tk_Offset(Bargraph, cursor), TK_CONFIG_NULL_OK},
  {TK_CONFIG_INT, "-digits", "digits", "Digits",
   DEF_BAR_DIGITS, Tk_Offset(Bargraph, digits), 0},
  {TK_CONFIG_SYNONYM, "-fg", "barcolor", (char *) NULL, (char *) NULL, 0, 0},
  {TK_CONFIG_FONT, "-font", "font", "Font",
   DEF_BAR_FONT, Tk_Offset(Bargraph, tkfont), 0},
  {TK_CONFIG_PIXELS, "-height", "height", "Height",
   DEF_BAR_LENGTH, Tk_Offset(Bargraph, max_height), 0},
  {TK_CONFIG_COLOR, "-highlightbackground", "highlightBackground",
   "HighlightBackground", HIGHLIGHT_BG,
   Tk_Offset(Bargraph, highlightBgColorPtr), 0},
  {TK_CONFIG_COLOR, "-highlightcolor", "highlightColor", "HighlightColor",
   HIGHLIGHT, Tk_Offset(Bargraph, highlightColorPtr), 0},
  {TK_CONFIG_PIXELS, "-highlightthickness", "highlightThickness",
   "HighlightThickness",
   DEF_BAR_HIGHLIGHT_WIDTH, Tk_Offset(Bargraph, highlightWidth), 0},
  {TK_CONFIG_SYNONYM, "-length", "height", (char *)NULL, (char *)NULL, 0, 0},
  {TK_CONFIG_DOUBLE, "-to", "to", "To",
   DEF_BAR_TO, Tk_Offset(Bargraph, toVal_u), 0},
  {TK_CONFIG_SYNONYM, "-max", "to", (char *) NULL, (char *) NULL, 0, 0},
  {TK_CONFIG_DOUBLE, "-from", "from", "From",
   DEF_BAR_FROM, Tk_Offset(Bargraph, fromVal_u), 0},
  {TK_CONFIG_SYNONYM, "-min", "from", (char *) NULL, (char *) NULL, 0, 0},
  {TK_CONFIG_CUSTOM, "-mode", "mode", "Mode",
   DEF_BAR_MODE, Tk_Offset(Bargraph, mode), 0, &modeOption},
  {TK_CONFIG_CUSTOM, "-orient", "orient", "Orient",
   DEF_BAR_ORIENT, Tk_Offset(Bargraph, orient), 0, &orientOption},
  {TK_CONFIG_PIXELS, "-padx", "padX", "Pad",
   DEF_BAR_PADX, Tk_Offset(Bargraph, padX), 0},
  {TK_CONFIG_PIXELS, "-pady", "padY", "Pad",
   DEF_BAR_PADY, Tk_Offset(Bargraph, padY), 0},
  {TK_CONFIG_RELIEF, "-relief", "relief", "Relief",
   DEF_BAR_RELIEF, Tk_Offset(Bargraph, relief), 0},
  {TK_CONFIG_BOOLEAN, "-showrange", "showRange", "ShowRange",
   DEF_BAR_SHOW_RANGE, Tk_Offset(Bargraph, showRange), 0},
  {TK_CONFIG_BOOLEAN, "-showvalue", "showValue", "ShowValue",
   DEF_BAR_SHOW_VALUE, Tk_Offset(Bargraph, showValue), 0},
  {TK_CONFIG_PIXELS, "-sliderlength", "sliderlength", "Sliderlength",
   DEF_BAR_SLIDER_LENGTH, Tk_Offset(Bargraph, sliderlength), 0},
  {TK_CONFIG_COLOR, "-textcolor", "textcolor", "Textcolor",
   DEF_BAR_TEXT_COLOR, Tk_Offset(Bargraph, textColorPtr),
   TK_CONFIG_COLOR_ONLY},
  {TK_CONFIG_COLOR, "-textcolor", "textcolor", "Textcolor",
   DEF_BAR_TEXT_MONO, Tk_Offset(Bargraph, textColorPtr),
   TK_CONFIG_MONO_ONLY},
  {TK_CONFIG_COLOR, "-tickcolor", "tickcolor", "Tickcolor",
   DEF_BAR_TICK_COLOR, Tk_Offset(Bargraph, tickColorPtr),
   TK_CONFIG_COLOR_ONLY},
  {TK_CONFIG_COLOR, "-tickcolor", "tickcolor", "Tickcolor",
   DEF_BAR_TICK_MONO, Tk_Offset(Bargraph, tickColorPtr),
   TK_CONFIG_MONO_ONLY},
  {TK_CONFIG_DOUBLE, "-tickinterval", "tickInterval", "TickInterval",
   DEF_BAR_TICK_INTERVAL, Tk_Offset(Bargraph, tickInterval_u), 0},
  {TK_CONFIG_PIXELS, "-ticklength", "tickLength", "TickLength",
   DEF_BAR_TICK_LENGTH, Tk_Offset(Bargraph, tickLength), 0},
  {TK_CONFIG_STRING, "-label", "label", "Label",
   DEF_BAR_LABEL, Tk_Offset(Bargraph, label), TK_CONFIG_NULL_OK},
  {TK_CONFIG_PIXELS, "-width", "width", "Width",
   DEF_BAR_WIDTH, Tk_Offset(Bargraph, width), 0},

  {TK_CONFIG_INT, "-tickwidth", "tickwidth", "Tickwidth",
   DEF_BAR_TICK_WIDTH, Tk_Offset(Bargraph, tickWidth), 0},

  {TK_CONFIG_BOOLEAN, "-showwing", "showWing", "ShowWing",
   DEF_BAR_SHOW_WINGS, Tk_Offset(Bargraph, showWing), 0},
  {TK_CONFIG_INT, "-wingwidth", "wingwidth", "Wingwidth",
   DEF_BAR_WING_WIDTH, Tk_Offset(Bargraph, wingWidth), 0},
  {TK_CONFIG_DOUBLE, "-wingtrigger", "wingTrigger", "WingTrigger",
   DEF_BAR_WING_TRIGGER, Tk_Offset(Bargraph, wingTrigger), 0},
  {TK_CONFIG_BOOLEAN, "-showwingoverlay", "wingOverlay", "WingOverlay",
   DEF_BAR_SHOW_WINGOVERLAY, Tk_Offset(Bargraph, wingOverlay), 0},
  {TK_CONFIG_CUSTOM, "-wingalgorithm", "wingAlgorithm", "WingAlgorithm",
   DEF_BAR_WING_ALGORITHM, Tk_Offset(Bargraph, wingAlgorithm),
   							0, &algorithmOption},
  {TK_CONFIG_COLOR, "-wingcolor", "wingcolor", "Wingcolor",
   DEF_BAR_BAR_FG_COLOR, Tk_Offset(Bargraph, wingColorPtr),
   TK_CONFIG_COLOR_ONLY},
  {TK_CONFIG_COLOR, "-wingcolor", "wingcolor", "Wingcolor",
   DEF_BAR_TICK_MONO, Tk_Offset(Bargraph, wingColorPtr),
   TK_CONFIG_MONO_ONLY},



  
  {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
   (char *) NULL, 0, 0}
};

/*
 * Forward declarations for procedures defined later in this file:
 */

static void	ComputeBargraphGeometry _ANSI_ARGS_((Bargraph *bgp));
static int	ConfigureBargraph _ANSI_ARGS_((Tcl_Interp *interp,
		    Bargraph *bgp, int argc, char **argv,
		    int flags));
static void	DestroyBargraph _ANSI_ARGS_((char *clientData));
static void	DisplayLabel _ANSI_ARGS_((Bargraph *bgp));
static void	DisplayValue _ANSI_ARGS_((Bargraph *bgp));
static void	DisplayAlabels _ANSI_ARGS_((Bargraph *bgp));
static void	DisplayBlabels _ANSI_ARGS_((Bargraph *bgp));
static void	ComputeTicks _ANSI_ARGS_((Bargraph *bgp));
static void	DisplayMinmax _ANSI_ARGS_((Bargraph *bgp));
static void	DisplayBaseV _ANSI_ARGS_((Bargraph *bgp));
static void	DisplayBar _ANSI_ARGS_((Bargraph *bgp));
static void	DisplayBargraph _ANSI_ARGS_((ClientData clientData));

static void     EraseWing(Bargraph *bgPtr);
static void     DrawWing(Bargraph *bgPtr, double y);

static void	BargraphCmdDeletedProc _ANSI_ARGS_((ClientData clientData));
static void	EventuallyRedrawBargraph _ANSI_ARGS_((Bargraph *bgp,
		    int what));
static void	BargraphEventProc _ANSI_ARGS_((ClientData clientData,
		    XEvent *eventPtr));
static int	BargraphWidgetCmd _ANSI_ARGS_((ClientData clientData,
		    Tcl_Interp *interp, int argc, char **argv));
static void	SetBargraphValue _ANSI_ARGS_((Bargraph *bgp,
						double value));
static int	ValueToPixel _ANSI_ARGS_((Bargraph *bgp, double val, int lim));
static void	format_double _ANSI_ARGS_((Bargraph *bgp,
					   char *result, double v));
static double	limit_dvalue _ANSI_ARGS_((double minv,double maxv,double val));
static int	limit_ivalue _ANSI_ARGS_((int minv, int maxv, int val));
       float	*do_labelloc(struct splineinfo *sp, int *niter, float *fret);
void		init_labelloc(struct splineinfo *sp, int algor, int lo, int hi,
                                int cheight, int nlab, float *lab);

/*
 *--------------------------------------------------------------
 *
 * Tk_BargraphCmd --
 *
 *	This procedure is invoked to process the "Bargraph" Tcl
 *	command.  See the user documentation for details on what
 *	it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */
int
Tk_BargraphCmd(clientData, interp, argc, argv)
     ClientData clientData;		/* Main window associated with
				 * interpreter. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int argc;			/* Number of arguments. */
     char **argv;		/* Argument strings. */
{
  Tk_Window tkwin = (Tk_Window) clientData;
  register Bargraph *bgPtr;
  Tk_Window new;

  if (argc < 2) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
		     argv[0], " pathName ?options?\"", (char *) NULL);
    return TCL_ERROR;
  }

  new = Tk_CreateWindowFromPath(interp, tkwin, argv[1], (char *) NULL);
  if (new == NULL) {
    return TCL_ERROR;
  }

  /*
   * Initialize fields that won't be initialized by ConfigureBargraph,
   * or which ConfigureBargraph expects to have reasonable values
   * (e.g. resource pointers).
   */
  bgPtr			= (Bargraph *) ckalloc(sizeof(Bargraph));

  bgPtr->tkwin		= new;
  bgPtr->display	= Tk_Display(new);
  bgPtr->interp		= interp;
  bgPtr->widgetCmd	= Tcl_CreateCommand(interp, Tk_PathName(bgPtr->tkwin),
					    BargraphWidgetCmd,
					    (ClientData) bgPtr,
					    BargraphCmdDeletedProc);

  bgPtr->border		= NULL;
  bgPtr->cursor		= None;
  bgPtr->relief		= TK_RELIEF_RAISED;
  bgPtr->highlightBgColorPtr	= NULL;
  bgPtr->highlightColorPtr	= NULL;
  bgPtr->highlightWidth	= 0;
  bgPtr->value		= 0.0;
  bgPtr->label		= NULL;
  bgPtr->barBorder	= NULL;
  bgPtr->barColor	= NULL;
  bgPtr->tkfont		= NULL;
  bgPtr->textColorPtr	= NULL;
  bgPtr->textGC		= None;
  bgPtr->tickColorPtr	= NULL;
  bgPtr->tickGC		= None;

  bgPtr->wingColorPtr   = NULL;
  bgPtr->wingGC         = None;
  bgPtr->wingoverlayGC  = None;
  bgPtr->wing_Aside     = 0;
  bgPtr->wing_Bside     = 0;
  bgPtr->wingAlgorithm	= ALGORITHM_POWELL;

  bgPtr->n_alabels	= 0;
  bgPtr->alabGC		= None;
  bgPtr->alabelstr	= NULL;
  bgPtr->alabelstr_u	= NULL;
  bgPtr->alabels	= NULL;
  bgPtr->Atkfont	= NULL;

  bgPtr->n_blabels	= 0;
  bgPtr->blabGC		= None;
  bgPtr->blabelstr	= NULL;
  bgPtr->blabelstr_u	= NULL;
  bgPtr->blabels	= NULL;
  bgPtr->Btkfont	= NULL;
    
  bgPtr->mode		= MODE_BARGRAPH;
  bgPtr->flags		= 0;
  bgPtr->numTicks	= 0;
  bgPtr->ticks		= NULL;
  bgPtr->flags		= 0;

  bgPtr->sp_a.ns        = -1;
  bgPtr->sp_a.nl        = -1;
  bgPtr->sp_a.x         = NULL;
  bgPtr->sp_a.y         = NULL;
  bgPtr->sp_a.y2        = NULL;
  bgPtr->sp_a.use_upper = 0;
  bgPtr->sp_a.use_lower = 0;

  bgPtr->sp_b.ns        = -1;
  bgPtr->sp_b.nl        = -1;
  bgPtr->sp_b.x         = NULL;
  bgPtr->sp_b.y         = NULL;
  bgPtr->sp_b.y2        = NULL;
  bgPtr->sp_b.use_upper = 0;
  bgPtr->sp_b.use_lower = 0;

  bgPtr->awing_showing  = 0;
  bgPtr->bwing_showing  = 0;

  Tk_SetClass(bgPtr->tkwin, "Bargraph");
  Tk_CreateEventHandler(bgPtr->tkwin,
			ExposureMask|StructureNotifyMask|FocusChangeMask,
			BargraphEventProc, (ClientData) bgPtr);

  if (ConfigureBargraph(interp, bgPtr, argc-2, argv+2, 0) != TCL_OK) {
    Tk_DestroyWindow(new);
    return TCL_ERROR;
  }

  Tcl_SetResult(interp, Tk_PathName(bgPtr->tkwin), TCL_STATIC);
  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * BargraphWidgetCmd --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to a widget managed by this module.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */
static int
BargraphWidgetCmd(clientData, interp, argc, argv)
     ClientData clientData;		/* Information about Bargraph
					 * widget. */
     Tcl_Interp *interp;			/* Current interpreter. */
     int argc;				/* Number of arguments. */
     char **argv;			/* Argument strings. */
{
  register Bargraph *bgPtr = (Bargraph *) clientData;
  int result = TCL_OK;
  unsigned int length;
  char c;

  if (argc < 2) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
		     argv[0], " option ?arg arg ...?\"", (char *) NULL);
    return TCL_ERROR;
  }
  Tcl_Preserve((ClientData) bgPtr);
  c = argv[1][0];
  length = strlen(argv[1]);
  if ((c == 'c') && (strncmp(argv[1], "cget", length) == 0)
      && (length >= 2)) {
    if (argc != 3) {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " cget option\"", (char *) NULL);
      goto error;
    }
    result = Tk_ConfigureValue(interp, bgPtr->tkwin, configSpecs,
			       (char *) bgPtr, argv[2], 0);
  } else if ((c == 'c') && (strncmp(argv[1], "configure", length) == 0)) {
    if (argc == 2) {
      result = Tk_ConfigureInfo(interp, bgPtr->tkwin, configSpecs,
				(char *) bgPtr, (char *) NULL, 0);
    } else if (argc == 3) {
      result = Tk_ConfigureInfo(interp, bgPtr->tkwin, configSpecs,
				(char *) bgPtr, argv[2], 0);
    } else {
      result = ConfigureBargraph(interp, bgPtr, argc-2, argv+2,
				 TK_CONFIG_ARGV_ONLY);
    }
  } else if ((c == 'g') && (strncmp(argv[1], "get", length) == 0)) {
    if (argc != 2) {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " get\"", (char *) NULL);
      goto error;
    }
    format_double(bgPtr, interp->result, bgPtr->value);
  } else if ((c == 's') && (strncmp(argv[1], "set", length) == 0)) {
    double value;

    if (argc != 3) {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " set value\"", (char *) NULL);
      goto error;
    }
    if (Tcl_GetDouble(interp, argv[2], &value) != TCL_OK) {
      goto error;
    }
    SetBargraphValue(bgPtr, value);
  } else {
    Tcl_AppendResult(interp, "bad option \"", argv[1],
		     "\":  must be configure, get, set",
		     (char *) NULL);
    goto error;
  }
  Tcl_Release((ClientData) bgPtr);
  return result;

error:
  Tcl_Release((ClientData) bgPtr);
  return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * DestroyBargraph --
 *
 *	This procedure is invoked by Tcl_EventuallyFree or Tcl_Release
 *	to clean up the internal structure of a button at a safe time
 *	(when no-one is using it anymore).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the Bargraph is freed up.
 *
 *----------------------------------------------------------------------
 */
static void
DestroyBargraph(clientData)
    char *clientData;	/* Info about Bargraph widget. */
{
  register Bargraph *bgPtr = (Bargraph *) clientData;
  
  if (bgPtr->textGC != None)
    Tk_FreeGC(bgPtr->display, bgPtr->textGC);
  
  if (bgPtr->alabGC != None)
    Tk_FreeGC(bgPtr->display, bgPtr->alabGC);
  
  if (bgPtr->blabGC != None)
    Tk_FreeGC(bgPtr->display, bgPtr->blabGC);
  
  if (bgPtr->tickGC != None)
    Tk_FreeGC(bgPtr->display, bgPtr->tickGC);

  if ((bgPtr->wingGC != None) && bgPtr->tkwin)
    Tk_FreeGC(bgPtr->display, bgPtr->wingGC);

  if ((bgPtr->wingoverlayGC != None) && bgPtr->tkwin)
    Tk_FreeGC(bgPtr->display, bgPtr->wingoverlayGC);

  if (bgPtr->ticks != NULL)
    ckfree((char *)bgPtr->ticks);

  if (bgPtr->alabelstr != NULL)
    ckfree(bgPtr->alabelstr);
  
  if (bgPtr->blabelstr != NULL)
    ckfree(bgPtr->blabelstr);
  
  if (bgPtr->alabels != NULL)
    Tcl_Free((char *) bgPtr->alabels);
  
  if (bgPtr->blabels != NULL)
    Tcl_Free((char *) bgPtr->blabels);

  free_sp(&bgPtr->sp_a);
  free_sp(&bgPtr->sp_b);
  
  /* free the configuration options in the widget */
  Tk_FreeOptions(configSpecs, (char *) bgPtr, bgPtr->display, 0);

  ckfree((char *) bgPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * ConfigureBargraph --
 *
 *	This procedure is called to process an argv/argc list, plus
 *	the Tk option database, in order to configure (or
 *	reconfigure) a Bargraph widget.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as colors, border width,
 *	etc. get set for bgPtr;  old resources get freed,
 *	if there were any.
 *
 *----------------------------------------------------------------------
 */
static int
ConfigureBargraph(interp, bgPtr, argc, argv, flags)
    Tcl_Interp *interp;		    /* Used for error reporting. */
    register Bargraph *bgPtr; /* Information about widget;  may or may
				     * not already have values for some fields
				     */
    int argc;			    /* Number of valid entries in argv. */
    char **argv;		    /* Arguments. */
    int flags;			    /* Flags to pass to Tk_ConfigureWidget. */
{
  XGCValues gcValues;
  GC newGC;
  unsigned int length;

  if (Tk_ConfigureWidget(interp, bgPtr->tkwin, configSpecs,
			 argc, argv, (char *) bgPtr, flags) != TCL_OK) {
    return TCL_ERROR;
  }

  /*
   * A few options need special processing, such as parsing the
   * orientation or setting the background from a 3-D border.
   */

  /*
   * Check for user-entered complex labels.
   * This is a list of values; each element is {value text ...}
   */
  if (!(bgPtr->alabelstr) || strcmp(bgPtr->alabelstr, bgPtr->alabelstr_u)) {
    /* There is a new value */
    length = strlen(bgPtr->alabelstr_u);
    if (bgPtr->alabelstr != NULL) {
      ckfree(bgPtr->alabelstr);
      bgPtr->alabelstr = NULL;
    }
    bgPtr->alabelstr = (char *) ckalloc(length+1);
    strcpy(bgPtr->alabelstr, bgPtr->alabelstr_u);
    if (bgPtr->alabels != NULL) {
      Tcl_Free((char *) bgPtr->alabels);
      bgPtr->alabels = NULL;
    }
    if (Tcl_SplitList(interp, bgPtr->alabelstr,
		      &bgPtr->n_alabels, &bgPtr->alabels) == TCL_ERROR) {
      Tcl_AppendResult(interp, "-alabels argument is not a proper list",
		       (char *) NULL);
      return TCL_ERROR;
    }
  }

  if (!(bgPtr->blabelstr) || strcmp(bgPtr->blabelstr, bgPtr->blabelstr_u)) {
    /* There is a new value */
    length = strlen(bgPtr->blabelstr_u);
    if (bgPtr->blabelstr != NULL) {
      ckfree(bgPtr->blabelstr);
      bgPtr->blabelstr = NULL;
    }
    bgPtr->blabelstr = (char *) ckalloc(length+1);
    strcpy(bgPtr->blabelstr, bgPtr->blabelstr_u);
    if (bgPtr->blabels != NULL) {
      Tcl_Free((char *) bgPtr->blabels);
      bgPtr->blabels = NULL;
    }
    if (Tcl_SplitList(interp, bgPtr->blabelstr,
		      &bgPtr->n_blabels, &bgPtr->blabels) == TCL_ERROR) {
      Tcl_AppendResult(interp, "-blabels argument is not a proper list",
		       (char *) NULL);
      return TCL_ERROR;
    }
  }

  /* Explicitly erase wings, in case their color was changed */
  EraseWing(bgPtr);

  /*
   * Make sure that the tick interval has the right sign so that
   * addition moves from fromVal to toVal.
   */
  bgPtr->tickInterval = fabs(bgPtr->tickInterval);

  /*
   * Make sure that the number of ticks isn't excessive: not more
   * than height/2 ticks.
   */
  if (bgPtr->tickInterval_u > 0.) {
    if ( 2.*(fabs(bgPtr->toVal_u - bgPtr->fromVal_u) /
	     bgPtr->tickInterval_u) > bgPtr->max_height) {
      Tcl_AppendResult(interp,
		       "Bad parameter combination: ",
		       "must have ((max-min)/tickInterval) < height/2",
		       (char *) NULL);
      return TCL_ERROR;
    }
  }
  bgPtr->toVal = bgPtr->toVal_u;
  bgPtr->fromVal = bgPtr->fromVal_u;
  /* Ensure that fromVal and toVal are never equal */
  if (bgPtr->toVal == bgPtr->fromVal) {
    bgPtr->toVal = 1.01 * bgPtr->fromVal;
    if (bgPtr->toVal == bgPtr->fromVal)
      bgPtr->toVal = 1.;
  }
  bgPtr->tickInterval = bgPtr->tickInterval_u;

  /*
   * Set the Bargraph value to itself;  all this does is to make sure
   * that the Bargraph's value is within the new acceptable range for
   * the Bargraph.
   */
  SetBargraphValue(bgPtr, bgPtr->value);

  Tk_SetBackgroundFromBorder(bgPtr->tkwin, bgPtr->border);

  gcValues.font = Tk_FontId(bgPtr->tkfont);
  gcValues.foreground = bgPtr->textColorPtr->pixel;
  newGC = Tk_GetGC(bgPtr->tkwin, GCForeground|GCFont, &gcValues);
  if (bgPtr->textGC != None) {
    Tk_FreeGC(bgPtr->display, bgPtr->textGC);
  }
  bgPtr->textGC = newGC;

  gcValues.font = Tk_FontId(bgPtr->Atkfont);
  gcValues.foreground = bgPtr->textColorPtr->pixel;
  newGC = Tk_GetGC(bgPtr->tkwin, GCForeground|GCFont, &gcValues);
  if (bgPtr->alabGC != None) {
    Tk_FreeGC(bgPtr->display, bgPtr->alabGC);
  }
  bgPtr->alabGC = newGC;

  gcValues.font = Tk_FontId(bgPtr->Btkfont);
  gcValues.foreground = bgPtr->textColorPtr->pixel;
  newGC = Tk_GetGC(bgPtr->tkwin, GCForeground|GCFont, &gcValues);
  if (bgPtr->blabGC != None) {
    Tk_FreeGC(bgPtr->display, bgPtr->blabGC);
  }
  bgPtr->blabGC = newGC;

  gcValues.foreground = bgPtr->tickColorPtr->pixel;
  newGC = Tk_GetGC(bgPtr->tkwin, GCForeground, &gcValues);
  if (bgPtr->tickGC != None) {
    Tk_FreeGC(bgPtr->display, bgPtr->tickGC);
  }
  bgPtr->tickGC = newGC;

  /* We draw the wings by xor'ing the stored value with the background.
   * To get the desired color value drawn, we have to store the xor'd value
   * here.
   */
  gcValues.foreground = bgPtr->wingColorPtr->pixel;
  gcValues.foreground ^= Tk_3DBorderColor(bgPtr->border)->pixel;
  gcValues.function = GXxor;
  gcValues.line_width = bgPtr->wingWidth;
  gcValues.cap_style = CapButt;
  gcValues.join_style = JoinMiter;
  newGC = Tk_GetGC(bgPtr->tkwin, GCLineWidth | GCCapStyle | GCJoinStyle |
                                GCFunction | GCForeground, &gcValues);
  if (bgPtr->wingGC != None) {
    Tk_FreeGC(bgPtr->display, bgPtr->wingGC);
  }
  bgPtr->wingGC = newGC;
  /* A line of same color as wing is drawn over the value line.
   * It differs from the above wing because there is no xor, and its width is 1.
   */
  gcValues.foreground = bgPtr->wingColorPtr->pixel;
  gcValues.function = GXcopy;
  gcValues.line_width = 1;
  newGC = Tk_GetGC(bgPtr->tkwin, GCLineWidth | GCCapStyle | GCJoinStyle |
                                GCFunction | GCForeground, &gcValues);
  if (bgPtr->wingoverlayGC != None) {
    Tk_FreeGC(bgPtr->display, bgPtr->wingoverlayGC);
  }
  bgPtr->wingoverlayGC = newGC;

  /*
   * Recompute display-related information, and let the geometry
   * manager know how much space is needed now.
   */

  ComputeBargraphGeometry(bgPtr);

  EventuallyRedrawBargraph(bgPtr, DISPLAY_ALL | CLEAR_NEEDED);
  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ComputeBargraphGeometry --
 *
 *	This procedure is called to compute various geometrical
 *	information for a Bargraph, such as where various things get
 *	displayed.  It's called when the window is reconfigured.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Display-related numbers get changed in *scrollPtr.  The
 *	geometry manager gets told about the window's preferred size.
 *
 *----------------------------------------------------------------------
 */
static void
ComputeBargraphGeometry(bgPtr)
    register Bargraph *bgPtr;		/* Information about widget. */
{
  char valueString[30];
  int bd, i, l, boolv;
  int minvpixels, maxvpixels;
  int win_width, win_height;
  int a_has_ticks=0;
  int b_has_ticks=0;
  Tk_FontMetrics fm, afm, bfm;
  char *s, **elt;
  int nelt;
  Tcl_Interp *interp = bgPtr->interp;

  Tk_GetFontMetrics(bgPtr->tkfont, &fm);
  Tk_GetFontMetrics(bgPtr->Atkfont, &afm);
  Tk_GetFontMetrics(bgPtr->Btkfont, &bfm);

  /* If wing-drawing is on, draw them when within 2/3 linespace of a tick */
  bgPtr->wing_Amindist = (int) (bgPtr->wingTrigger * afm.linespace + 0.5);
  bgPtr->wing_Bmindist = (int) (bgPtr->wingTrigger * bfm.linespace + 0.5);

  /* Ensure that the base value is between the minimum and maximum values */
  bgPtr->baseVal = limit_dvalue(bgPtr->fromVal, bgPtr->toVal, bgPtr->baseVal);

  /*
   * Calculate text extents for the value.
   * This value is used when bargraph is horizontal or vertical
   */
  format_double(bgPtr, valueString, bgPtr->fromVal);
  minvpixels = Tk_TextWidth(bgPtr->tkfont, valueString, strlen(valueString));
  format_double(bgPtr, valueString, bgPtr->toVal);
  maxvpixels = Tk_TextWidth(bgPtr->tkfont, valueString, strlen(valueString));
  if (bgPtr->showValue) {
    bgPtr->valuePixels = MAX(minvpixels, maxvpixels);
  } else {
    bgPtr->valuePixels = 0;
  }

  /* Check for space required by complex labels */
  bgPtr->alabelPixels = 0;
  for (l=0, i=0; i<bgPtr->n_alabels; i++) {
    s = bgPtr->alabels[i];
    if (Tcl_SplitList(interp, s, &nelt, &elt) == TCL_ERROR) {
      /* ignore this value */
    } else if (nelt >= 2) {
      l = Tk_TextWidth(bgPtr->Atkfont, elt[1], strlen(elt[1]));
      Tcl_Free((char *) elt);
      if (nelt == 2) {
	  a_has_ticks = 1;
      } else if (!a_has_ticks && nelt > 2) {
	if (Tcl_GetBoolean(interp, elt[2], &boolv) == TCL_OK)
	  if (boolv)
	    a_has_ticks = 1;
      }
    }
    if (bgPtr->alabelPixels < l)
      bgPtr->alabelPixels = l;
  }
  bgPtr->blabelPixels = 0;
  for (l=0,i=0; i<bgPtr->n_blabels; i++) {
    s = bgPtr->blabels[i];
    if (Tcl_SplitList(interp, s, &nelt, &elt) == TCL_ERROR) {
      /* ignore this value */
    } else if (nelt >= 2) {
      l = Tk_TextWidth(bgPtr->Btkfont, elt[1], strlen(elt[1]));
      Tcl_Free((char *) elt);
      if (nelt == 2) {
	b_has_ticks = 1;
      } else if (!b_has_ticks && nelt > 2) {
	if (Tcl_GetBoolean(interp, elt[2], &boolv) == TCL_OK)
	  if (boolv)
	    b_has_ticks = 1;
      }
    }
    if (bgPtr->blabelPixels < l)
      bgPtr->blabelPixels = l;
  }
  bd = bgPtr->borderWidth;

  if (bgPtr->orient != ORIENT_VERTICAL) {
    /* HORIZONTAL */
    /*
     *	 -------------------------------------------
     *	 |\_______________________________________/|
     *	 | |	    Label  (optional)		 | |
     *	 | |-------------------------------------| |
     *	 | |	   min	 (optional)	max	 | |
     *	 | |-------------------------------------| |
     *	 | |	    alabels(optional)		 | |
     *	 | |-------------------------------------| |
     *	 | |	   |	  Ticks (opt)		 | |
     *	 | | Value | #####################	 | |
     *	 | | (opt) | #####################	 | |
     *	 | |-------------------------------------| |
     *	 | |	    blabels(optional)		 | |
     *	 |/---------------------------------------\|
     *	  ------------------------------------------
     *			^-- BorderWidth
     */

    if (bgPtr->tickInterval != 0.)
      bgPtr->minmaxPixels = fm.linespace + bgPtr->padY;
    else
      bgPtr->minmaxPixels = 0;

    if (fm.linespace > bgPtr->width )
      bgPtr->width = fm.linespace;

    /* Left edge of bar area. */
    bgPtr->offset.bot_edge = bd + bgPtr->padX +
      MAX(bgPtr->showValue*(bgPtr->valuePixels + bgPtr->padX),
	  SHOW_RANGE(bgPtr)*minvpixels/2);

    /* Right edge of bar area */
    bgPtr->offset.top_edge = bgPtr->offset.bot_edge + bgPtr->max_height;

    /* Base value position */
    bgPtr->offset.base = ValueToPixel(bgPtr, bgPtr->baseVal, 1);

    /* Top of label string */
    bgPtr->offset.label_y = bd;

    /* Right edge of minmax "max" value. */
    bgPtr->offset.minmax1_x = bgPtr->offset.top_edge + maxvpixels/2;

    /* Left edge of minmax "min" value. */
    bgPtr->offset.minmax0_x = bgPtr->offset.bot_edge - minvpixels/2;

    /* Top edge of minmax "max" and "min" values. */
    bgPtr->offset.minmax1_y = 
    bgPtr->offset.minmax0_y = bgPtr->offset.label_y +
      (bgPtr->label != NULL)*(fm.linespace + 2*bgPtr->padY);

    /* Top edge of alabels.  Note that we assume the minmax values
     * will be printed in a box that doesn't include space for
     * descenders, since numbers have none. */
    bgPtr->offset.lab_a = bgPtr->offset.minmax0_y + 
      SHOW_RANGE_OR_BASEV(bgPtr)*(fm.ascent + bgPtr->padY);

    /* Top edge of upper ticks */
    bgPtr->offset.tick_a = bgPtr->offset.lab_a +
		HAS_ALAB(bgPtr)*(afm.linespace+bgPtr->padY +
			a_has_ticks*bgPtr->tickLength);

    /* Top edge of bar */
    bgPtr->offset.bar_y = bgPtr->offset.tick_a +
		(SHOW_TICKS(bgPtr) || HAS_ALAB(bgPtr))*bgPtr->tickLength;

    /* Top edge of value string */
    bgPtr->offset.value_y = bgPtr->offset.bar_y +
      (bgPtr->width - fm.linespace)/2;

    /* Top edge of lower ticks */
    bgPtr->offset.tick_b = bgPtr->offset.bar_y + bgPtr->width;

    /* Top edge of blabels */
    bgPtr->offset.lab_b = bgPtr->offset.tick_b +
		HAS_BLAB(bgPtr)*(bgPtr->tickLength + bgPtr->padY +
			b_has_ticks*bgPtr->tickLength);

    win_width = (SHOW_RANGE(bgPtr) ?
		bgPtr->offset.minmax1_x : bgPtr->offset.top_edge) +
		    (bgPtr->borderWidth);

    win_height = bgPtr->offset.lab_b +
		HAS_BLAB(bgPtr)*(bfm.linespace + bgPtr->padY) +
		bgPtr->borderWidth;
  } else {
    /* the bargraph is vertical */
    /*
     *	________________________________________
     *	|\____________________________________/|
     *	| |	       Label (opt)	     | |
     *	| |----------------------------------| |
     *	| |		      |		     | |
     *	| | MAX(opt)	      |	  ###	     | |
     *	| |		      |	  ###	     | |
     *	| |	 Alabs	 Ticks|	  ###  Blabs | |
     *	| |	 (opt)	 (opt)|	  ###  (opt) | |
     *	| |		      |	  ###	     | |
     *	| |		      |	  ###	     | |
     *	| | MIN(opt)	      |	  ###	     | |
     *	| |----------------------------------| |
     *	| |	       Value (opt)	     | |
     *	|/------------------------------------\|
     *	----------------------------------------
     *	     ^-- borderWidth
     */

    /* Calculate the extents for the ticks text */
    if ( SHOW_RANGE_OR_BASEV(bgPtr) ) {
	format_double(bgPtr, valueString, bgPtr->fromVal);
	bgPtr->minmaxPixels = Tk_TextWidth(bgPtr->tkfont,
					   valueString, strlen(valueString));

	format_double(bgPtr, valueString, bgPtr->toVal);
	l = Tk_TextWidth(bgPtr->tkfont, valueString, strlen(valueString));
	if (bgPtr->minmaxPixels < l)
	    bgPtr->minmaxPixels = l;

	bgPtr->minmaxPixels += bgPtr->padX;
    } else {
	bgPtr->minmaxPixels = 0;
    }

    /*
     *	Value is always displayed directly below the bar, so the bar needs
     *	to be wide enough so that the value text isn't clipped.
     */
    bgPtr->width = MAX(bgPtr->width, bgPtr->valuePixels);

    /* Top of label string */
    bgPtr->offset.label_y = bd;

    /* top of bar area */
    bgPtr->offset.top_edge = bgPtr->offset.label_y +
      (bgPtr->label ? (fm.linespace + 2*bgPtr->padY) :
       (SHOW_RANGE_OR_BASEV(bgPtr) ? (fm.linespace/2 + bgPtr->padY) : 0));

    /* Bottom of bar area */
    bgPtr->offset.bot_edge = bgPtr->offset.top_edge + bgPtr->max_height;

    /* base value */
    bgPtr->offset.base = ValueToPixel(bgPtr, bgPtr->baseVal, 1);

    /* Left edge of minmax "max" and "min" values. */
    bgPtr->offset.minmax1_x =
    bgPtr->offset.minmax0_x = bd + bgPtr->padX;

    /* Baseline of minmax "max" value */
    bgPtr->offset.minmax1_y = bgPtr->offset.top_edge + fm.ascent/2;

    /* Baseline of minmax "min" value */
    bgPtr->offset.minmax0_y = bgPtr->offset.bot_edge + fm.ascent/2;

    /* Left edge of alabels */
    bgPtr->offset.lab_a = bd + bgPtr->padX +
	    SHOW_RANGE_OR_BASEV(bgPtr)*(bgPtr->minmaxPixels + bgPtr->padX);

    /* Left edge of left-side ticks */
    bgPtr->offset.tick_a = bgPtr->offset.lab_a +
	    HAS_ALAB(bgPtr)*(bgPtr->alabelPixels + bgPtr->padX);

    /* Left edge of bar */
    bgPtr->offset.bar_x = bgPtr->offset.tick_a +
	    (SHOW_TICKS(bgPtr) || a_has_ticks)*bgPtr->tickLength;

    /* left/right center of the value string */
    bgPtr->offset.value_x = bgPtr->offset.bar_x + bgPtr->width/2;

    /* Top of the value string */
    bgPtr->offset.value_y = bgPtr->offset.bot_edge + bgPtr->padY/2;

    /* Left edge of the right-side ticks */
    bgPtr->offset.tick_b = bgPtr->offset.bar_x + bgPtr->width;

    /* Left edge of the blabel text */
    bgPtr->offset.lab_b = bgPtr->offset.tick_b +
	    b_has_ticks*(bgPtr->tickLength + bgPtr->padX);

    win_width = bgPtr->offset.lab_b +
		  HAS_BLAB(bgPtr)*(bgPtr->blabelPixels + bgPtr->padX) + bd;

    win_height = (bgPtr->showValue ? 
		  (bgPtr->offset.value_y + fm.ascent) :
		  (SHOW_RANGE_OR_BASEV(bgPtr) ?
		   (bgPtr->offset.bot_edge + fm.ascent/2) :
		   bgPtr->offset.bot_edge))
      + bgPtr->padY/2 + bd;
  }

  Tk_GeometryRequest(bgPtr->tkwin, win_width, win_height);
  Tk_SetInternalBorder(bgPtr->tkwin, bgPtr->borderWidth);
}

/*
 * Display the window's label.
 */
static void
DisplayLabel(bgPtr)
    Bargraph *bgPtr;
{
  register Tk_Window tkwin = bgPtr->tkwin;
  int bd = bgPtr->borderWidth;
  int win_width = Tk_Width(tkwin);
  int x, width;
  Tk_FontMetrics fm;

  if (bgPtr->label == NULL) return;

  Tk_GetFontMetrics(bgPtr->tkfont, &fm);

  width = Tk_TextWidth(bgPtr->tkfont, bgPtr->label, strlen(bgPtr->label));
  if (width < win_width - 2*bd)
    x = (win_width - width)/2;
  else
    x = bd + bgPtr->padX;

  Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin), bgPtr->border,
		     bd, bgPtr->offset.label_y + bgPtr->padY,
		     win_width - 2*bd, fm.linespace + bgPtr->padY,
		     0, TK_RELIEF_FLAT);
  Tk_DrawChars(bgPtr->display, Tk_WindowId(tkwin),
	       bgPtr->textGC, bgPtr->tkfont,
	       bgPtr->label, strlen(bgPtr->label), x,
	       bgPtr->offset.label_y + fm.ascent);
}

/*
 * Display the value
 */
static void
DisplayValue(bgPtr)
    Bargraph *bgPtr;
{
  register Tk_Window tkwin = bgPtr->tkwin;
  int twidth, bd = bgPtr->borderWidth;
  char valueString[40];
  Tk_FontMetrics fm;

  Tk_GetFontMetrics(bgPtr->tkfont, &fm);

  format_double(bgPtr, valueString, bgPtr->value);
  twidth = Tk_TextWidth(bgPtr->tkfont, valueString, strlen(valueString));

  if (bgPtr->orient == ORIENT_VERTICAL) {
    /* Display the value, centered below the bar.  */
    /* Note: if you change (padY/2) + ... + (padY/2) to be
     * (padY + ...), then you'll overshoot by 1 pixel in the downward
     * direction when padY is odd.
     */
    Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin), bgPtr->border,
		       bgPtr->offset.value_x - bgPtr->valuePixels/2,
		       bgPtr->offset.value_y - bgPtr->padY/2,
		       bgPtr->valuePixels,
		       bgPtr->padY/2 + fm.ascent + bgPtr->padY/2,
		       0, TK_RELIEF_FLAT);
    Tk_DrawChars(bgPtr->display, Tk_WindowId(tkwin), bgPtr->textGC,
		 bgPtr->tkfont,
		 valueString, strlen(valueString),
		 bgPtr->offset.value_x - twidth/2,
		 bgPtr->offset.value_y + fm.linespace);
  } else {
    /* HORIZONTAL */
    Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin), bgPtr->border,
		       bd + bgPtr->padX, bgPtr->offset.value_y,
		       bgPtr->valuePixels,
		       fm.linespace + bgPtr->padY,
		       0, TK_RELIEF_FLAT);
    Tk_DrawChars(bgPtr->display, Tk_WindowId(tkwin), bgPtr->textGC,
		 bgPtr->tkfont,
		 valueString, strlen(valueString),
		 bd + bgPtr->padX + bgPtr->valuePixels - twidth,
		 bgPtr->offset.value_y + fm.linespace);
  }
}

/*
 * Display the alabels
 * Note that ticks are "scheduled" for drawing, but not yet drawn.
 */
static void
DisplayAlabels(bgPtr)
    Bargraph *bgPtr;
{
  register Tk_Window tkwin = bgPtr->tkwin;
  int boolv, awidth;

  char *s, **elt;
  int i, j, nelt, tickoff;
  double d;
  int d_t, d_l;
  Tcl_Interp *interp = bgPtr->interp;
  Tk_FontMetrics afm;

  float *label_adj, label_in[MAX_LAB_ADJ];
  int niter;
  float fret;

  Tk_GetFontMetrics(bgPtr->Atkfont, &afm);

  clear_labelloc(&bgPtr->sp_a);

  label_adj = NULL;             /* have no adjusted positions */
if (bgPtr->orient == ORIENT_VERTICAL) {
    if (bgPtr->n_alabels < MAX_LAB_ADJ) {
      /* We aren't dealing w/ a ridiculous number.  Mangle positions to do
       * avoid overlap as much as possible.
       */
      for (j=0, i = 0; i < bgPtr->n_alabels; i++) {
        s = bgPtr->alabels[i];
        tickoff = 0;
        if (Tcl_SplitList(interp, s, &nelt, &elt) != TCL_ERROR && nelt >= 2) {
          d = limit_dvalue(bgPtr->fromVal,
                              bgPtr->toVal, strtod(elt[0], NULL));
          ValueToPixel(bgPtr, d, 1);
          label_in[j++] = bgPtr->pixval;
        }
      }
      #if 0
	for (j=0, i = 0; i < bgPtr->n_alabels; i++) {
	  fprintf(stderr, "label_in[%d]=%f\n", j, label_in[j]); j++;
	}
      #endif
      if (j > 1) {
          init_labelloc(&bgPtr->sp_a, bgPtr->wingAlgorithm,
	  		bgPtr->offset.bot_edge-bgPtr->max_height,
                        bgPtr->offset.bot_edge, afm.linespace, j, label_in);
          label_adj = do_labelloc(&bgPtr->sp_a, &niter, &fret);
      }
    }
  }

  /* OK, now do the tick and label stuff */
  for (j = 0, i = 0; i < bgPtr->n_alabels; i++) {
    s = bgPtr->alabels[i];
    tickoff = 0;
    if (Tcl_SplitList(interp, s, &nelt, &elt) == TCL_ERROR) {
      /* ignore this value */
    } else if (nelt >= 2) {
      d = limit_dvalue(bgPtr->fromVal, bgPtr->toVal, strtod(elt[0], NULL));
	d_t = ValueToPixel(bgPtr, d, 1);
      d_l = label_adj ? ((int) label_adj[j++]) : d_t;
      s = elt[1];
      awidth = Tk_TextWidth(bgPtr->Atkfont, s, strlen(s));
      if (bgPtr->orient == ORIENT_VERTICAL) {
	Tk_DrawChars(bgPtr->display, Tk_WindowId(tkwin),
		     bgPtr->alabGC, bgPtr->Atkfont,
		     s, strlen(s),
		     bgPtr->offset.lab_a + bgPtr->alabelPixels - awidth,
		     d_l + afm.descent);
	if (nelt >= 2) {
	  /* Draw a tick if no flag, or flag is true.
	   * (Ignore errors in value).
	   */ 
	   if (nelt == 2) {
	    boolv = 1;
	   } else {
	    Tcl_GetBoolean(interp, elt[2], &boolv);
	   }

	  if (boolv) {
	    /* Draw a tick */
	    bgPtr->ticks[bgPtr->numTicks].x1 = bgPtr->offset.tick_a;
	    bgPtr->ticks[bgPtr->numTicks].x2 = bgPtr->offset.tick_a +
	      bgPtr->tickLength;
	    bgPtr->ticks[bgPtr->numTicks].y1 = d_l;
	      bgPtr->ticks[bgPtr->numTicks].y2 = d_t;
	    bgPtr->numTicks++;
	  }
	}
      } else {
	/* HORIZONTAL */
	Tk_DrawChars(bgPtr->display, Tk_WindowId(tkwin),
		     bgPtr->alabGC, bgPtr->Atkfont,
		     s, strlen(s), d_l - awidth/2,
		     bgPtr->offset.lab_a + afm.ascent);
	if (nelt >= 2) {
	  /* Draw a tick if no flag, or flag is true.
	   * (Ignore errors in value).
	   */ 
	   if (nelt == 2) {
	    boolv = 1;
	   } else {
	    Tcl_GetBoolean(interp, elt[2], &boolv);
	   }

	  if (boolv) {
	    bgPtr->ticks[bgPtr->numTicks].y1 = bgPtr->offset.tick_a;
	    bgPtr->ticks[bgPtr->numTicks].y2 = bgPtr->offset.tick_a + 
	      bgPtr->tickLength;
	    bgPtr->ticks[bgPtr->numTicks].x1 = d_l;
	      bgPtr->ticks[bgPtr->numTicks].x2 = d_t;
	    bgPtr->numTicks++;
	  }
	}
      }
      Tcl_Free((char *) elt);
    }
  }
}

/*
 * Display the blabels
 * Note that ticks are "scheduled" for drawing, but not yet drawn.
 */
static void
DisplayBlabels(bgPtr)
    Bargraph *bgPtr;
{
  register Tk_Window tkwin = bgPtr->tkwin;
  int boolv, bwidth;
  char *s, **elt;
  int i, j, nelt, tickoff;
  int d_t, d_l;
  double d;
  Tcl_Interp *interp = bgPtr->interp;
  Tk_FontMetrics bfm;

  float *label_adj, label_in[MAX_LAB_ADJ];
  int niter;
  float fret;

  Tk_GetFontMetrics(bgPtr->Btkfont, &bfm);

  clear_labelloc(&bgPtr->sp_b);

  /* Make an array of label positions.  If we are vertically oriented,
   * adjust this array so as to minimize label overlap.  Skip this step
   * if there is not enough room to place all labels w/o overlap.
   */
  label_adj = NULL;             /* have no adjusted positions */
  if (bgPtr->orient == ORIENT_VERTICAL) {
    if (bgPtr->n_blabels < MAX_LAB_ADJ) {
      /* We aren't dealing w/ a ridiculous number.  Mangle positions to do
       * avoid overlap as much as possible.
       */
      for (j=0, i = 0; i < bgPtr->n_blabels; i++) {
        s = bgPtr->blabels[i];
        tickoff = 0;
        if (Tcl_SplitList(interp, s, &nelt, &elt) != TCL_ERROR && nelt >= 2) {
          d = limit_dvalue(bgPtr->fromVal,
                            bgPtr->toVal, strtod(elt[0], NULL));
          ValueToPixel(bgPtr, d, 1);
          label_in[j++] = bgPtr->pixval;
        }
      }
      if (j > 1) {
          init_labelloc(&bgPtr->sp_b, bgPtr->wingAlgorithm,
	  		bgPtr->offset.bot_edge-bgPtr->max_height,
                        bgPtr->offset.bot_edge, bfm.linespace, j, label_in);
          label_adj = do_labelloc(&bgPtr->sp_b, &niter, &fret);
      }
    }
  }

  /* OK, now do the tick and label stuff */
  for (j = 0, i = 0; i < bgPtr->n_blabels; i++) {
    s = bgPtr->blabels[i];
    tickoff = 0;
    if (Tcl_SplitList(interp, s, &nelt, &elt) == TCL_ERROR) {
      /* ignore this value */
    } else if (nelt >= 2) {
      d = limit_dvalue(bgPtr->fromVal, bgPtr->toVal, strtod(elt[0], NULL));
          d_t = ValueToPixel(bgPtr, d, 1);
          /* Label position */
          d_l = label_adj ? ((int) label_adj[j++]) : d_t;
          #if 0
              printf("d_l = %d  d_t = %d\n", d_l, d_t);
          #endif
      s = elt[1];
      bwidth = Tk_TextWidth(bgPtr->Btkfont, s, strlen(s));
      if (bgPtr->orient == ORIENT_VERTICAL) {
	Tk_DrawChars(bgPtr->display, Tk_WindowId(tkwin),
		     bgPtr->blabGC, bgPtr->Btkfont,
		     s, strlen(s),
		     bgPtr->offset.lab_b,
		     d_l + bfm.descent);
	if (nelt >= 2) {
	  /* Draw a tick if no flag, or flag is true.
	   * (Ignore errors in value).
	   */ 
	   if (nelt == 2) {
	    boolv = 1;
	   } else {
	    Tcl_GetBoolean(interp, elt[2], &boolv);
	   }
	  if (boolv) {
	    /* Draw a tick */
	    bgPtr->ticks[bgPtr->numTicks].x1 = bgPtr->offset.tick_b;
	    bgPtr->ticks[bgPtr->numTicks].x2 = bgPtr->offset.tick_b +
	      bgPtr->tickLength;
              bgPtr->ticks[bgPtr->numTicks].y1 = d_t;
              bgPtr->ticks[bgPtr->numTicks].y2 = d_l;
	    bgPtr->numTicks++;
	  }
	}
      } else {
	/* HORIZONTAL */
	Tk_DrawChars(bgPtr->display, Tk_WindowId(tkwin),
		     bgPtr->blabGC, bgPtr->Btkfont,
		     s, strlen(s),
		     d_l - bwidth/2,
		     bgPtr->offset.lab_b + bfm.ascent);
	if (nelt >= 2) {
	  /* Draw a tick if no flag, or flag is true.
	   * (Ignore errors in value).
	   */ 
	   if (nelt == 2) {
	    boolv = 1;
	   } else {
	    Tcl_GetBoolean(interp, elt[2], &boolv);
	   }
	  if (boolv) {
	    bgPtr->ticks[bgPtr->numTicks].y1 = bgPtr->offset.tick_b;
	    bgPtr->ticks[bgPtr->numTicks].y2 = bgPtr->offset.tick_b + 
	      bgPtr->tickLength;
	    bgPtr->ticks[bgPtr->numTicks].x1 = d_l;
	      bgPtr->ticks[bgPtr->numTicks].x2 = d_t;
	    bgPtr->numTicks++;
	  }
	}
      }
      Tcl_Free((char *) elt);
    }
  }
}

/*
 * compute the ticks
 */
static void
ComputeTicks(bgPtr)
    Bargraph *bgPtr;
{
  double v, start, end;

  start = MIN(bgPtr->fromVal, bgPtr->toVal);
  end = MAX(bgPtr->fromVal, bgPtr->toVal);

  if (bgPtr->orient == ORIENT_VERTICAL) {
    for ( v = bgPtr->baseVal; v >= start;
	  v-=bgPtr->tickInterval, bgPtr->numTicks++ ) {
      bgPtr->ticks[bgPtr->numTicks].x1 = bgPtr->offset.tick_a;
      bgPtr->ticks[bgPtr->numTicks].x2 = bgPtr->offset.tick_a + bgPtr->tickLength;
      bgPtr->ticks[bgPtr->numTicks].y1 =
	bgPtr->ticks[bgPtr->numTicks].y2 = ValueToPixel(bgPtr, v, 1);
    }

    for ( v = bgPtr->baseVal+bgPtr->tickInterval; v <= end;
	 v+=bgPtr->tickInterval, bgPtr->numTicks++ ) {
      bgPtr->ticks[bgPtr->numTicks].x1 = bgPtr->offset.tick_a;
      bgPtr->ticks[bgPtr->numTicks].x2 = bgPtr->offset.tick_a + bgPtr->tickLength;
      bgPtr->ticks[bgPtr->numTicks].y1 =
	bgPtr->ticks[bgPtr->numTicks].y2 = ValueToPixel(bgPtr, v, 1);
    }
  } else {
    for ( v = bgPtr->baseVal; v >= start;
	  v-=bgPtr->tickInterval, bgPtr->numTicks++ ) {
      bgPtr->ticks[bgPtr->numTicks].y1 = bgPtr->offset.tick_a;
      bgPtr->ticks[bgPtr->numTicks].y2 = bgPtr->offset.tick_a + bgPtr->tickLength;
      bgPtr->ticks[bgPtr->numTicks].x1 =
	bgPtr->ticks[bgPtr->numTicks].x2 = ValueToPixel(bgPtr, v, 1);
    }

    for ( v = bgPtr->baseVal+bgPtr->tickInterval; v <= end;
	 v+=bgPtr->tickInterval, bgPtr->numTicks++ ) {
      bgPtr->ticks[bgPtr->numTicks].y1 = bgPtr->offset.tick_a;
      bgPtr->ticks[bgPtr->numTicks].y2 = bgPtr->offset.tick_a + bgPtr->tickLength;
      bgPtr->ticks[bgPtr->numTicks].x1 =
	bgPtr->ticks[bgPtr->numTicks].x2 = ValueToPixel(bgPtr, v, 1);
    }
  }
}

/*
 * Display the min and max values
 */
static void
DisplayMinmax(bgPtr)
    Bargraph *bgPtr;
{
  register Tk_Window tkwin = bgPtr->tkwin;
  char valueString[40];
  int width, len, x, y;
  Tk_FontMetrics fm;
  Tk_GetFontMetrics(bgPtr->tkfont, &fm);

  format_double(bgPtr, valueString, bgPtr->fromVal);
  len = strlen(valueString);
  width = Tk_TextWidth(bgPtr->tkfont, valueString, len);

  if (bgPtr->orient == ORIENT_VERTICAL) {
    x = bgPtr->offset.minmax0_x +
      SHOW_RANGE_OR_BASEV(bgPtr)*bgPtr->minmaxPixels - width;
    y = bgPtr->offset.minmax0_y;
  } else {
    x = bgPtr->offset.minmax0_x;
    y = bgPtr->offset.minmax0_y + fm.ascent;
  }
  Tk_DrawChars(bgPtr->display, Tk_WindowId(tkwin), bgPtr->textGC,
	       bgPtr->tkfont, valueString, len, x, y);

  format_double(bgPtr, valueString, bgPtr->toVal);
  len = strlen(valueString);
  width = Tk_TextWidth(bgPtr->tkfont, valueString, len);

  if (bgPtr->orient == ORIENT_VERTICAL) {
    x = bgPtr->offset.minmax1_x +
      SHOW_RANGE_OR_BASEV(bgPtr)*bgPtr->minmaxPixels - width;
    y = bgPtr->offset.minmax1_y;
  } else {
    x = bgPtr->offset.minmax1_x - width;
    y = bgPtr->offset.minmax1_y + fm.ascent;
  }
  Tk_DrawChars(bgPtr->display, Tk_WindowId(tkwin), bgPtr->textGC,
	       bgPtr->tkfont, valueString, len, x, y);
}

/*
 * Display the base value
 */
static void
DisplayBaseV(bgPtr)
    Bargraph *bgPtr;
{
  register Tk_Window tkwin = bgPtr->tkwin;
  double basev = bgPtr->baseVal;
  char valueString[40];
  int width, len, x, y;

  format_double(bgPtr, valueString, basev);
  len = strlen(valueString);
  width = Tk_TextWidth(bgPtr->tkfont, valueString, len);

  if (bgPtr->orient == ORIENT_VERTICAL) {
    Tk_FontMetrics fm;
    Tk_GetFontMetrics(bgPtr->tkfont, &fm);
    x = bgPtr->offset.lab_a + HAS_ALAB(bgPtr)*bgPtr->alabelPixels - width;
    y = ValueToPixel(bgPtr, basev, 1) + fm.descent;
  } else {
    x = ValueToPixel(bgPtr, basev, 1) - width/2;
    y = bgPtr->offset.lab_a;
  }
  Tk_DrawChars(bgPtr->display, Tk_WindowId(tkwin), bgPtr->textGC,
	       bgPtr->tkfont, valueString, len, x, y);
}

/*
 * Display the bar.
 */
static void
DisplayBar(bgPtr)
    Bargraph *bgPtr;
{
  register Tk_Window tkwin = bgPtr->tkwin;
  int length, y, a, b;
  double yy;
  int barBorderWidth = bgPtr->barBorderWidth;

	  EraseWing(bgPtr);

  if (bgPtr->mode == MODE_SLIDER) {
    /* In order to get a central line, draw this as two half-bars
     * around the value. Ensure the bar has a !0 borderwidth so that
     * the value is visible.
     */
    if (barBorderWidth == 0)
      barBorderWidth++;
    length = bgPtr->sliderlength/2;
    y = ValueToPixel(bgPtr, bgPtr->value, 0);
    yy = bgPtr->pixval;
    a = limit_ivalue(bgPtr->offset.bot_edge, bgPtr->offset.top_edge, y - length);
    b = limit_ivalue(bgPtr->offset.bot_edge, bgPtr->offset.top_edge, y + length);
    y = limit_ivalue(bgPtr->offset.bot_edge, bgPtr->offset.top_edge, y);
  } else {
    /* Compute ValueToPixel for endpoints, then difference these.
     * This will ensure that the length matches to the tick marks,
     * instead of sometimes generating off-by-one errors.
     */
    a = ValueToPixel( bgPtr, bgPtr->baseVal, 1);
    b = ValueToPixel( bgPtr, bgPtr->value, 1);
    if (a < b) {
      y = a;
      length = b + 1 - a;
    } else {
      y = b;
      length = a + 1 - b;
    }
  }

  if (bgPtr->orient == ORIENT_VERTICAL) {
    /* Erase entire bar region */
    Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin), bgPtr->border,
	bgPtr->offset.bar_x,				/* x */
	bgPtr->offset.bot_edge - bgPtr->max_height,	/* y */
	bgPtr->width,					/* width */
	bgPtr->max_height,				/* height */
		       0, TK_RELIEF_FLAT);

    /* Draw bar background */
    Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin),
	bgPtr->barBorder, 
	bgPtr->offset.bar_x,				/* x */
	bgPtr->offset.bot_edge - bgPtr->max_height,	/* y */
	bgPtr->width,					/* width */
	bgPtr->max_height,				/* height */
	bgPtr->barBorderWidth, bgPtr->barRelief );

    /* Draw bar */
    if (bgPtr->mode == MODE_SLIDER) {
      /*
       * Two bars: a->y, y->b
       */
      if (y > a) {
	Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin),
	  bgPtr->barColor, 
	  bgPtr->offset.bar_x,				/* x */
	  a,						/* y */
	  bgPtr->width,					/* width */
	  y + 1 - a,					/* height */
	  barBorderWidth, bgPtr->barRelief );
      }
      if (b > y) {
	Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin),
	  bgPtr->barColor, 
	  bgPtr->offset.bar_x,				/* x */
	  y,						/* y */
	  bgPtr->width,					/* width */
	  b + 1 - y,					/* height */
	  barBorderWidth, bgPtr->barRelief );
      }
    } else {
      /*
       * Bar from y, length "length".
       */
      Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin),
	  bgPtr->barColor, 
	  bgPtr->offset.bar_x,				/* x */
	  y,						/* y */
	  bgPtr->width,					/* width */
	  length,					/* height */
	  barBorderWidth, bgPtr->barRelief );
    }

  } else {
    /* HORIZONTAL */

    /* Erase entire bar region */
    Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin), bgPtr->border,
	bgPtr->offset.bot_edge,				/* x */
	bgPtr->offset.bar_y,				/* y */
	bgPtr->max_height,				/* width */
	bgPtr->width,					/* height */
		       0, TK_RELIEF_FLAT);

    /* Draw bar background */
    Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin),
	bgPtr->barBorder, 
	bgPtr->offset.bot_edge,				/* x */
	bgPtr->offset.bar_y,				/* y */
	bgPtr->max_height,				/* width */
	bgPtr->width,					/* height */
	bgPtr->barBorderWidth, bgPtr->barRelief );

    /* Draw bar */
    if (bgPtr->mode == MODE_SLIDER) {
      /*
       * Two bars: a->y, y->b
       */
      if (y > a) {
	Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin),
	    bgPtr->barColor, 
	    a,						/* x */
	    bgPtr->offset.bar_y,			/* y */
	    y + 1 - a,					/* width */
	    bgPtr->width,				/* height */
	    barBorderWidth, bgPtr->barRelief );
      }
      if (b > y) {
	Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin),
	    bgPtr->barColor, 
	    y,						/* x */
	    bgPtr->offset.bar_y,			/* y */
	    b + 1 - y,					/* width */
	    bgPtr->width,				/* height */
	    barBorderWidth, bgPtr->barRelief );
      }

    } else {
      /*
       * Bar from y, length "length".
       */
      Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin),
	  bgPtr->barColor, 
	  y,						/* x */
	  bgPtr->offset.bar_y,				/* y */
	  length,					/* width */
	  bgPtr->width,					/* height */
	  barBorderWidth, bgPtr->barRelief );
    }
  }
   yy = bgPtr->pixval;
   DrawWing(bgPtr, yy);

}

/*
 *--------------------------------------------------------------
 *
 * DisplayBargraph --
 *
 *	This procedure redraws the contents of a vert. or hor. Bargraph
 *	window.	 It is invoked as a do-when-idle handler, so it only
 *	runs when there's nothing else for the application to do.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information appears on the screen.
 *
 *--------------------------------------------------------------
 */
static void
DisplayBargraph(clientData)
     ClientData clientData;	/* Information about widget. */
{
  register Bargraph *bgPtr = (Bargraph *) clientData;
  register Tk_Window tkwin = bgPtr->tkwin;
  double basev = bgPtr->baseVal;
  double minv = bgPtr->fromVal;
  double maxv = bgPtr->toVal;
  int max_ticks;
  double pixel_interval;

  /* When we are done, there will be no redraw pending; clear that bit. */
  bgPtr->flags &= ~REDRAW_PENDING;
  if ((bgPtr->tkwin == NULL) || !Tk_IsMapped(tkwin)) {
    return;
  }

  if (bgPtr->flags & CLEAR_NEEDED) {
    XFillRectangle(bgPtr->display, Tk_WindowId(bgPtr->tkwin),
		   Tk_3DBorderGC(tkwin, bgPtr->border, TK_3D_FLAT_GC),
		   0, 0, Tk_Width(tkwin), Tk_Height(tkwin));
  }

  /*
   * Display the label, centered in the window if there is enough space.
   * Otherwise left justified and clipped on the right.
   */
  if (bgPtr->label && (bgPtr->flags & DISPLAY_LABEL))
    DisplayLabel(bgPtr);

  /*
   *  Display the value, right justified to the left of the bar and centered
   *  within the bar.
   */
  if (bgPtr->showValue && (bgPtr->flags & DISPLAY_VALUE))
    DisplayValue(bgPtr);

  bgPtr->numTicks = 0;
  if ( bgPtr->tickInterval != 0.) {
    pixel_interval = bgPtr->max_height *
      bgPtr->tickInterval /
      ((minv < maxv) ? (maxv-minv) : (minv-maxv));
    max_ticks = bgPtr->max_height / (int)pixel_interval + 1 +
      bgPtr->n_alabels +	/* in case there are -tick'd -alabels */
      bgPtr->n_blabels;	/* in case there are -tick'd -blabels */
  } else {
    max_ticks = 1 +
      bgPtr->n_alabels +	/* in case there are -tick'd -alabels */
      bgPtr->n_blabels;	/* in case there are -tick'd -blabels */
  }
  if ((bgPtr->flags & DISPLAY_LABELS)) {
    if (bgPtr->ticks != NULL) ckfree((char *)bgPtr->ticks);
    bgPtr->ticks = (XSegment *) ckalloc( (max_ticks+2) * sizeof(XSegment));
  }

  /*
   * draw alabels, blabels.  Ticks for same are computed but not yet drawn.
   */
  if (HAS_ALAB(bgPtr) && (bgPtr->flags & DISPLAY_LABELS))
    DisplayAlabels(bgPtr);

  if (HAS_BLAB(bgPtr) && (bgPtr->flags & DISPLAY_LABELS))
    DisplayBlabels(bgPtr);

  /* Compute the ticks */
  if (SHOW_TICKS(bgPtr) && (bgPtr->flags & DISPLAY_TICKS))
    ComputeTicks(bgPtr);

  /* If any ticks were created, draw them.  */
  if (bgPtr->numTicks > 0) {
    XDrawSegments(bgPtr->display, Tk_WindowId(tkwin), bgPtr->tickGC,
		  bgPtr->ticks, bgPtr->numTicks);
  }

  /* draw fromVal and toVal */
  if (SHOW_RANGE(bgPtr) && (bgPtr->flags & DISPLAY_LABELS))
    DisplayMinmax(bgPtr);

  /* draw base value if required */
  if (basev != minv && basev != maxv && (bgPtr->flags & DISPLAY_LABELS))
    DisplayBaseV(bgPtr);

  /* * Display the bar */

  if ((bgPtr->flags & DISPLAY_BAR))
    DisplayBar(bgPtr);

  /* Display the border */
  if ((bgPtr->flags & DISPLAY_BORDER))
    Tk_Draw3DRectangle(tkwin, Tk_WindowId(tkwin), bgPtr->border,
		       0, 0, Tk_Width(tkwin), Tk_Height(tkwin),
		       bgPtr->borderWidth, bgPtr->relief);

  /* All done with the updating */
  bgPtr->flags &= ~DISPLAY_ALL;
}

/*
 *----------------------------------------------------------------------
 *
 * ValueToPixel --
 *
 *	Convert a value to the x- or y-coordinate corresponding to that value,
 *	depending on whether the Bargraph is vertical or horizontal,
 *	respectively.
 *
 * Results:
 *	An integer value giving the pixel location corresponding
 *	to the value.  If the "limit" flag is set, then the returned
 *	value is limited to the fromVal..toVal range; otherwise,
 *	it can have any value.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
static int
ValueToPixel(bgPtr, value, limit)
    register Bargraph *bgPtr;	/* Information about widget. */
    double value;
    int limit;
{
  double minv = bgPtr->fromVal;
  double maxv = bgPtr->toVal;
  int length;

  if (limit)
    value = limit_dvalue(minv, maxv, value);

  /* Distance in pixels from minv; can be positive or negative. */
  length = (int) (bgPtr->max_height * (value - minv) / (maxv - minv));

  if (bgPtr->orient == ORIENT_VERTICAL) {
    bgPtr->pixval = bgPtr->offset.bot_edge - length;
  } else {
    bgPtr->pixval = bgPtr->offset.bot_edge + length;
  }
  return (int) bgPtr->pixval;
}

/*
 *--------------------------------------------------------------
 *
 * BargraphEventProc --
 *
 *	This procedure is invoked by the Tk dispatcher for various
 *	events on Bargraphs.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	When the window gets deleted, internal structures get
 *	cleaned up.  When it gets exposed, it is redisplayed.
 *
 *--------------------------------------------------------------
 */
static void
BargraphEventProc(clientData, eventPtr)
    ClientData clientData;	/* Information about window. */
    XEvent *eventPtr;		/* Information about event. */
{
  Bargraph *bgPtr = (Bargraph *) clientData;

  switch (eventPtr->type) {
  case Expose:
    if (eventPtr->xexpose.count == 0)
      EventuallyRedrawBargraph(bgPtr, DISPLAY_ALL | CLEAR_NEEDED);
    break;
  case DestroyNotify:
    if (bgPtr->tkwin != NULL) {
      bgPtr->tkwin = NULL;
      Tcl_DeleteCommandFromToken(bgPtr->interp, bgPtr->widgetCmd);
    }

    Tcl_CancelIdleCall(DisplayBargraph, (ClientData) bgPtr);
    Tcl_EventuallyFree((ClientData) bgPtr, DestroyBargraph);
    break;
  case ConfigureNotify:
    ComputeBargraphGeometry(bgPtr);
    EventuallyRedrawBargraph(bgPtr, DISPLAY_ALL | CLEAR_NEEDED);
    break;
  case FocusIn:
    if (eventPtr->xfocus.detail != NotifyInferior) {
      bgPtr->flags |= GOT_FOCUS;
      if (bgPtr->highlightWidth > 0) {
	EventuallyRedrawBargraph(bgPtr, DISPLAY_ALL);
      }
    }
    break;
  case FocusOut:
    if (eventPtr->xfocus.detail != NotifyInferior) {
      bgPtr->flags &= ~GOT_FOCUS;
      if (bgPtr->highlightWidth > 0) {
	EventuallyRedrawBargraph(bgPtr, DISPLAY_ALL);
      }
    }
    break;
  }
}

/*
 *--------------------------------------------------------------
 *
 * SetBargraphValue --
 *
 *	This procedure changes the value of a Bargraph and invokes
 *	a Tcl command to reflect the current position of a Bargraph
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	A Tcl command is invoked, and an additional error-processing
 *	command may also be invoked.  The Bargraph's slider is redrawn.
 *
 *--------------------------------------------------------------
 */
static void
SetBargraphValue(bgPtr, value)
     register Bargraph *bgPtr;	/* Info about widget. */
     double value;		/* New value for Bargraph.  Gets
				 * adjusted if it's off the Bargraph. */
{
  if (value == bgPtr->value)
    return;

  bgPtr->value = value;
  EventuallyRedrawBargraph(bgPtr, DISPLAY_BAR | DISPLAY_VALUE);
}

/*
 *----------------------------------------------------------------------
 *
 * BargraphCmdDeletedProc --
 *
 *	This procedure is invoked when a widget command is deleted.  If
 *	the widget isn't already in the process of being destroyed,
 *	this command destroys it.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The widget is destroyed.
 *
 *----------------------------------------------------------------------
 */
static void
BargraphCmdDeletedProc(clientData)
    ClientData clientData;	/* Pointer to widget record for widget. */
{
  Bargraph *bgPtr = (Bargraph *) clientData;
  Tk_Window tkwin = bgPtr->tkwin;

  /*
   * This procedure could be invoked either because the window was
   * destroyed and the command was then deleted (in which case tkwin
   * is NULL) or because the command was deleted, and then this procedure
   * destroys the widget.
   */

  if (tkwin != NULL) {
    bgPtr->tkwin = NULL;
    Tk_DestroyWindow(tkwin);
  }
}

/*
 *--------------------------------------------------------------
 *
 * EventuallyRedrawBargraph --
 *
 *	Arrange for part or all of a Bargraph widget to redrawn at
 *	the next convenient time in the future.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	If "what" is DISPLAY_BAR then just the slider and the
 *	value readout will be redrawn;	if "what" is DISPLAY_ALL
 *	then the entire widget will be redrawn.
 *
 *--------------------------------------------------------------
 */
static void
EventuallyRedrawBargraph(bgPtr, what)
    register Bargraph *bgPtr;	/* Information about widget. */
    int what;			/* What to redraw */
{

  /* Make sure it's gonna redraw everything we want */
  bgPtr->flags |= what;
  if ( (bgPtr->tkwin == NULL) || !Tk_IsMapped(bgPtr->tkwin) ||
       (bgPtr->flags & REDRAW_PENDING) ) {
    return;
  }

  bgPtr->flags |= REDRAW_PENDING;
  Tcl_DoWhenIdle(DisplayBargraph, (ClientData) bgPtr);

}

/*
 * Format a double value for printing.
 * If fractional part is .0..., trim it off
 * so that it looks like an integer value.
 */

static void
format_double(bgPtr, result, v)
     Bargraph *bgPtr;
     char *result;	/* result buffer; assumed large enough */
     double v;		/* value to format */
{
  char *p, *s;

#if 0
  Tcl_PrintDouble(bgPtr->interp, v, result);
#else
  sprintf(result, "%.*f", bgPtr->digits, v);
#endif

  p = strchr(result, '.');
  if (p) {
    /* There is a decimal point; trim trailing 0's */
    for (s = p + strlen(p); s > p && *(s-1) == '0'; s--)
      *(s-1) = '\0';
    /* Assertion: s pts to terminating null.
     * If prev char is '.', remove it.
     */
    if ((s-1) == p)
      *p = '\0';
  }
}

/* Keep a value between minv and maxv.	*/
double
limit_dvalue(minv, maxv, value)
     double minv, maxv, value;
{
  if ( maxv > minv ) {
    if ( value > maxv )
      value = maxv;
    else if ( value < minv )
      value = minv;
  } else {
    /* maxv <= minv */
    if ( value < maxv )
      value = maxv;
    else if ( value > minv )
      value = minv;
  }
  return value;
}

int
limit_ivalue(minv, maxv, value)
     int minv, maxv, value;
{
  if ( maxv > minv ) {
    if ( value > maxv )
      value = maxv;
    else if ( value < minv )
      value = minv;
  } else {
    /* maxv <= minv */
    if ( value < maxv )
      value = maxv;
    else if ( value > minv )
      value = minv;
  }
  return value;
}

/*
 * Erase xor'd wings, but note that this doesn't erase the line.
 * overlaid on the slider or bar.
 */
static void
EraseWing(Bargraph *bgPtr)
{
    register Tk_Window tkwin = bgPtr->tkwin;

    if (!SHOW_WINGS(bgPtr))
        return;

    if (bgPtr->awing_showing) {
        /* Erase prior wings */
        if (bgPtr->wing_Aside) {
            XDrawLines(Tk_Display(tkwin), Tk_WindowId(tkwin),
                bgPtr->wingGC, bgPtr->Awing, 3, CoordModeOrigin);
        }
    }
    if (bgPtr->bwing_showing) {
        /* Erase prior wings */
        if (bgPtr->wing_Bside) {
            XDrawLines(Tk_Display(tkwin), Tk_WindowId(tkwin),
                bgPtr->wingGC, bgPtr->Bwing, 3, CoordModeOrigin);
        }
    }
}

static void
DrawWing(Bargraph *bgPtr,
        double y)       /* The ValueToPixel value, but in f.p. */
{
    register Tk_Window tkwin = bgPtr->tkwin;
    float y_Aside, y_Bside, y_Amin, y_Bmin;
    float Acrowding, Bcrowding;
    void do_splint(struct splineinfo *sp, double r, float *yinterp,
                                        float *mindist, float *crowding);

    if (!SHOW_WINGS(bgPtr))
        return;

    do_splint(&bgPtr->sp_a, y, &y_Aside, &y_Amin, &Acrowding);
    do_splint(&bgPtr->sp_b, y, &y_Bside, &y_Bmin, &Bcrowding);
    #if 0
        printf("y=%.2f  y_Aside=%.2f y_Amin=%.2f y_Acrowding=%.2f\n",
                                        y, y_Aside, y_Amin, y_Acrowding);
    #endif


    if (bgPtr->orient == ORIENT_VERTICAL) {
        bgPtr->wing_Aside = (y_Aside >= 0.);
        if (bgPtr->wing_Aside && y_Amin < bgPtr->wing_Amindist &&
                                        Acrowding <= bgPtr->wing_Amindist) {
            bgPtr->Awing[0].x = bgPtr->offset.lab_a;
            bgPtr->Awing[0].y = (int) y_Aside;
            bgPtr->Awing[1].x = bgPtr->offset.tick_a;
            bgPtr->Awing[1].y = (int) y_Aside;
            bgPtr->Awing[2].x = bgPtr->Awing[1].x + bgPtr->tickLength;
            bgPtr->Awing[2].y = (int) y;
            XDrawLines(Tk_Display(tkwin), Tk_WindowId(tkwin),
                    bgPtr->wingGC, bgPtr->Awing, 3, CoordModeOrigin);
            bgPtr->awing_showing = 1;
        } else {
            bgPtr->awing_showing = 0;
        }
        bgPtr->wing_Bside = (y_Bside >= 0.);
        if (bgPtr->wing_Bside && y_Bmin < bgPtr->wing_Bmindist &&
                                        Bcrowding <= bgPtr->wing_Bmindist) {
            bgPtr->Bwing[0].x = bgPtr->offset.tick_b;
            bgPtr->Bwing[0].y = (int) y;
            bgPtr->Bwing[1].x = bgPtr->offset.tick_b + bgPtr->tickLength;
            bgPtr->Bwing[1].y = (int) y_Bside;
            bgPtr->Bwing[2].x = Tk_Width(tkwin) - bgPtr->borderWidth;
            bgPtr->Bwing[2].y = (int) y_Bside;
            XDrawLines(Tk_Display(tkwin), Tk_WindowId(tkwin),
                    bgPtr->wingGC, bgPtr->Bwing, 3, CoordModeOrigin);
            bgPtr->bwing_showing = 1;
        } else {
            bgPtr->bwing_showing = 0;
        }
        if (bgPtr->wingOverlay &&
                        (bgPtr->awing_showing || bgPtr->bwing_showing)) {
            XDrawLine(Tk_Display(tkwin), Tk_WindowId(tkwin),
                    bgPtr->wingoverlayGC, bgPtr->offset.bar_x, (int) y,
                    bgPtr->offset.bar_x + bgPtr->width, (int) y);
        }
    } else {
        /* HORIZONTAL */
        /* Wings not implemented for horizontal! */
    }
}

