#include <stdio.h>
#include <math.h>
#include <stddef.h>
#include <stdlib.h>
#define ITMAX 200

#include "pkg.h"
#include "pfit.h"

/* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ */

/*
 *
 * Public Routines
 */

void Pfit_free_sp(struct splineinfo *sp);

void Pfit_initLabPosn(struct splineinfo *sp, int algorithm,
			int lo, int hi, int cheight, int nlab, float *lab);
void Pfit_clearLabPosn(struct splineinfo *sp);
float *Pfit_mkLabPosn(struct splineinfo *sp, int *niter, float *fret);
void Pfit_splint(struct splineinfo *sp, double x, float *yinterp,
					float *mindist, float *crowding);

/* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ */

/*
 *
 * Numerical Recipes Routines (or code derived therefrom)
 */

static void powell(float p[], float **xi, int n, float ftol,
	int *iter, float *fret, float (*func)(float []));
static void splint_ab(float xa[], float ya[], float y2a[], float y2wgt,
	    int n, float x, float *y, float *A, float *B,
	    int crowd_uselower, int crowd_useupper, float *crowding);
static void spline(float x[], float y[], int n, float yp1,
					float ypn, float y2[]);

/* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ */

/*
 *
 * Free substitutes for variants of NR code
 */
static void nrerror(char error_text[]);
static float *vector(long n);
static int *ivector(long n);
static float **matrix(long r, long c);
static void free_vector(float *v);
static void free_ivector(int *v);
static void free_matrix(float **m);

/* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ */

static float distance(float p[]);

static float *mkLabPosn_powell(struct splineinfo *sp, int *nit, float *f);
static float *mkLabPosn_uniform(struct splineinfo *sp, int *nit, float *f);

static struct {
    int np;		/* input: number of positions */
    int alg;		/* input: algorithm */
    int *idx;		/* work: pre-sort indices */
    float *p_desired;	/* input: positions, desired */
    float *p;		/* work: positions, iterating */
    float *q;		/* work: helper for sorting/unsorting */
    float **xi;		/* work: work matrix */
    struct splineinfo *sp;
    float height;	/* for convenience: upper - lower */
    float cheight;	/* input: height of characters */
    float chalfheight;	/* computed: half height of characters */
    float ftol;		/* something small: tolerance for stopping */
    int ncalls;		/* how many times distance is called */
} widg = {0,	/* np */
	ALGO_POWELL,	/* alg */
	NULL,	/* idx */
	NULL,	/* p_desired */
	NULL,	/* p */
	NULL,	/* q */
	NULL};		/* xi */

/* ************************************************************************ */

#ifdef PROGRAM
int
main(int argc, char **argv)
{
    float d, ftol, fret, sumsq;
    int i, j, n, niter;
    int lower, upper, height;
    float *p, **xi;
    float *res;
    float x[1000] = {3, 9, 12, 16, 17, 20};

    if (argc <= 4) {
	n = 6;
	lower = 0;
	upper = 100;
	height = 10;
    } else {
	lower = atoi(argv[1]);
	upper = atoi(argv[2]);
	height = atoi(argv[3]);
	n = argc - 4;
	for (i=4; i<argc; i++) {
	    x[i-4] = atof(argv[i]);
	}
    }
    Pfit_initLabPosn(0, 0, lower, upper, height, n, x);

    res = Pfit_mkLabPosn(0, &niter, &fret);
    printf("niter %d   fret %f\n", niter, fret);
    for (sumsq = 0., i=1; i<=widg.np; i++) {
	d = res[i-1] - widg.p_desired[i];
	sumsq += d * d;
	printf("%2d  %8.2f  %8.2f  (%f)\n", i, widg.p_desired[i], res[i-1], d);
    }
    printf("rms = %f\n", sqrt(sumsq/(widg.np-1)));

}
#endif

/* ************************************************************************ */

void
Pfit_splint(
    struct splineinfo *sp,
    double x,
    float *yinterp,
    float *mindist,
    float *crowding)
{
    float a, b;

    if (sp->ns <= 0) {
	*yinterp = -1.;
    } else {
	splint_ab(sp->x, sp->y, sp->y2, .0, sp->ns, (float) x, yinterp,
			    &a, &b, sp->use_lower, sp->use_upper, crowding);
	if ((sp->upper - (float) x) == a && !sp->use_upper) {
	    *mindist = b;
	} else if (((float) x - sp->lower) == b && !sp->use_lower) {
	    *mindist = a;
	} else if (a < b) {
	    *mindist = a;
	} else {
	    *mindist = b;
	}
    }
}

/* ************************************************************************ */

void
Pfit_clearLabPosn(
    struct splineinfo *sp)
{
    sp->ns = -1;
}

/* ************************************************************************ */

float *
Pfit_mkLabPosn(
    struct splineinfo *sp,
    int *niter,		/* output: number of iterations */
    float *fret)	/* output: final return value from the distance func */
{
    if (widg.alg == ALGO_UNIFORM) {
	return mkLabPosn_uniform(sp, niter, fret);
    } else {
	return mkLabPosn_powell(sp, niter, fret);
    }
    /* NOTREACHED */
}

/* ************************************************************************ */

static float *
mkLabPosn_powell(
    struct splineinfo *sp,
    int *niter,		/* output: number of iterations */
    float *fret)	/* output: final return value from the distance func */
{
    int i, j;
#if DEBUG_LAB
    {
	printf("np = %d\n", widg.np);
	printf("ftol = %f\n", widg.ftol);
	printf("lower = %f\n", sp->lower);
	printf("upper = %f\n", sp->upper);
	printf("height = %f\n", widg.height);
	printf("cheight = %f\n", widg.cheight);
	printf("chalfheight = %f\n", widg.chalfheight);
	for (i=1; i<=widg.np; i++) printf("%7.2f ", widg.p[i]); printf("\n");
	for (i=1; i<=widg.np; i++) printf("%7.2f ", widg.p_desired[i]); printf("\n");
	printf("---------------------\n");
    }
#endif

    powell(widg.p, widg.xi, widg.np, widg.ftol, niter, fret, distance);

#if DEBUG_LAB
    for (i=1; i<=widg.np; i++) printf("%7.2f ", widg.p[i]); printf("\n");
#endif

    /* We now have desired and fitted positions (aka tick and label positions).
     * These also represent the x and y=f(x) positions for spline fitting.
     * Therefore copy them to the spline arrays and set up the spline interp;
     * do this before the positions become unsorted.
     */
    j = 0;
    sp->use_lower = 1;
    if (widg.p[1] > sp->lower && widg.p_desired[1] > sp->lower) {
	j++;
	sp->use_lower = 0;	/* lower is "fake"; don't need ticks there */
	sp->x[j] = sp->lower;
	sp->y[j] = sp->lower;
    }

    for (i = 1; i <= widg.np; i++) {
	j++;
	sp->x[j] = widg.p_desired[i];
	sp->y[j] = widg.p[i];
    }

    sp->use_upper = 1;
    if (widg.p[widg.np] < sp->upper && widg.p_desired[widg.np] < sp->upper) {
	j++;
	sp->use_upper = 0;	/* upper is "fake"; don't need ticks there */
	sp->x[j] = sp->upper;
	sp->y[j] = sp->upper;
    }
    sp->ns = j;

    /* And set up the spline interp. */
    spline(sp->x, sp->y, sp->ns, 1.e31, 1.e31, sp->y2);
#if DEBUG_SPLINE
    printf("after spline\n");
    for (i = 1; i <= sp->ns; i++) {
	printf("x[%d] = %.2f     y[%d] = %.2f    y2[%d] = %.2f\n",
	i, sp->x[i], i, sp->y[i], i, sp->y2[i]);
    }
#endif

    /*
     * The Pfit_initLabPosn routine sorted the p_desired array before powell()
     * got called.  Rearrange the order now to match the original p[] and
     * p_desired[] order.  Note that the idx[] array is in order of
     * ascending p_desired, _before_ p_desired was sorted.
     */
    for (i = 1; i <= widg.np; i++)
	widg.q[widg.idx[i]] = widg.p_desired[i];
    for (i = 1; i <= widg.np; i++)
	widg.p_desired[i] = widg.q[i];

    for (i = 1; i <= widg.np; i++)
	widg.q[widg.idx[i]] = widg.p[i];
    for (i = 1; i <= widg.np; i++)
	widg.p[i] = widg.q[i];

#if DEBUG_LABEL
    printf("niter %d   fret %f\n", *niter, *fret);
    for (sumsq = 0., i=1; i<=widg.np; i++) {
	d = widg.p[i] - widg.p_desired[i];
	sumsq += d * d;
	printf("%2d  %8.2f  %8.2f  (%f)\n", i, widg.p_desired[i], widg.p[i], d);
    }
    printf("rms = %f\n", sqrt(sumsq/(widg.np-1)));
#endif
    return (&widg.p[1]);
}

/* ************************************************************************ */

static float *
mkLabPosn_uniform(
    struct splineinfo *sp,
    int *niter,		/* output: number of iterations */
    float *fret)	/* output: final return value from the distance func */
{
    int i, j;
    float dx;

    dx = widg.height / widg.np;
    for (i=1; i<=widg.np; i++) {
	widg.p[i] = sp->lower + (i-0.5)*dx;
    }

    /* We now have desired and fitted positions (aka tick and label positions).
     * These also represent the x and y=f(x) positions for spline fitting.
     * Therefore copy them to the spline arrays and set up the spline interp;
     * do this before the positions become unsorted.
     */
    j = 0;
    sp->use_lower = 1;
    if (widg.p[1] > sp->lower && widg.p_desired[1] > sp->lower) {
	j++;
	sp->use_lower = 0;	/* lower is "fake"; don't need ticks there */
	sp->x[j] = sp->lower;
	sp->y[j] = sp->lower;
    }

    for (i = 1; i <= widg.np; i++) {
	j++;
	sp->x[j] = widg.p_desired[i];
	sp->y[j] = widg.p[i];
    }

    sp->use_upper = 1;
    if (widg.p[widg.np] < sp->upper && widg.p_desired[widg.np] < sp->upper) {
	j++;
	sp->use_upper = 0;	/* upper is "fake"; don't need ticks there */
	sp->x[j] = sp->upper;
	sp->y[j] = sp->upper;
    }
    sp->ns = j;



#if (DEBUG_LAB)
    for (i=1; i<=widg.np; i++) printf("%7.2f ", widg.p[i]); printf("\n");
#endif

    /*
     * The Pfit_initLabPosn routine sorted the p_desired array before do_*()
     * got called.  Rearrange the order now to match the original p[] and
     * p_desired[] order.  Note that the idx[] array is in order of
     * ascending p_desired, _before_ p_desired was sorted.
     */
    for (i = 1; i <= widg.np; i++)
	widg.q[widg.idx[i]] = widg.p_desired[i];
    for (i = 1; i <= widg.np; i++)
	widg.p_desired[i] = widg.q[i];

    for (i = 1; i <= widg.np; i++)
	widg.q[widg.idx[i]] = widg.p[i];
    for (i = 1; i <= widg.np; i++)
	widg.p[i] = widg.q[i];

#if DEBUG_LAB
    printf("++++++++++++\n");
    for (i=1; i<=widg.np; i++) printf("%7.2f ", widg.p[i]); printf("\n");
    for (i=1; i<=widg.np; i++) printf("%7.2f ", widg.p_desired[i]); printf("\n");
#endif
#if DEBUG_LABEL
    {
    float sumsq;
    printf("niter %d   fret %f\n", *niter, *fret);
    for (sumsq = 0., i=1; i<=widg.np; i++) {
	d = widg.p[i] - widg.p_desired[i];
	sumsq += d * d;
	printf("%2d  %8.2f  %8.2f  (%f)\n", i, widg.p_desired[i], widg.p[i], d);
    }
    }
    printf("rms = %f\n", sqrt(sumsq/(widg.np-1)));
#endif
    return (&widg.p[1]);
}

/* ************************************************************************ */

/* For use in qsort.
 * Input: two idx values.  Don't compare actual idx values; instead, use
 * them to compare the p_desired values.  Thus, the idx values will be
 * re-ordered according to the p_desired values.
 */
static int
compare_p(
    const void *i1,
    const void *i2)
{
    float p1, p2;
    p1 = widg.p_desired[* (const int *) i1];
    p2 = widg.p_desired[* (const int *) i2];

    if (p1 < p2)
	return -1;
    else if (p1 == p2)
	return 0;
    else
	return 1;
}

/* ************************************************************************ */

void
Pfit_free_sp(struct splineinfo *sp)
{
    if (!sp)
	return;

    if (sp->nl <= 0)
	return;

    if (sp->x) {
	free_vector(sp->x);
	sp->x = NULL;
    }
    if (sp->y) {
	free_vector(sp->y);
	sp->y = NULL;
    }
    if (sp->y2) {
	free_vector(sp->y2);
	sp->y2 = NULL;
    }
}

/* ************************************************************************ */

void
Pfit_initLabPosn(
    struct splineinfo *sp,
    int algorithm,	/* ALGO_POWELL, ALGO_UNIFORM, ... */
    int lo,		/* lower end of widget, pixels */
    int hi,		/* upper end of widget, pixels */
    int cheight,	/* character height, pixels */
    int nlab,		/* number of labels to fix up */
    float *lab)		/* label positions, pixels */
{
    int i, j;
    float step;

    if (nlab < 1)
	return;

    widg.alg = algorithm;

    sp->lower = (float) lo;
    sp->upper = (float) hi;
    widg.sp = sp;
    widg.height = sp->upper - sp->lower;
    widg.cheight = (float) cheight;
    widg.chalfheight = ((float) cheight) / 2.0;
    widg.ftol = 1e-6;
    widg.ncalls = 0;

    if (widg.idx) {
	free_ivector(widg.idx);
	widg.idx = NULL;
    }
    if (widg.p) {
	free_vector(widg.p);
	widg.p = NULL;
    }
    if (widg.q) {
	free_vector(widg.q);
	widg.q = NULL;
    }
    if (widg.p_desired) {
	free_vector(widg.p_desired);
	widg.p_desired = NULL;
    }
    if (widg.xi) {
	free_matrix(widg.xi);
	widg.xi = NULL;
    }
    Pfit_free_sp(sp);

    widg.np = nlab;
    widg.p_desired = vector(widg.np);
    widg.idx = ivector(widg.np);
    widg.p = vector(widg.np);
    widg.q = vector(widg.np);
    sp->nl = widg.np + 2;
    sp->x = vector(sp->nl);	/* 2 extra points so that */
    sp->y = vector(sp->nl);	/* we can add spline info */
    sp->y2 = vector(sp->nl);	/* at the endpoints */
    widg.xi = matrix(widg.np, widg.np);

    /* widg.p[] contains the starting locations; they will iterate
     * to the final locations.
     */
    step = widg.height / nlab;
    for (i = 1; i <= nlab; i++) {
	/* widg.p[i] = step/2. + (i-1) * step; */
	widg.p[i] = sp->lower + step/2. + (i-1) * step;
    }

    /*
     * Work array
     */
    for (j = 1; j <=widg.np; j++) {
	for (i = 1; i <=widg.np; i++) {
	    widg.xi[j][i] = 0.0;
	}
    }
    for (i = 1; i <=widg.np; i++) {
	widg.xi[i][i] = 1.0;
    }

    /*
     * Desired label locations -- get as close to these as we can.
     */
    for (i = 1; i <= nlab; i++) {
	widg.p_desired[i] = lab[i-1];
    }
    /*
     * Record indices of desired locations.  Then sort them so that
     * the p_desired's are in increasing order.
     */
    for (i = 1; i <= nlab; i++) {
	widg.idx[i] = i;
    }
    qsort(&widg.idx[1], widg.np, sizeof(widg.idx[1]), compare_p);
    /*
     * The idx's are now in order of ascending p_desired: idx[i] is the
     * index of the i'th ascending p_desired[].
     * Change p_desired to be in the same order.
     */
    for (i = 1; i <= nlab; i++)
	widg.q[i] = widg.p_desired[widg.idx[i]];
    for (i = 1; i <= nlab; i++)
	widg.p_desired[i] = widg.q[i];

}

/* assumes p[] and p_desired[] are in increasing order */
static float
distance(float p[])
{
    #if DEBUG_DISTANCE
    static int ncall=0;
    #endif
    int i;
    float d, d2;
    float dist;

    widg.ncalls++;

    #if DEBUG_DISTANCE
	for (i=1; i<=widg.np; i++) printf("p=%.3f  ", p[i]); printf("\n");
    #endif

    dist = 0.;
    for (i = 1; i <=widg.np; i++) {
	#if 0
	    d = fabs(p[i] - widg.p_desired[i]);
	#else
	    d = pow((p[i] - widg.p_desired[i]), 2.0);
	#endif

	#if DEBUG_DISTANCE
	    printf("   p[%d]=%.3f ; %.3f ... d = %.3f\n",
				i, p[i], widg.p_desired[i], d);
	#endif

	#if 1
	if (p[i] < (widg.sp->lower + widg.chalfheight)) {
	    /* off the bottom */
	    d += 2 * widg.cheight;
	    #if DEBUG_DISTANCE
		printf("v  d=%.1f\n", d);
	    #endif
	} else if (p[i] > (widg.sp->upper - widg.chalfheight)) {
	    /* off the top */
	    d += 2 * widg.cheight;
	    #if DEBUG_DISTANCE
		printf("^  d=%.1f\n", d);
	    #endif
	}
	#endif

	#if 1
	if (i > 1) {
	    d2 = p[i] - p[i-1];
	    if (d2 < 0.) {
		/* out of order! */
		d += 500000.;
		#if DEBUG_DISTANCE
		    printf("O  d=%.1f\n", d);
		#endif
	    } else if (d2 < widg.cheight) {
		/* partial or full overlap ; charge according to amount.
		 * Note that using  -k1 + (k2 * overlap) [where k2 > k1]
		 * means that it's better to have two slight overlap than
		 * one larger overlap.
		 */
		d += 150. * (widg.cheight - d2);
		#if DEBUG_DISTANCE
		    printf("o  d=%.1f d2=%.6f p[%d]=%.2f p[%d]=%.2f\n",
		    		d, d2, i, p[i], i-1, p[i-1]);
		#endif
	    }
	}
	#endif
	dist += d;
    }
    #if DEBUG_DISTANCE
	printf("ncall=%d dist=%f\n", ++ncall, dist);
    #endif
    return dist;
}


/* ==================================================================== */
/* ==================================================================== */
/* Freely-redistributable re-implementations of some NR code.
 * Note that we've modified the interfaces a bit, so DON'T try to
 * use this with Genuine(TM) Numerical Recipes code.
 */
/* ==================================================================== */
/* ==================================================================== */




/* ******************************************************************** */

static void
nrerror(char *errmsg)
{
    fprintf(stderr,"numerical code error: %s\n", errmsg);
}

/* ******************************************************************** */

static float *
vector(long n)
/* allocate a float array with valid indexes 0..n */
{
    float *vec;

    if (!(vec = (float *)malloc(((n+1)*sizeof(float)))))
	nrerror("allocation failure in vector()");
    return vec;
}

/* ******************************************************************** */

static int *
ivector(long n)
/* allocate an int array with valid indexes 0..n */
{
	int *vec;

	if (!(vec = (int *)malloc(((n+1)*sizeof(int)))))
	    nrerror("allocation failure in ivector()");
	return vec;
}

/* ******************************************************************** */

static float **
matrix(long nrow, long ncol)
/* allocate a float matrix with valid indices range 0..nrow, 0..ncol */
{
    int i;
    float **mat; /* ptr to array of pointers to rows */

    /* allocate pointers to rows */
    if (!(mat = (float **) malloc((nrow+1)*sizeof(float*)))) {
	nrerror("allocation failure 1 in matrix()");
	return NULL;
    }

    /* allocate rows and set pointers to them */
    if (!(mat[1] = (float *) malloc((nrow*(ncol+1))*sizeof(float)))) {
	nrerror("allocation failure 2 in matrix()");
	return NULL;
    }

    for (i=2; i <= ncol; i++)
	mat[i] = mat[i-1] + ncol;

    return mat;
}

/* ******************************************************************** */

static void
free_vector(float *v)
{
    free((void *) v);
}

/* ******************************************************************** */

static void
free_ivector(int *v)
{
    free((void *) v);
}

/* ******************************************************************** */

static void
free_matrix(float **mat)
{
    free((void *) (mat[1]));
    free((void *) (mat));
}

/* ******************************************************************** */
/* ******************************************************************** */
#ifdef HAVE_NR
    #include "recipes.c"
#else
    #include "recipes_no.c"
#endif
