/*
 * tkPie.c --
 *
 *	This module implements a piegraph widget for the Tk toolkit.
 *
 *	Based on previous Pie widget, completely rewritten.
 */

#include "pkg.h"

/* 0==M_DEGREE == 3 o'clock, increasing counter-clockwise */
#define	M_DEGREE		23040	/* 360 * 64 for XArcs */
#define	deg64_2_rads(d)		(M_PI * 2 * ((d) / M_DEGREE))

/* these are the allocated colors when noone chooses one */
#define MAX_COLOR		16
char *PieColors[MAX_COLOR] = { "green", "purple", "moccasin", "red", "yellow",
"skyblue", "grey75", "blue", "chocolate", "lavender", "sea green", "white",
"orange", "magenta", "cyan", "ivory" };

/*
 * Pie elts are a doubly linked list.
 * We need to maintain a rearrangeable order without a lot of overhead.
 * Since elts should never really exceed 50, this works best.
 *
 * The elts link to next and prev.  The first elts' prev points to
 * the last elt (for appending), but the last elts' next is NULL.
 */
typedef struct TkPieElt {
    char *name;			/* name of the elt */
    char *label;		/* user provided label of the elt */
    double value;		/* value of the elt */
    int explode;		/* if != 0, the distance to explode this
				 * elt from pie */
    Tk_3DBorder fg;		/* elt color */
    struct TkPieElt *next, *prev;
} TkPieElt;

typedef struct {
    Tk_Window tkwin;		/* Window that embodies the entry. NULL
				 * means that the window has been destroyed
				 * but the data structures haven't yet been
				 * cleaned up.*/
    Display *display;		/* Display containing widget.  Used, among
				 * other things, so that resources can be
				 * freed even after tkwin has gone away. */
    Tcl_Interp *interp;		/* Interpreter associated with entry. */
    Tcl_Command widgetCmd;	/* Token for entry's widget command. */
    Tk_OptionTable optionTable;	/* Table that defines configuration options
				 * available for this widget. */

    /* Border stuff */
    Tk_3DBorder border;		/* Structure used to draw 3-D border */
    int borderWidth;		/* Width of 3-D border around window */
    Tk_Cursor cursor;		/* Current cursor for window. */
    int relief;			/* 3-d effect: TK_RELIEF_RAISED, etc. */
    XColor *highlightBgColorPtr;/* Color for drawing traversal highlight
				 * area when highlight is off. */
    XColor *highlightColorPtr;	/* Color for drawing traversal highlight. */
    int highlightWidth;		/* Width in pixels of highlight to draw
				 * around widget when it has the focus.
				 * <= 0 means don't draw a highlight. */

    /* Pie stuff */
    int radius;			/* of the pie, config option */
    int angle;			/* viewing angle of the pie */
    int radX, radY;		/* actual radii given radius * angle */
    int shadow;			/* shadow, viewed when angle != 0 */
    int sliceborder;		/* whether a border should be drawn */
    int padX, padY;		/* X,Y padding of elements */
    double totalValue;		/* sum of elements, tracked */
    int numSlices;		/* number of slices, tracked */
    int maxExplode;		/* max explode value, tracked */
    int maxValueLen;		/* max value string length, tracked */
    int reqKeyLen, useKeyLen;	/* requested with of the key box */
    int reqValueLen, useValueLen;/* max value string length, tracked */
    int reqPercLen, usePercLen;	/* max percent string length, tracked */
    int reqLabelLen, useLabelLen;/* max label length, tracked */
    int autoColor;		/* auto color assignment var */
    TkPieElt *firstElt;	/* the actual slices, in linked list
			 * this allows us a cheap ordering system */

    /* misc configuration options */
    int precision;		/* option */
    char *legend;		/* permutations of KLVP for display order */
    int origin;			/* origin for pie elements */
    char *takeFocus;		/* Value of -takefocus option;  not used in
				 * the C code, but used by keyboard traversal
				 * scripts.  Malloc'ed, but may be NULL. */

    /* Text stuff */
    Tk_Font tkfont;		/* Information about text font, or NULL. */
    Tk_Font smallfont;		/* Information about values font, or NULL. */
    XColor *textColorPtr;	/* Color for drawing text.  */
    GC textGC, sfGC;		/* GC for drawing text and values.  */
    char *label;		/* label for pie */
    int fontHeight, labelHeight;/* for geometry computations */

    long int flags;		/* private flags about state */
} TkPie;

#define DEF_PIE_ANGLE			"0"
#define DEF_PIE_BG_COLOR		NORMAL_BG
#define DEF_PIE_BG_MONO			WHITE
#define DEF_PIE_BORDER_WIDTH		"2"
#define DEF_PIE_CURSOR			""
#define DEF_PIE_SLICE_BORDER		"1"
#define DEF_PIE_FONT			DEF_FONT
#define DEF_PIE_SMALL_FONT		DEF_SMALL_FONT
#define DEF_PIE_FG			BLACK
#define DEF_PIE_HIGHLIGHT_WIDTH		"0"
#define DEF_PIE_KEYBOX_SIZE		"100"
/* Key Value Percent Legend */
#define DEF_PIE_LEGEND			"kvpl"
#define DEF_PIE_LABEL			(char *) NULL
#define DEF_PIE_ORIGIN			"0"
#define DEF_PIE_PRECISION		"2"
#define DEF_PIE_PADX			"2"
#define DEF_PIE_PADY			"2"
#define DEF_PIE_RADIUS			"80"
#define DEF_PIE_RELIEF			"raised"
#define DEF_PIE_SHADOW			"0"
#define DEF_PIE_TAKE_FOCUS		(char *) NULL

static Tk_OptionSpec optionSpecs[] = {
  {TK_OPTION_INT, "-angle", "angle", "angle",
   DEF_PIE_ANGLE, -1, Tk_Offset(TkPie, angle), 0, 0, 0},
  {TK_OPTION_BORDER, "-background", "background", "Background",
   DEF_PIE_BG_COLOR, -1, Tk_Offset(TkPie, border),
   0, (ClientData) DEF_PIE_BG_MONO, 0},
  {TK_OPTION_SYNONYM, "-bd", (char *) NULL, (char *) NULL,
   (char *) NULL, 0, -1, 0, (ClientData) "-borderwidth", 0},
  {TK_OPTION_SYNONYM, "-bg", (char *) NULL, (char *) NULL,
   (char *) NULL, 0, -1, 0, (ClientData) "-background", 0},
  {TK_OPTION_PIXELS, "-borderwidth", "borderwidth", "Borderwidth",
   DEF_PIE_BORDER_WIDTH, -1, Tk_Offset(TkPie, borderWidth), 0, 0, 0},
  {TK_OPTION_CURSOR, "-cursor", "cursor", "Cursor",
   DEF_PIE_CURSOR, -1, Tk_Offset(TkPie, cursor), TK_CONFIG_NULL_OK, 0, 0},
  {TK_OPTION_SYNONYM, "-fg", (char *) NULL, (char *) NULL,
   (char *) NULL, 0, -1, 0, (ClientData) "-foreground", 0},
  {TK_OPTION_FONT, "-font", "font", "Font",
   DEF_PIE_FONT, -1, Tk_Offset(TkPie, tkfont), 0, 0, 0},
  {TK_OPTION_COLOR, "-foreground", "foreground", "Foreground",
   DEF_PIE_FG, -1, Tk_Offset(TkPie, textColorPtr), 0, 0, 0},
  {TK_OPTION_COLOR, "-highlightbackground", "highlightBackground",
   "HighlightBackground", HIGHLIGHT_BG, -1,
   Tk_Offset(TkPie, highlightBgColorPtr), 0, 0, 0},
  {TK_OPTION_COLOR, "-highlightcolor", "highlightColor", "HighlightColor",
   HIGHLIGHT, -1, Tk_Offset(TkPie, highlightColorPtr), 0, 0, 0},
  {TK_OPTION_PIXELS, "-highlightthickness", "highlightThickness",
   "HighlightThickness", DEF_PIE_HIGHLIGHT_WIDTH, -1,
   Tk_Offset(TkPie, highlightWidth), 0, 0, 0},
  {TK_OPTION_STRING, "-label", "label", "Label",
   DEF_PIE_LABEL, -1, Tk_Offset(TkPie, label), TK_CONFIG_NULL_OK, 0, 0},
  {TK_OPTION_STRING, "-legend", "legend", "Legend",
   DEF_PIE_LEGEND, -1, Tk_Offset(TkPie, legend), 0, 0, 0},
  {TK_OPTION_INT, "-origin", "origin", "Origin",
   DEF_PIE_ORIGIN, -1, Tk_Offset(TkPie, origin), 0, 0, 0},
  {TK_OPTION_INT, "-precision", "precision", "Precision",
   DEF_PIE_PRECISION, -1, Tk_Offset(TkPie, precision), 0, 0, 0},
  {TK_OPTION_PIXELS, "-padx", "padX", "Pad",
   DEF_PIE_PADX, -1, Tk_Offset(TkPie, padX), 0, 0, 0},
  {TK_OPTION_PIXELS, "-pady", "padY", "Pad",
   DEF_PIE_PADY, -1, Tk_Offset(TkPie, padY), 0, 0, 0},
  {TK_OPTION_PIXELS, "-radius", "radius", "Radius",
   DEF_PIE_RADIUS, -1, Tk_Offset(TkPie, radius), 0, 0, 0},
  {TK_OPTION_RELIEF, "-relief", "relief", "Relief",
   DEF_PIE_RELIEF, -1, Tk_Offset(TkPie, relief), 0, 0, 0},
  {TK_OPTION_PIXELS, "-shadow", "shadow", "shadow",
   DEF_PIE_SHADOW, -1, Tk_Offset(TkPie, shadow), 0, 0, 0},
  {TK_OPTION_BOOLEAN, "-sliceborder", "sliceBorder", "SliceBorder",
   DEF_PIE_SLICE_BORDER, -1, Tk_Offset(TkPie, sliceborder), 0, 0, 0},
  {TK_OPTION_FONT, "-smallfont", "smallFont", "Font",
   DEF_PIE_SMALL_FONT, -1, Tk_Offset(TkPie, smallfont),
   TK_CONFIG_NULL_OK, 0, 0},
  {TK_OPTION_STRING, "-takefocus", "takeFocus", "TakeFocus",
   DEF_PIE_TAKE_FOCUS, -1, Tk_Offset(TkPie, takeFocus),
   TK_CONFIG_NULL_OK, 0, 0},
  {TK_OPTION_END, (char *) NULL, (char *) NULL, (char *) NULL,
   (char *) NULL, 0, -1, 0, 0, 0}
};

#define DEF_PIE_VALUE		"0.0" /* must be 0 */
#define DEF_PIE_EXPLODE		"0"

/*
 * The default specification for configuring elts
 * Done like this to make the command line parsing easy
 */
static Tk_ConfigSpec pieEltConfig[] = {
  {TK_CONFIG_BORDER, "-foreground", "foreground", "Foreground",
   NULL, Tk_Offset(TkPieElt, fg), TK_CONFIG_NULL_OK},
  {TK_CONFIG_SYNONYM, "-fg", "foreground", (char *) NULL,
   (char *) NULL, 0, 0 },
  {TK_CONFIG_STRING, "-label", "label", "Label",
   DEF_PIE_LABEL, Tk_Offset(TkPieElt, label), TK_CONFIG_NULL_OK},
  {TK_CONFIG_DOUBLE, "-value", "value", "Value",
   DEF_PIE_VALUE, Tk_Offset(TkPieElt, value), 0},
  {TK_CONFIG_PIXELS, "-explode", "explode", "Explode",
   DEF_PIE_EXPLODE, Tk_Offset(TkPieElt, explode), 0},
  {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
   (char *) NULL, 0, 0 }
};

static Tk_OptionSpec eltOptionSpecs[] = {
  {TK_OPTION_BORDER, "-foreground", "foreground", "Foreground",
   NULL, -1, Tk_Offset(TkPieElt, fg), TK_CONFIG_NULL_OK, 0, 0},
  {TK_OPTION_SYNONYM, "-fg", (char *) NULL, (char *) NULL,
   (char *) NULL, 0, -1, 0, (ClientData) "-foreground", 0},
  {TK_OPTION_STRING, "-label", "label", "Label",
   DEF_PIE_LABEL, -1, Tk_Offset(TkPieElt, label), TK_CONFIG_NULL_OK, 0, 0},
  {TK_OPTION_DOUBLE, "-value", "value", "Value",
   DEF_PIE_VALUE, -1, Tk_Offset(TkPieElt, value), 0, 0, 0},
  {TK_OPTION_PIXELS, "-explode", "explode", "Explode",
   DEF_PIE_EXPLODE, -1, Tk_Offset(TkPieElt, explode), 0, 0, 0},
  {TK_OPTION_END, (char *) NULL, (char *) NULL, (char *) NULL,
   (char *) NULL, 0, -1, 0, 0, 0}
};

/* Private flag bits */
#define REDRAW_PENDING		(1<<0)
#define AVOID_REDRAW		(1<<1)
#define CLEAR_NEEDED		(1<<2)
#define KEY_CHANGED		(1<<3)
#define GOT_FOCUS		(1<<4)
#define DRAW_HIGHLIGHT		(1<<6)

/*
 * The following tables define the widget commands (and sub-
 * commands) and map the indexes into the string tables into 
 * enumerated types used to dispatch the widget command.
 */

static char *commandNames[] = {
    "cget", "configure", "delete", "explode",
    "itemcget", "itemconfigure", "lower", "names",
    "order", "raise", "set", "swap", "value", (char *) NULL
};

enum command {
    CMD_CGET, CMD_CONFIGURE, CMD_DELETE, CMD_EXPLODE,
    CMD_ITEMCGET, CMD_ITEMCONFIGURE, CMD_LOWER, CMD_NAMES,
    CMD_ORDER, CMD_RAISE, CMD_SET, CMD_SWAP, CMD_VALUE
};

/*
 * Forward declarations for procedures defined later in this file:
 */

static int		PieWidgetObjCmd _ANSI_ARGS_((ClientData clientData,
				Tcl_Interp *interp, int objc,
				Tcl_Obj *CONST objv[]));
static void		DestroyPie _ANSI_ARGS_((ClientData clientData));
static int		ConfigurePie _ANSI_ARGS_((Tcl_Interp *interp,
				TkPie *piePtr, int objc,
				Tcl_Obj *CONST objv[], int flags));
static void		DisplayPie _ANSI_ARGS_((ClientData clientData));
static void		ComputePieGeometry _ANSI_ARGS_((TkPie *piePtr));
static void		PieEventProc _ANSI_ARGS_((ClientData clientData,
			    XEvent *eventPtr));
static void		PieCmdDeletedProc _ANSI_ARGS_((ClientData clientData));
static void		EventuallyRedrawPie _ANSI_ARGS_((TkPie *piePtr,
							 long int flags));
#define RedrawPie(p)	EventuallyRedrawPie((p), (p)->flags)

static void		Pie_FreeElt _ANSI_ARGS_((TkPie *piePtr,
						   TkPieElt *eltPtr));
static TkPieElt *	Pie_FindElt _ANSI_ARGS_((TkPie *piePtr, char *name));
static int		Pie_GetElt _ANSI_ARGS_((TkPie *piePtr, char *name,
						  TkPieElt **eltPtr));
static TkPieElt *	Pie_CreateElt _ANSI_ARGS_((TkPie *piePtr,
						     char *name));
static void		Pie_DeleteElt _ANSI_ARGS_((TkPie *piePtr,
						     TkPieElt *eltPtr));
static int		Pie_SwapElt _ANSI_ARGS_((TkPieElt *eltPtr,
						   TkPieElt *swapElt));
static int		Pie_LowerElt _ANSI_ARGS_((TkPie *piePtr,
						    TkPieElt *eltPtr,
						    TkPieElt *belowThis));
static int		Pie_RaiseElt _ANSI_ARGS_((TkPie *piePtr,
						    TkPieElt *eltPtr,
						    TkPieElt *aboveThis));
static int		ConfigurePieElt _ANSI_ARGS_((Tcl_Interp *interp,
				register TkPie *piePtr,
				register TkPieElt *eltPtr,
				int argc, char **argv, int flags));

/*
 * ---------------------------------------------------------------
 *
 * Tk_PieCmd --
 *
 *	This procedure is invoked to process the "pie" Tk command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *---------------------------------------------------------------
 */
int
Tk_PieObjCmd(clientData, interp, objc, objv)
     ClientData clientData;
     Tcl_Interp *interp;
     int objc;			/* Number of arguments. */
     Tcl_Obj *CONST objv[];	/* Argument objects. */
{
    register TkPie *piePtr;
    static Tk_OptionTable optionTable;
    Tk_Window tkwin;

    if (optionTable == NULL) {
	Tcl_CmdInfo info;
	char *name;

	/*
	 * We haven't created the option tables for this widget class
	 * yet.  Do it now and save the table as the clientData for
	 * the command, so we'll have access to it in future
	 * invocations of the command.
	 */

	optionTable = Tk_CreateOptionTable(interp, optionSpecs);
	name = Tcl_GetString(objv[0]);
	Tcl_GetCommandInfo(interp, name, &info);
	info.objClientData = (ClientData) optionTable;
	Tcl_SetCommandInfo(interp, name, &info);
    }

    if (objc < 2) {
	Tcl_WrongNumArgs(interp, 1, objv, "pathName ?options?");
	return TCL_ERROR;
    }

    tkwin = Tk_CreateWindowFromPath(interp, Tk_MainWindow(interp),
            Tcl_GetString(objv[1]), (char *) NULL);
    if (tkwin == NULL) {
	return TCL_ERROR;
    }

    piePtr = (TkPie *) ckalloc(sizeof(TkPie));

    piePtr->tkwin	= tkwin;
    piePtr->display	= Tk_Display(tkwin);
    piePtr->interp	= interp;
    piePtr->widgetCmd	= Tcl_CreateObjCommand(interp,
				Tk_PathName(piePtr->tkwin), PieWidgetObjCmd,
				(ClientData) piePtr, PieCmdDeletedProc);
    piePtr->optionTable = optionTable;

    piePtr->border	= NULL;
    piePtr->borderWidth	= 2;
    piePtr->cursor	= None;
    piePtr->relief	= TK_RELIEF_RAISED;
    piePtr->highlightBgColorPtr	= NULL;
    piePtr->highlightColorPtr	= NULL;
    piePtr->highlightWidth	= 0;

    piePtr->totalValue	= 0;
    piePtr->angle	= 0;
    piePtr->shadow	= 0;
    piePtr->sliceborder	= 0;
    piePtr->radius	= 0;
    piePtr->radX	= 0;
    piePtr->radY	= 0;
    piePtr->padX	= 0;
    piePtr->padY	= 0;
    piePtr->numSlices	= 0;
    piePtr->maxExplode	= 0;
    piePtr->maxValueLen	= 0;
    piePtr->reqKeyLen	= 0;
    piePtr->reqPercLen	= 0;
    piePtr->reqValueLen	= 0;
    piePtr->reqLabelLen	= 0;
    piePtr->autoColor	= 0;
    piePtr->firstElt	= NULL;

    piePtr->precision	= 0;
    piePtr->origin	= 0;
    piePtr->legend	= NULL;
    piePtr->takeFocus	= NULL;

    piePtr->tkfont	= NULL;
    piePtr->smallfont	= NULL;
    piePtr->textColorPtr	= NULL;
    piePtr->textGC	= None;
    piePtr->sfGC	= None;
    piePtr->label	= NULL;
    piePtr->fontHeight	= 0;
    piePtr->labelHeight	= 0;

    piePtr->flags		= 0;

    Tk_SetClass(piePtr->tkwin, "Pie");
    Tk_CreateEventHandler(piePtr->tkwin, 
			  ExposureMask|StructureNotifyMask|FocusChangeMask,
			  PieEventProc, (ClientData)piePtr);

    if ((Tk_InitOptions(interp, (char *) piePtr, optionTable, tkwin)
	    != TCL_OK) ||
	    (ConfigurePie(interp, piePtr, objc-2, objv+2, 0) != TCL_OK)) {
	Tk_DestroyWindow(piePtr->tkwin);
	return TCL_ERROR;
    }

    Tcl_SetStringObj(Tcl_GetObjResult(interp), Tk_PathName(piePtr->tkwin), -1);
    return TCL_OK;
}

static int
PieWidgetObjCmd(clientData, interp, objc, objv)
     ClientData clientData;
     Tcl_Interp *interp;
     int objc;			/* Number of arguments. */
     Tcl_Obj *CONST objv[];	/* Argument objects. */
{
    register TkPie *piePtr = (TkPie *) clientData;
    int cmdIndex, result = TCL_OK;
    Tcl_Obj *objPtr, *resultPtr;
    TkPieElt *eltPtr;

    if (objc < 2) {
	Tcl_WrongNumArgs(interp, 1, objv, "option ?arg arg ...?");
	return TCL_ERROR;
    }

    /* parse the first parameter */
    result = Tcl_GetIndexFromObj(interp, objv[1], commandNames,
				 "option", 0, &cmdIndex);
    if (result != TCL_OK) {
	return result;
    }
    resultPtr = Tcl_GetObjResult(interp);
    Tcl_Preserve((ClientData) piePtr);

    switch (cmdIndex) {
    case CMD_CGET:
	if (objc != 3) {
	    Tcl_WrongNumArgs(interp, 1, objv, "cget option");
	    result = TCL_ERROR;
	} else {
	    objPtr = Tk_GetOptionValue(interp, (char *) piePtr,
		    piePtr->optionTable, objv[2], piePtr->tkwin);
	    if (objPtr == NULL) {
		result = TCL_ERROR;
	    } else {
		Tcl_SetObjResult(interp, objPtr);
	    }
	}
	break;

    case CMD_CONFIGURE:
	if (objc <= 3) {
	    objPtr = Tk_GetOptionInfo(interp, (char *) piePtr,
				      piePtr->optionTable,
				      (objc == 3) ? objv[2] : (Tcl_Obj *) NULL,
				      piePtr->tkwin);
	    if (objPtr == NULL) {
		result = TCL_ERROR;
	    } else {
		Tcl_SetObjResult(interp, objPtr);
	    }
	} else {
	    result = ConfigurePie(interp, piePtr, objc-2, objv+2, 0);
	}
	break;

    case CMD_DELETE:
	if (objc < 3) {
	    Tcl_WrongNumArgs(interp, 1, objv,
			     "delete slice ?slice? ...");
	    result = TCL_ERROR;
	} else {
	    int i, redraw = 0;

	    /* go through all the slices once */
	    for (eltPtr = piePtr->firstElt; eltPtr != NULL;
		 eltPtr = eltPtr->next) {
		/* check each argument for a match against this slice's name */
		for (i = 2; i < objc; i++) {
		    if (Tcl_StringMatch(eltPtr->name,
					Tcl_GetString(objv[i]))) {
			Pie_DeleteElt(piePtr, eltPtr);
			redraw = 1;
			break;
		    }
		}
	    }
	    if (redraw) {
		ComputePieGeometry(piePtr);
		RedrawPie(piePtr);
	    }
	}
	break;

    case CMD_EXPLODE:
	if ((objc < 3) || (objc != 3 && (objc&1))) {
	    /* only 3 or an even number of args allowed */
	    Tcl_WrongNumArgs(interp, 1, objv,
			     "explode slice ?value? ?slice value ...?");
	    result = TCL_ERROR;
	    break;
	}
	if (objc == 3) {
	    if (Pie_GetElt(piePtr, Tcl_GetString(objv[2]),
			   &eltPtr) != TCL_OK) {
		result = TCL_ERROR;
		break;
	    }
	    Tcl_SetIntObj(resultPtr, eltPtr->explode);
	} else {
	    int i, value;

	    for (i = 2; i < objc; i += 2) {
		if ((Pie_GetElt(piePtr, Tcl_GetString(objv[i]),
				&eltPtr) != TCL_OK) ||
		    (Tcl_GetIntFromObj(interp, objv[i+1], &value)
		     != TCL_OK)) {
		    result = TCL_ERROR;
		    break;
		}
		eltPtr->explode = value;
	    }
	    ComputePieGeometry(piePtr);
	    RedrawPie(piePtr);
	}
	break;

    case CMD_ITEMCGET: {
	if (objc != 4) {
	    Tcl_WrongNumArgs(interp, 1, objv, "itemcget slice option");
	    result = TCL_ERROR;
	} else if (Pie_GetElt(piePtr, Tcl_GetString(objv[2]), &eltPtr)
		   != TCL_OK) {
	    result = TCL_ERROR;
	} else {
	    result = Tk_ConfigureValue(interp, piePtr->tkwin, pieEltConfig,
				(char *) eltPtr, Tcl_GetString(objv[3]), 0);
	}
	break;
    }

    case CMD_ITEMCONFIGURE: {
	int new = 0;

	if (objc < 3) {
	    Tcl_WrongNumArgs(interp, 1, objv, "itemconfigure slice ?options?");
	    result = TCL_ERROR;
	    break;
	}
	if ((eltPtr = Pie_FindElt(piePtr, Tcl_GetString(objv[2]))) == NULL) {
	    eltPtr = Pie_CreateElt(piePtr, Tcl_GetString(objv[2]));
	    new = 1;
	}
	if (objc == 3) {
	    result = Tk_ConfigureInfo(interp, piePtr->tkwin, pieEltConfig,
				      (char *) eltPtr, (char *) NULL, 0);
	} else if (objc == 4) {
	    result = Tk_ConfigureInfo(interp, piePtr->tkwin, pieEltConfig,
				      (char *) eltPtr, Tcl_GetString(objv[3]),
				      0);
	} else {
	    int i;
	    char **argv;

	    argv = (char **) ckalloc((objc - 3 + 1) * sizeof(char *));
	    for (i = 0; i < objc-3; i++) {
		argv[i] = Tcl_GetString(objv[i+3]);
	    }
	    argv[i] = NULL;
	    result = ConfigurePieElt(interp, piePtr, eltPtr, objc-3, argv,
				     new ? 0 : TK_CONFIG_ARGV_ONLY);
	    ckfree((char *) argv);
	}
	break;
    }

    case CMD_LOWER: {
	TkPieElt *belowThis = NULL;

	if ((objc != 4) && (objc != 3)) {
	    Tcl_WrongNumArgs(interp, 1, objv, "lower slice ?belowThis?");
	    result = TCL_ERROR;
	} else if ((Pie_GetElt(piePtr, Tcl_GetString(objv[2]),
			       &eltPtr) != TCL_OK) ||
		   ((objc == 4) && Pie_GetElt(piePtr, Tcl_GetString(objv[3]),
					      &belowThis) != TCL_OK)) {
	    result = TCL_ERROR;
	} else if (Pie_LowerElt(piePtr, eltPtr, belowThis)) {
	    RedrawPie(piePtr);
	}
	break;
    }

    case CMD_NAMES:
	if ((objc != 2) && (objc != 3)) {
	    Tcl_WrongNumArgs(interp, 1, objv, "names ?pattern?");
	    result = TCL_ERROR;
	} else {
	    char *pattern = NULL;

	    if (objc == 3) {
		pattern = Tcl_GetString(objv[2]);
	    }
	    for (eltPtr = piePtr->firstElt; eltPtr != NULL;
		 eltPtr = eltPtr->next) {
		if ((objc == 2) || Tcl_StringMatch(eltPtr->name, pattern)) {
		    objPtr = Tcl_NewStringObj(eltPtr->name, -1);
		    if (Tcl_ListObjAppendElement(interp, resultPtr, objPtr)
			!= TCL_OK) {
			Tcl_DecrRefCount(objPtr); /* free unneeded name obj */
			result = TCL_ERROR;
			break;
		    }
		}
	    }
	}
	break;

    case CMD_ORDER:
	if (objc < 3) {
	    Tcl_WrongNumArgs(interp, 1, objv,
			     "order slice ?slice ...?");
	    result = TCL_ERROR;
	} else {
	    int i, redraw = 0;

	    /*
	     * start at the last arg and come back,
	     * so the order of raises is OK
	     */
	    for (i = objc-1; i >= 2; i--) {
		if (Pie_GetElt(piePtr, Tcl_GetString(objv[i]),
			       &eltPtr) != TCL_OK) {
		    return TCL_ERROR;
		}
		redraw += Pie_RaiseElt(piePtr, eltPtr, (TkPieElt *)NULL);
	    }
	    if (redraw) {
		RedrawPie(piePtr);
	    }
	}
	break;

    case CMD_RAISE: {
	TkPieElt *aboveThis = NULL;

	if ((objc != 4) && (objc != 3)) {
	    Tcl_WrongNumArgs(interp, 1, objv, "raise slice ?aboveThis?");
	    result = TCL_ERROR;
	} else if ((Pie_GetElt(piePtr, Tcl_GetString(objv[2]),
			       &eltPtr) != TCL_OK) ||
		   ((objc == 4) && Pie_GetElt(piePtr, Tcl_GetString(objv[3]),
					      &aboveThis) != TCL_OK)) {
	    result = TCL_ERROR;
	} else if (Pie_RaiseElt(piePtr, eltPtr, aboveThis)) {
	    RedrawPie(piePtr);
	}
	break;
    }

    case CMD_SET:
	if ((objc < 3) || (objc != 3 && (objc&1))) {
	    /* only 3 or an even number of args allowed */
	    Tcl_WrongNumArgs(interp, 1, objv,
			     "set slice ?value? ?slice value ...?");
	    result = TCL_ERROR;
	    break;
	}
	if (objc == 3) {
	    if (Pie_GetElt(piePtr, Tcl_GetString(objv[2]),
			   &eltPtr) != TCL_OK) {
		result = TCL_ERROR;
		break;
	    }
	    Tcl_SetDoubleObj(resultPtr, piePtr->totalValue);
	} else {
	    int i;
	    double value;

	    for (i = 2; i < objc; i += 2) {
		/* Get the double first, in case it throws an error */
		if (Tcl_GetDoubleFromObj(interp, objv[i+1], &value)
		    != TCL_OK) {
		    result = TCL_ERROR;
		    break;
		}
		eltPtr = Pie_FindElt(piePtr, Tcl_GetString(objv[i]));
		if (eltPtr == NULL) {
		    /* set will actually create slices */
		    eltPtr = Pie_CreateElt(piePtr, Tcl_GetString(objv[i]));
		    piePtr->flags |= AVOID_REDRAW;
		    ConfigurePieElt(interp, piePtr, eltPtr, 0, NULL, 0);
		    piePtr->flags &= ~AVOID_REDRAW;
		}
		piePtr->totalValue += value - eltPtr->value;
		eltPtr->value = value;
	    }
	    ComputePieGeometry(piePtr);
	    RedrawPie(piePtr);
	}
	break;

    case CMD_SWAP: {
	TkPieElt *swapElt;

	if (objc != 4) {
	    Tcl_WrongNumArgs(interp, 1, objv, "swap slice1 slice2");
	    result = TCL_ERROR;
	} else if ((Pie_GetElt(piePtr, Tcl_GetString(objv[2]),
			       &eltPtr) != TCL_OK) ||
		   (Pie_GetElt(piePtr, Tcl_GetString(objv[3]),
			       &swapElt) != TCL_OK)) {
	    result = TCL_ERROR;
	} else if (Pie_SwapElt(eltPtr, swapElt)) {
	    RedrawPie(piePtr);
	}
	break;
    }

    case CMD_VALUE:
	if (objc != 2) {
	    Tcl_WrongNumArgs(interp, 1, objv, "value");
	    result = TCL_ERROR;
	} else {
	    Tcl_SetDoubleObj(resultPtr, piePtr->totalValue);
	}
	break;
    }

    Tcl_Release((ClientData) piePtr);
    return result;
}

static void
DestroyPie(clientData)
     ClientData clientData;
{
    register TkPie *piePtr = (TkPie *) clientData;

    if (piePtr->flags & REDRAW_PENDING) {
        Tcl_CancelIdleCall(DisplayPie, (ClientData) piePtr);
    }
    piePtr->flags = 0;

    Tcl_DeleteCommandFromToken(piePtr->interp, piePtr->widgetCmd);

    if (piePtr->textGC != None) {
	Tk_FreeGC(piePtr->display, piePtr->textGC);
    }
    if (piePtr->sfGC != None) {
	Tk_FreeGC(piePtr->display, piePtr->sfGC);
    }
  
    if (piePtr->firstElt) {
	TkPieElt *nextElt, *eltPtr = piePtr->firstElt;

	while (eltPtr != NULL) {
	    nextElt = eltPtr->next;
	    Pie_FreeElt(piePtr, eltPtr);
	    eltPtr = nextElt;
	}
    }
  
    /* free the configuration options in the widget */
    Tk_FreeConfigOptions((char *) piePtr, piePtr->optionTable, piePtr->tkwin);
    piePtr->tkwin = NULL;
    Tcl_EventuallyFree((ClientData) piePtr, TCL_DYNAMIC);
}

static int
ConfigurePie(interp, piePtr, objc, objv, flags)
     Tcl_Interp *interp;
     register TkPie *piePtr;
     int objc;
     Tcl_Obj *CONST objv[];
     int flags;
{
    Tk_SavedOptions savedOptions;
    Tcl_Obj *errorResult = NULL;
    int error;

    XGCValues gcValues;
    Tk_FontMetrics fm;
    char *oldLegend = NULL, *legend, *tmp = NULL;
    int cWidth, len, legWidth, kSeen, lSeen, pSeen, vSeen;

    if (piePtr->legend) {
	oldLegend = (char *)ckalloc(strlen(piePtr->legend)+1);
	strcpy(oldLegend, piePtr->legend);
    }

    for (error = 0; error <= 1; error++) {
	if (!error) {
	    /*
	     * First pass: set options to new values.
	     */

	    if (Tk_SetOptions(interp, (char *) piePtr,
		    piePtr->optionTable, objc, objv,
		    piePtr->tkwin, &savedOptions, (int *) NULL) != TCL_OK) {
		continue;
	    }
	} else {
	    /*
	     * Second pass: restore options to old values.
	     */

	    errorResult = Tcl_GetObjResult(interp);
	    Tcl_IncrRefCount(errorResult);
	    Tk_RestoreSavedOptions(&savedOptions);
	}

	/*
	 * A few other options also need special processing, such as parsing
	 * the geometry and setting the background from a 3-D border.
	 */

	Tk_SetBackgroundFromBorder(piePtr->tkwin, piePtr->border);

	/* manage constraints */
	piePtr->borderWidth	= MAX(piePtr->borderWidth, 0);
	piePtr->highlightWidth	= MAX(piePtr->highlightWidth, 0);
	piePtr->precision	= MAX(piePtr->precision, 0);
	piePtr->padX		= MAX(piePtr->padX, 0);
	piePtr->padY		= MAX(piePtr->padY, 0);
	piePtr->radius		= MAX(piePtr->radius, 1);
	piePtr->shadow		= MAX(piePtr->shadow, 0);
	piePtr->angle		= MAX(piePtr->angle, 0);
	piePtr->angle		= MIN(piePtr->angle, 90);
	piePtr->origin		= piePtr->origin % 360;

	/* apply tilt to pie if necessary */
	if (piePtr->angle) {
	    piePtr->radX = piePtr->radius;
	    /* invert angle from 90, as the adjustment is more natural */
	    piePtr->radY = piePtr->radius * (90-piePtr->angle)/90;
	} else {
	    piePtr->radX = piePtr->radY = piePtr->radius;
	}

	gcValues.font = Tk_FontId(piePtr->tkfont);
	gcValues.foreground = piePtr->textColorPtr->pixel;
	gcValues.graphics_exposures = False;
	if (piePtr->textGC != None) {
	    Tk_FreeGC(piePtr->display, piePtr->textGC);
	}
	piePtr->textGC = Tk_GetGC(piePtr->tkwin,
		GCForeground|GCFont|GCGraphicsExposures, &gcValues);

	if (piePtr->sfGC != None) {
	    Tk_FreeGC(piePtr->display, piePtr->sfGC);
	    piePtr->sfGC = None;
	}
	if (piePtr->smallfont != NULL) {
	    gcValues.font = Tk_FontId(piePtr->smallfont);
	    gcValues.foreground = piePtr->textColorPtr->pixel;
	    piePtr->sfGC = Tk_GetGC(piePtr->tkwin, GCForeground|GCFont,
				    &gcValues);
	}

	Tk_GetFontMetrics(piePtr->tkfont, &fm);
	piePtr->fontHeight = fm.linespace;
	piePtr->labelHeight = (piePtr->label?(piePtr->fontHeight+2*piePtr->padY):0);

	cWidth = Tk_TextWidth(piePtr->tkfont, "0", 1);
	legWidth = 0;
	piePtr->reqPercLen	= 0;
	piePtr->reqLabelLen	= 0;
	piePtr->reqValueLen	= 0;
	piePtr->reqKeyLen	= 0;

	kSeen = lSeen = pSeen = vSeen = 0;
	legend = piePtr->legend;
	while (*legend) {
	    len = (int)strtol(legend, &tmp, 10);
	    if (legend == tmp) {
		/* if no number is specified,
		 * -1 indicates max desired length
		 */
		len = -1;
	    }
	    while (isspace(*tmp)) { tmp++; }
	    switch (*tmp) {
	    case 'k': case 'K':
		if (kSeen) goto badchar;
		piePtr->reqKeyLen = len; kSeen++; break;
	    case 'p': case 'P':
		if (pSeen) goto badchar;
		piePtr->reqPercLen = len; pSeen++; break;
	    case 'l': case 'L':
		if (lSeen) goto badchar;
		piePtr->reqLabelLen = len; lSeen++; break;
	    case 'v': case 'V':
		if (vSeen) goto badchar;
		piePtr->reqValueLen = len; vSeen++; break;
	    default:
	    badchar:
		Tcl_AppendResult(interp, "legend option \"", piePtr->legend,
				 "\" must only consist of whitespace ",
				 "and one instance each of ?number?[kplv], ",
				 "no repeating of a character", (char *)NULL);
		ckfree(piePtr->legend);
		piePtr->legend = oldLegend;
		return TCL_ERROR;
	    }
	    if (*tmp) {
		legend = tmp+1;
		legWidth += piePtr->padX;
	    }
	}
	ckfree(oldLegend);

	break;
    }
    if (!error) {
	Tk_FreeSavedOptions(&savedOptions);
    }

    ComputePieGeometry(piePtr);
    RedrawPie(piePtr);
    if (error) {
	Tcl_SetObjResult(interp, errorResult);
	Tcl_DecrRefCount(errorResult);
	return TCL_ERROR;
    } else {
	return TCL_OK;
    }
}

static void
DisplayPie(clientData)
     ClientData clientData;
{
    register TkPie *piePtr = (TkPie *) clientData;
    register Tk_Window tkwin = piePtr->tkwin;
    Display *display = piePtr->display;
    Pixmap pixmap;
    double extent, perpt, start, middle;
    char *options, str[64], *dispStr;
    TkPieElt *eltPtr;
    Tk_FontMetrics fm;
    int len, x, y, cWidth, valWidth, pieWidth, pX, pY, xoff, yoff;
    int bd = piePtr->borderWidth;
    int fontHeight = piePtr->fontHeight;

    piePtr->flags &= ~REDRAW_PENDING;
    if ((piePtr->tkwin == NULL) || !Tk_IsMapped(tkwin)) {
	return;
    }

    /* Pie uses a pixmap for drawing in a double-buffered manner */
    pixmap = Tk_GetPixmap(display, Tk_WindowId(tkwin),
			  Tk_Width(tkwin), Tk_Height(tkwin), Tk_Depth(tkwin));
    /* Clears out the pixmap structure */
    Tk_Fill3DRectangle(tkwin, pixmap, piePtr->border, 0, 0,
		       Tk_Width(tkwin), Tk_Height(tkwin), 0, TK_RELIEF_FLAT);

    piePtr->flags = 0;

    if (piePtr->label != NULL) {
	len = strlen(piePtr->label);
	cWidth = Tk_TextWidth(piePtr->tkfont, piePtr->label, len);

	if (cWidth < Tk_Width(tkwin)-2*bd) {
	    x = (Tk_Width(tkwin)-2*bd - cWidth) / 2;
	} else {
	    /* text wider than window - it will be clipped */
	    x = bd + piePtr->padX;
	}
	Tk_DrawChars(display, pixmap, piePtr->textGC, piePtr->tkfont,
		     piePtr->label, len, x, fontHeight+piePtr->padY);
    }

    /* empty pie case */
    if ((piePtr->numSlices == 0) || (piePtr->totalValue == 0)) {
	XFillArc(display, pixmap, piePtr->textGC,
		 bd + piePtr->maxExplode,
		 bd + piePtr->maxExplode + piePtr->labelHeight,
		 (unsigned) piePtr->radX*2,	/* width */
		 (unsigned) piePtr->radY*2,	/* height */
		 0,			/* angle1 */
		 M_DEGREE);		/* angle2 */
	/* no shadow */
	goto finish;
    }

    /* determine offsets due to little values */
    if (piePtr->smallfont != NULL) {
	Tk_GetFontMetrics(piePtr->smallfont, &fm);
	cWidth = Tk_TextWidth(piePtr->smallfont, "0", 1);
	valWidth = piePtr->maxValueLen*cWidth;
    } else {
	fm.ascent = 0;
	valWidth = 0;
    }

    perpt = M_DEGREE / piePtr->totalValue;
    /* corner of arc rectangles for pie */
    pX = bd + piePtr->maxExplode + valWidth;
    pY = bd + piePtr->maxExplode + piePtr->labelHeight + fm.ascent;

    /* draw the shadow first */
    if (piePtr->shadow && piePtr->angle) {
	/* adjust center of pie */
	pY += piePtr->shadow;

	start = (M_DEGREE/4.0)-(piePtr->origin*64.0);

	for (eltPtr = piePtr->firstElt; eltPtr != NULL;
	     eltPtr = eltPtr->next) {
	    start += ((start<0.0)?M_DEGREE:((start>M_DEGREE)?-M_DEGREE:0));

	    /* Ignore empty slices */
	    if (eltPtr->value == 0.0) {
		continue;
	    }

	    extent = -eltPtr->value * perpt;
	    middle = deg64_2_rads(start+extent/2.0);
	    if (eltPtr->explode) {
		/* this gets the offset for the exploded piece */
		xoff = (int) +(eltPtr->explode*cos(middle));
		yoff = (int) -(eltPtr->explode*sin(middle));
	    } else {
		xoff = yoff = 0;
	    }
	    /* fill the arc with the desired color */
	    XFillArc(display, pixmap,
		    Tk_3DBorderGC(tkwin, eltPtr->fg, TK_3D_DARK_GC),
		    pX + xoff, pY + yoff,
		    (unsigned) piePtr->radX*2, (unsigned) piePtr->radY*2,
		    (int) start, (int) extent);
	    /* shadows don't get borders */

	    start += extent;
	}
	pY -= piePtr->shadow;
    }

    start = (M_DEGREE/4.0)-(piePtr->origin*64.0);

    cWidth = Tk_TextWidth(piePtr->tkfont, "0", 1);
    dispStr = (char *)ckalloc((MAX(MAX(piePtr->useValueLen,piePtr->usePercLen),
				   piePtr->useLabelLen)+1) * sizeof(char));
    pieWidth = 2*(pX+piePtr->radX+piePtr->padX);
    y = piePtr->labelHeight + piePtr->padY;

    for (eltPtr = piePtr->firstElt; eltPtr != NULL; eltPtr = eltPtr->next) {
	start += ((start<0.0)?M_DEGREE:((start>M_DEGREE)?-M_DEGREE:0));

	/* Ignore empty slices */
	if (eltPtr->value == 0.0) {
	    continue;
	}

	extent = -eltPtr->value * perpt;
	middle = deg64_2_rads(start+extent/2.0);
	if (eltPtr->explode) {
	    /* this gets the offset for the exploded piece */
	    xoff = (int) +(eltPtr->explode*cos(middle));
	    yoff = (int) -(eltPtr->explode*sin(middle));
	} else {
	    xoff = yoff = 0;
	}
	/* fill the arc with the desired color */
	XFillArc(display, pixmap,
		Tk_3DBorderGC(tkwin, eltPtr->fg, TK_3D_FLAT_GC),
		pX + xoff, pY + yoff,
		(unsigned) piePtr->radX*2, (unsigned) piePtr->radY*2,
		(int) start, (int) extent);
	if (piePtr->sliceborder) {
	    /* draw a line around the arc having the text color */
	    XDrawArc(display, pixmap, piePtr->textGC,
		    pX + xoff, pY + yoff,
		    (unsigned) piePtr->radX*2, (unsigned) piePtr->radY*2,
		    (int) start, (int) extent);
	    XDrawLine(display, pixmap, piePtr->textGC,
		    pX+xoff + piePtr->radX, pY+yoff + piePtr->radY,
		    pX+xoff +(int)(piePtr->radX*(cos(deg64_2_rads(start))+1)),
		    pY+yoff -(int)(piePtr->radY*(sin(deg64_2_rads(start))-1)));
	    /* Close of the second side to be safe */
	    XDrawLine(display, pixmap, piePtr->textGC,
		    pX+xoff + piePtr->radX, pY+yoff + piePtr->radY,
		    pX+xoff +(int)(piePtr->radX*(cos(deg64_2_rads(start+extent))+1)),
		    pY+yoff -(int)(piePtr->radY*(sin(deg64_2_rads(start+extent))-1)));
	}
	/* draw the value at the pie */
	/* FIX: must account for value lengths */
	if (piePtr->smallfont != NULL) {
	    int cx = pX+xoff, cy = pY+yoff;
	    sprintf(str, "%.*f", piePtr->precision, eltPtr->value);
	    sprintf(dispStr, "%.*s", piePtr->useValueLen, str);
	    len = strlen(dispStr);
	    cx += +(int)((piePtr->radX+piePtr->padX) * (cos(middle)+1));
	    cy += -(int)((piePtr->radY+piePtr->padY) * (sin(middle)-1));
	    if (cx < pX+xoff+piePtr->radX) {
		/* between 6 && 12 o'clock */
		valWidth = Tk_TextWidth(piePtr->smallfont, dispStr, len);
		cx -= valWidth;
	    }
	    if (cy > pY+yoff+piePtr->radY) {
		/* between 3 && 9 o'clock */
		cy += fm.ascent + (piePtr->angle?piePtr->shadow:0);
	    }
	    Tk_DrawChars(display, pixmap, piePtr->sfGC, piePtr->smallfont,
			 dispStr, len, cx, cy);
	}

	start += extent;

	/*
	 * Draw this slice's part of the legend
	 */

	x = pieWidth;
	options = piePtr->legend;
	if (piePtr->useKeyLen == -1) {
	    XFillRectangle(display, pixmap,
		    Tk_3DBorderGC(tkwin, eltPtr->fg, TK_3D_FLAT_GC), x, y,
		    (unsigned) Tk_Width(tkwin)-bd-x-piePtr->padX,
		    (unsigned) fontHeight);
	}
	while (*options) {
	    switch (*options) {
	    case 'k': case 'K' :	/* key */
		if (piePtr->useKeyLen > 0) {
		    XFillRectangle(display, pixmap,
				Tk_3DBorderGC(tkwin, eltPtr->fg,
					TK_3D_FLAT_GC),
			    x, y, (unsigned) fontHeight,
			    (unsigned) fontHeight);
		    x += fontHeight + piePtr->padX;
		}
		break;
	    case 'p': case 'P':	/* percent */
		sprintf(str, "%.*f%%", piePtr->precision,
			100.0 * eltPtr->value / piePtr->totalValue);
		    sprintf(dispStr, "%*.*s", piePtr->usePercLen,
			    piePtr->usePercLen, str);

		    Tk_DrawChars(display, pixmap, piePtr->textGC,
			    piePtr->tkfont, dispStr, (int) strlen(dispStr),
			x, y+fontHeight-piePtr->padY);
		x += piePtr->usePercLen*cWidth + piePtr->padX;
		break;
	    case 'v': case 'V':	/* value */
		sprintf(str, "%.*f", piePtr->precision, eltPtr->value);
		    sprintf(dispStr, "%*.*s", piePtr->useValueLen,
			    piePtr->useValueLen, str);
	
		    Tk_DrawChars(display, pixmap, piePtr->textGC,
			    piePtr->tkfont, dispStr, (int) strlen(dispStr),
			x, y+fontHeight-piePtr->padY);
		x += piePtr->useValueLen*cWidth + piePtr->padX;
		break;
	    case 'l': case 'L' :	/* label */
		if (eltPtr->label) {
			Tk_DrawChars(display, pixmap, piePtr->textGC,
				piePtr->tkfont, eltPtr->label,
				(int) strlen(eltPtr->label),
			    x, y+fontHeight-piePtr->padY);
		}
		x += piePtr->useLabelLen*cWidth + piePtr->padX;
		break;
	    default:
		/* ignore spurious values */
		break;
	    }
	    options++;
	}
	y += fontHeight + piePtr->padY;
    }
    ckfree((char *)dispStr);

    finish:
    /*
     * Draw the border and focus highlight last, so they will overwrite
     * any text that extends past the viewable part of the window.
     */

    if (piePtr->relief != TK_RELIEF_FLAT) {
	Tk_Draw3DRectangle(tkwin, pixmap, piePtr->border,
			   piePtr->highlightWidth, piePtr->highlightWidth,
			   Tk_Width(tkwin) - 2*piePtr->highlightWidth,
			   Tk_Height(tkwin) - 2*piePtr->highlightWidth,
			   piePtr->borderWidth, piePtr->relief);
    }
    if (piePtr->highlightWidth != 0) {
	Tk_DrawFocusHighlight(tkwin,
			      Tk_GCForColor((piePtr->flags & GOT_FOCUS) ?
					    piePtr->highlightColorPtr : 
					    piePtr->highlightBgColorPtr, pixmap),
			      piePtr->highlightWidth, pixmap);
    }
    piePtr->flags &= ~DRAW_HIGHLIGHT;

    /*
     * Everything's been redisplayed;  now copy the pixmap onto the screen
     * and free up the pixmap.
     */
    XCopyArea(display, pixmap, Tk_WindowId(tkwin), piePtr->textGC,
	      0, 0, (unsigned) Tk_Width(tkwin), (unsigned) Tk_Height(tkwin),
	      0, 0);
    Tk_FreePixmap(display, pixmap);
}

static void
ComputePieGeometry(piePtr)
     TkPie *piePtr;  
{
    int width, height, legHeight, legWidth, cWidth, len;
    int maxLabelLen, maxPercLen, maxValueLen, elems = 0;
    TkPieElt *eltPtr;
    char str[64];

    /*
     * These checks are done for exactness
     */
    piePtr->maxExplode = maxLabelLen = maxPercLen = maxValueLen = 0;
    for (eltPtr = piePtr->firstElt; eltPtr != NULL;
	 eltPtr = eltPtr->next) {
	/* Check for longest label */
	if ((len = strlen(eltPtr->label)) > maxLabelLen)
	    maxLabelLen = len;
	/* Check for longest value string */
	sprintf(str, "%.*f", piePtr->precision, eltPtr->value);
	if ((len = strlen(str)) > maxValueLen) {
	    maxValueLen = len;
	}
	/* Check for longest percent string */
	sprintf(str, "%.*f%%", piePtr->precision,
		100.0 * eltPtr->value / piePtr->totalValue);
	if ((len = strlen(str)) > maxPercLen) {
	    maxPercLen = len;
	}
	/* Check for farthest explode */
	if (eltPtr->explode > piePtr->maxExplode) {
	    piePtr->maxExplode = eltPtr->explode;
    }
	elems++;
    }
    piePtr->maxValueLen = maxValueLen;

    /*
     * Dimensions for the Pie
     */
    height = 2 * (piePtr->radY + piePtr->maxExplode)
	+ (piePtr->angle ? piePtr->shadow : 0);
    width = 2 * (piePtr->radX + piePtr->maxExplode
	    + piePtr->borderWidth + piePtr->padX);
    if (piePtr->smallfont != NULL) {
	Tk_FontMetrics fm;
	Tk_GetFontMetrics(piePtr->smallfont, &fm);
	height += 2*fm.ascent;
	cWidth = Tk_TextWidth(piePtr->smallfont, "0", 1);
	width += 2*piePtr->maxValueLen*cWidth;
    }

    /* this is the height the legend needs */
    legHeight = piePtr->numSlices * (piePtr->fontHeight + piePtr->padY);
    /* width that the legend needs */
    cWidth = Tk_TextWidth(piePtr->tkfont, "0", 1);
    legWidth = 0;

    piePtr->useKeyLen = piePtr->reqKeyLen;
    piePtr->usePercLen = ((piePtr->reqPercLen<0) ?
			  maxPercLen : piePtr->reqPercLen);
    piePtr->useLabelLen = ((piePtr->reqLabelLen<0) ?
			   maxLabelLen : piePtr->reqLabelLen);
    piePtr->useValueLen = ((piePtr->reqValueLen<0) ?
			   maxValueLen : piePtr->reqValueLen);

    legWidth += ((piePtr->reqKeyLen>0) ? piePtr->fontHeight+piePtr->padX : 0)
	+ (piePtr->usePercLen+piePtr->useLabelLen+piePtr->useValueLen)*cWidth;
    if (piePtr->usePercLen) legWidth += piePtr->padX;
    if (piePtr->useValueLen) legWidth += piePtr->padX;
    if (piePtr->useLabelLen) legWidth += piePtr->padX;

    /* establish dimensions for whole widget */
    height += piePtr->labelHeight + 2 * (piePtr->borderWidth + piePtr->padY);
    if (legWidth) {
    width += legWidth;
	if (legHeight > height) {
	    height += legHeight - height;
	}
    }

    Tk_GeometryRequest(piePtr->tkwin, width, height);
    Tk_SetInternalBorder(piePtr->tkwin, piePtr->borderWidth);
}

static void
PieEventProc(clientData,eventPtr)
     ClientData clientData;
     XEvent *eventPtr;
{
    TkPie *piePtr = (TkPie *) clientData;
  
    switch (eventPtr->type) {
    case Expose:
	if (eventPtr->xexpose.count == 0)
	    RedrawPie(piePtr);
	break;
    case DestroyNotify:
	DestroyPie(clientData);
	break;
    case ConfigureNotify:
	ComputePieGeometry(piePtr);
	RedrawPie(piePtr);
	break;
    case FocusIn:
	if (eventPtr->xfocus.detail != NotifyInferior) {
	    piePtr->flags |= GOT_FOCUS;
	    if (piePtr->highlightWidth > 0) {
		EventuallyRedrawPie(piePtr, DRAW_HIGHLIGHT);
	    }
	}
	break;
    case FocusOut:
	if (eventPtr->xfocus.detail != NotifyInferior) {
	    piePtr->flags &= ~GOT_FOCUS;
	    if (piePtr->highlightWidth > 0) {
		EventuallyRedrawPie(piePtr, DRAW_HIGHLIGHT);
	    }
	}
	break;
    }
}  

/*
 *----------------------------------------------------------------------
 *
 * PieCmdDeletedProc --
 *
 *	This procedure is invoked when a widget command is deleted.  If
 *	the widget isn't already in the process of being destroyed,
 *	this command destroys it.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The widget is destroyed.
 *
 *----------------------------------------------------------------------
 */
static void
PieCmdDeletedProc(clientData)
     ClientData clientData;	/* Pointer to widget record for widget. */
{
    TkPie *piePtr = (TkPie *) clientData;
    Tk_Window tkwin = piePtr->tkwin;

    /*
     * This procedure could be invoked either because the window was
     * destroyed and the command was then deleted (in which case tkwin
     * is NULL) or because the command was deleted, and then this procedure
     * destroys the widget.
     */

    if (tkwin != NULL) {
	piePtr->tkwin = NULL;
	Tk_DestroyWindow(tkwin);
    }
}

static void
EventuallyRedrawPie(piePtr, flags)
     register TkPie *piePtr;	/* Information about widget. */
     long int flags;		/* Which parts of the pie are to be drawn */
{
    if ((piePtr->tkwin == NULL) || !Tk_IsMapped(piePtr->tkwin) ||
	(piePtr->flags & REDRAW_PENDING)) {
	piePtr->flags |= flags;
	return;
    }

    piePtr->flags = flags | REDRAW_PENDING;
    Tcl_DoWhenIdle(DisplayPie, (ClientData)piePtr);
}

static void
Pie_FreeElt(TkPie *piePtr, TkPieElt *eltPtr)
{
    if (eltPtr->name) ckfree((char *)eltPtr->name);
    Tk_FreeOptions(pieEltConfig, (char *) eltPtr, piePtr->display, 0);
    ckfree((char *) eltPtr);
}

static TkPieElt *
Pie_FindElt(TkPie *piePtr, char *name)
{
    TkPieElt *eltPtr;

    for (eltPtr = piePtr->firstElt; eltPtr != NULL;
	 eltPtr = eltPtr->next) {
	if ((eltPtr->name != NULL) && strcmp(eltPtr->name, name)==0) {
	    return eltPtr;
	}
    }
    /* the calling party has to deal with errors */
    return (TkPieElt *)NULL;
}

static int
Pie_GetElt(TkPie *piePtr, char *name, TkPieElt **eltPtr)
{
    if ((*eltPtr = Pie_FindElt(piePtr, name)) == NULL) {
	Tcl_AppendStringsToObj(Tcl_GetObjResult(piePtr->interp),
			       "unknown pie slice \"", name, "\"",
			       (char *) NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

static TkPieElt *
Pie_CreateElt(TkPie *piePtr, char *name)
{
    TkPieElt *eltPtr;

    /* No check for an equally name elt is made */
    eltPtr		= (TkPieElt *)ckalloc(sizeof(TkPieElt));
    eltPtr->name	= (char *)ckalloc((strlen(name)+1)*sizeof(char));
    strcpy(eltPtr->name, name);
    eltPtr->label	= NULL;
    eltPtr->fg		= NULL;
    eltPtr->value	= 0.0;
    eltPtr->explode	= 0;
    eltPtr->next	= NULL;
    eltPtr->prev	= NULL;

    if (piePtr->firstElt == NULL) {
	/* the only elt */
	piePtr->firstElt = eltPtr;
    } else if (piePtr->firstElt->prev == NULL) {
	/* only one elt */
	piePtr->firstElt->prev = eltPtr;
	piePtr->firstElt->next = eltPtr;
	eltPtr->prev = piePtr->firstElt;
    } else {
	/* general case */
	TkPieElt *prevElt = piePtr->firstElt->prev;
	piePtr->firstElt->prev = eltPtr;
	eltPtr->prev = prevElt;
	prevElt->next = eltPtr;
    }

    piePtr->totalValue += eltPtr->value;
    piePtr->numSlices++;
    return eltPtr;
}

static void
Pie_DeleteElt(TkPie *piePtr, TkPieElt *eltPtr)
{
    if (eltPtr == piePtr->firstElt) {
	/* It was the first elt */
	if (eltPtr->next == NULL) {
	    /* Pie_FreeElt(piePtr, eltPtr); */
	    piePtr->firstElt = NULL;
	} else {
	    eltPtr->next->prev = eltPtr->prev;
	    piePtr->firstElt = eltPtr->next;
	}
    } else if (eltPtr->next == NULL) {
	/* It was the last elt */
	piePtr->firstElt->prev = eltPtr->prev;
	eltPtr->prev->next = NULL;
    } else {
	/* It was the somewhere in the middle */
	eltPtr->prev->next = eltPtr->next;
	eltPtr->next->prev = eltPtr->prev;
    }
    Pie_FreeElt(piePtr, eltPtr);
    piePtr->totalValue -= eltPtr->value;
    piePtr->numSlices--;
}

static int
Pie_SwapElt(TkPieElt *eltPtr, TkPieElt *swapElt)
{
    TkPieElt tmpElt;

    if (eltPtr == NULL || swapElt == NULL || eltPtr == swapElt) {
	return 0;
    }
    tmpElt.name		= swapElt->name;
    tmpElt.label	= swapElt->label;
    tmpElt.value	= swapElt->value;
    tmpElt.explode	= swapElt->explode;
    tmpElt.fg		= swapElt->fg;
    swapElt->name	= eltPtr->name;
    swapElt->label	= eltPtr->label;
    swapElt->value	= eltPtr->value;
    swapElt->explode	= eltPtr->explode;
    swapElt->fg		= eltPtr->fg;
    eltPtr->name	= tmpElt.name;
    eltPtr->label	= tmpElt.label;
    eltPtr->value	= tmpElt.value;
    eltPtr->explode	= tmpElt.explode;
    eltPtr->fg		= tmpElt.fg;
    return 1;
}

static int
Pie_LowerElt(TkPie *piePtr, TkPieElt *eltPtr, TkPieElt *belowThis)
{
    if (eltPtr == NULL || eltPtr == belowThis ||
	piePtr->firstElt == NULL ||
	piePtr->firstElt->next == NULL) {
	return 0;
    }
    if (belowThis == NULL || belowThis->next == NULL) {
	/* put this elt on the bottom */
	if (eltPtr->next == NULL) {
	    return 0;
	} else if (eltPtr == piePtr->firstElt) {
	    piePtr->firstElt = eltPtr->next;
	    eltPtr->prev->next = eltPtr;
	    eltPtr->next = NULL;
	} else {
	    /* cut out the elt */
	    eltPtr->prev->next = eltPtr->next;
	    eltPtr->next->prev = eltPtr->prev;
	    /* put it last */
	    eltPtr->next = NULL;
	    eltPtr->prev = piePtr->firstElt->prev;
	    eltPtr->prev->next = eltPtr;
	    piePtr->firstElt->prev = eltPtr;
	}
    } else {
	/* put this elt after belowThis */
	if (belowThis->next == eltPtr) {
	    return 0;
	} else if (eltPtr == piePtr->firstElt) {
	    piePtr->firstElt = eltPtr->next;
	} else {
	    eltPtr->prev->next = eltPtr->next;
	}
	eltPtr->next->prev = eltPtr->prev;
	/* put it back in the right place */
	eltPtr->next = belowThis->next;
	eltPtr->prev = belowThis;
	belowThis->next->prev = eltPtr;
	belowThis->next = eltPtr;
    }
    return 1;
}

static int
Pie_RaiseElt(TkPie *piePtr, TkPieElt *eltPtr, TkPieElt *aboveThis)
{
    if (eltPtr == NULL || eltPtr == aboveThis ||
	piePtr->firstElt == NULL ||
	piePtr->firstElt->next == NULL) {
	return 0;
    }
    if (aboveThis == NULL || aboveThis == piePtr->firstElt) {
	/* put this elt on the top */
	if (eltPtr == piePtr->firstElt) {
	    /* can happen if aboveThis == NULL */
	    return 0;
	} else if (eltPtr == piePtr->firstElt->prev) {
	    /* it was last */
	    eltPtr->prev->next = NULL;
	    eltPtr->next = piePtr->firstElt;
	} else {
	    /* cut out the elt */
	    eltPtr->prev->next = eltPtr->next;
	    eltPtr->next->prev = eltPtr->prev;
	    /* put it first */
	    eltPtr->next = piePtr->firstElt;
	    eltPtr->prev = piePtr->firstElt->prev;
	    piePtr->firstElt->prev = eltPtr;
	}
	piePtr->firstElt = eltPtr;
    } else {
	/* put this elt before aboveThis */
	if (aboveThis->prev == eltPtr) {
	    return 0;
	} else if (eltPtr == piePtr->firstElt) {
	    piePtr->firstElt = eltPtr->next;
	} else {
	    eltPtr->prev->next = eltPtr->next;
	}
	eltPtr->next->prev = eltPtr->prev;
	/* put it back in the right place */
	eltPtr->next = aboveThis;
	eltPtr->prev = aboveThis->prev;
	aboveThis->prev->next = eltPtr;
	aboveThis->prev = eltPtr;
    }
    return 1;
}

static int
ConfigurePieElt(interp, piePtr, eltPtr, argc, argv, flags)
     Tcl_Interp *interp;
     register TkPie *piePtr;
     register TkPieElt *eltPtr;
     int argc;
     char **argv;
     int flags;
{
    double value = eltPtr->value;

    /* slice has fg, label, value, explode, && name, gc, prev, next */
    if (Tk_ConfigureWidget(interp, piePtr->tkwin, pieEltConfig,
			   argc, argv, (char *) eltPtr, flags) != TCL_OK)
	return TCL_ERROR;

    if (eltPtr->label == NULL) {
	/* Just a strdup using ckalloc */
	eltPtr->label = (char *)ckalloc((strlen(eltPtr->name)+1)*sizeof(char));
	strcpy(eltPtr->label, eltPtr->name);
    }

    piePtr->totalValue += eltPtr->value - value;

    if (eltPtr->fg == NULL) {
	/*
	 * Allocate from the default list - this doesn't check for duplicates,
	 * but the user should really set colors anyway
	 */
	eltPtr->fg = Tk_Get3DBorder(interp, piePtr->tkwin,
				    PieColors[piePtr->autoColor]);
	/* We hope that the names never return an error on any system... */
	if (eltPtr->fg == NULL) {
	    Tcl_AddErrorInfo(interp, "\n    (static name in pie code)");
	    return TCL_ERROR;
	}
	piePtr->autoColor = (piePtr->autoColor+1)%MAX_COLOR;
    }

    if (!(piePtr->flags & AVOID_REDRAW)) {
	ComputePieGeometry(piePtr);
	RedrawPie(piePtr);
    }
    return TCL_OK;
}
