/* 
 * tkSpinbox.c --
 *
 *	This module implements spinbox widgets for the Tk
 *	toolkit.  An spinbox displays a string and allows
 *	the string to be edited.
 *
 * Copyright (c) 1990-1994 The Regents of the University of California.
 * Copyright (c) 1994-1997 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * RCS: @(#) $Id: tkSpinbox.c,v 1.1.1.1 2000/07/27 00:06:46 de Exp $
 */

#ifdef SPINBOX_IN_CORE
#include "tkInt.h"
#include "default.h"
#endif

#include "pkg.h"

/*
 * A data structure of the following type is kept for each spinbox
 * widget managed by this file:
 */

typedef struct {
    Tk_Window tkwin;		/* Window that embodies the spinbox. NULL
				 * means that the window has been destroyed
				 * but the data structures haven't yet been
				 * cleaned up.*/
    Display *display;		/* Display containing widget.  Used, among
				 * other things, so that resources can be
				 * freed even after tkwin has gone away. */
    Tcl_Interp *interp;		/* Interpreter associated with spinbox. */
    Tcl_Command widgetCmd;	/* Token for spinbox's widget command. */
    Tk_OptionTable optionTable;	/* Table that defines configuration options
				 * available for this widget. */


    /*
     * Fields that are set by widget commands other than "configure".
     */
     
    char *string;		/* Pointer to storage for string;
				 * NULL-terminated;  malloc-ed. */
    int insertPos;		/* Character index before which next typed
				 * character will be inserted. */

    /*
     * Information about what's selected, if any.
     */

    int selectFirst;		/* Character index of first selected
				 * character (-1 means nothing selected. */
    int selectLast;		/* Character index just after last selected
				 * character (-1 means nothing selected. */
    int selectAnchor;		/* Fixed end of selection (i.e. "select to"
				 * operation will use this as one end of the
				 * selection). */

    /*
     * Information for scanning:
     */

    int scanMarkX;		/* X-position at which scan started (e.g.
				 * button was pressed here). */
    int scanMarkIndex;		/* Character index of character that was at
				 * left of window when scan started. */

    /*
     * Configuration settings that are updated by Tk_ConfigureWidget.
     */

    Tk_3DBorder normalBorder;	/* Used for drawing border around whole
				 * window, plus used for background. */
    Tk_3DBorder activeBorder;	/* Used for drawing border around active
				 * buttons. */
    Tk_3DBorder buttonBorder;	/* Used for drawing border around buttons. */
    int borderWidth;		/* Width of 3-D border around window. */
    Tk_Cursor cursor;		/* cursor for entry window, or None. */
    Tk_Cursor bCursor;		/* cursor for buttons, or None. */
    int exportSelection;	/* Non-zero means tie internal widget
				 * selection to X selection. */
    Tk_Font tkfont;		/* Information about text font, or NULL. */
    XColor *fgColorPtr;		/* Text color in normal mode. */
    XColor *disabledFgColorPtr;	/* Text color in disabled mode. */
    XColor *highlightBgColorPtr;/* Color for drawing traversal highlight
				 * area when highlight is off. */
    XColor *highlightColorPtr;	/* Color for drawing traversal highlight. */
    int highlightWidth;		/* Width in pixels of highlight to draw
				 * around widget when it has the focus.
				 * <= 0 means don't draw a highlight. */
    Tk_3DBorder insertBorder;	/* Used to draw vertical bar for insertion
				 * cursor. */
    int insertBorderWidth;	/* Width of 3-D border around insert cursor. */
    int insertOffTime;		/* Number of milliseconds cursor should spend
				 * in "off" state for each blink. */
    int insertOnTime;		/* Number of milliseconds cursor should spend
				 * in "on" state for each blink. */
    int insertWidth;		/* Total width of insert cursor. */
    Tk_Justify justify;		/* Justification to use for text within
				 * window. */
    int relief;			/* 3-D effect: TK_RELIEF_RAISED, etc. */
    int bdRelief;		/* 3-D effect: TK_RELIEF_RAISED, etc. */
    int buRelief;		/* 3-D effect: TK_RELIEF_RAISED, etc. */
    Tk_3DBorder selBorder;	/* Border and background for selected
				 * characters. */
    int selBorderWidth;		/* Width of border around selection. */
    XColor *selFgColorPtr;	/* Foreground color for selected text. */
    XColor *selDisFgColorPtr;	/* Dis Foreground color for selected text. */
    int state;		        /* Normal or disabled.  Spinbox is read-only
				 * when disabled. */
    char *textVarName;		/* Name of variable (malloc'ed) or NULL.
				 * If non-NULL, spinbox's string tracks the
				 * contents of this variable and vice versa. */
    char *takeFocus;		/* Value of -takefocus option;  not used in
				 * the C code, but used by keyboard traversal
				 * scripts.  Malloc'ed, but may be NULL. */
    int prefWidth;		/* Desired width of window, measured in
				 * average characters. */
    char *scrollCmd;		/* Command prefix for communicating with
				 * scrollbar(s).  Malloc'ed.  NULL means
				 * no command to issue. */
    char *command;		/* Command to invoke for spin buttons.
				 * NULL means no command to issue. */

    /*
     * Fields whose values are derived from the current values of the
     * configuration settings above.
     */

    int numBytes;		/* Length of string in bytes. */
    int numChars;		/* Length of string in characters.  Both
				 * string and displayString have the same
				 * character length, but may have different
				 * byte lengths due to being made from
				 * different UTF-8 characters. */
    char *displayString;	/* String to use when displaying.  This may
				 * be a pointer to string, or a pointer to
				 * malloced memory with the same character
				 * length as string but whose characters
				 * are all equal to showChar. */
    int numDisplayBytes;	/* Length of displayString in bytes. */
    int inset;			/* Number of pixels on the left and right
				 * sides that are taken up by XPAD, borderWidth
				 * (if any), and highlightWidth (if any). */
    Tk_TextLayout textLayout;	/* Cached text layout information. */
    int layoutX, layoutY;	/* Origin for layout. */
    int leftX;			/* X position at which character at leftIndex
				 * is drawn (varies depending on justify). */
    int leftIndex;		/* Character index of left-most character
				 * visible in window. */
    Tcl_TimerToken insertBlinkHandler;
				/* Timer handler used to blink cursor on and
				 * off. */
    GC textGC;			/* For drawing normal text. */
    GC selTextGC;		/* For drawing selected text. */
    GC highlightGC;		/* For drawing traversal highlight. */
    int avgWidth;		/* Width of average character. */
    int bWidth;			/* Width of a button control. */
    int flags;			/* Miscellaneous flags  see below for
				 * definitions. */

    char *validateCmd;          /* Command prefix to use when invoking
				 * validate command.  NULL means don't
				 * invoke commands.  Malloc'ed. */
    char *invalidCmd;		/* Command called when a validation returns 0
				 * (successfully fails), defaults to {}. */
    int validate;               /* Non-zero means try to validate */
    int selElement;		/* currently selected control */
    int curElement;		/* currently mouseover control */

    int repeatDelay;		/* repeat delay */
    int repeatInterval;		/* repeat interval */

    double fromValue;		/* Value corresponding to left/top of dial */
    double toValue;		/* Value corresponding to right/bottom
				 * of dial */
    double increment;		/* If > 0, all values are rounded to an
				 * even multiple of this value. */
    int wrap;			/* whether to wrap around when spinning */
    char *formatBuf;		/* string into which to format value.
				 * Malloc'ed. */
    char *reqFormat;		/* Sprintf conversion specifier used for the
				 * value that the users requests. Malloc'ed. */
    char *valueFormat;		/* Sprintf conversion specifier used for
				 * the value. */
    char digitFormat[10];	/* Sprintf conversion specifier computed from
				 * digits and other information; used for
				 * the value. */

    char *valueStr;		/* Values List. Malloc'ed. */
    Tcl_Obj *listObj;		/* Pointer to the list object being used */
    int eIndex;			/* Holds the current index into elements */
    int nElements;		/* Holds the current count of elements */
} Spinbox;

/*
 * Assigned bits of "flags" fields of Spinbox structures, and what those
 * bits mean:
 *
 * REDRAW_PENDING:		Non-zero means a DoWhenIdle handler has
 *				already been queued to redisplay the spinbox.
 * BORDER_NEEDED:		Non-zero means 3-D border must be redrawn
 *				around window during redisplay.  Normally
 *				only text portion needs to be redrawn.
 * CURSOR_ON:			Non-zero means insert cursor is displayed at
 *				present.  0 means it isn't displayed.
 * GOT_FOCUS:			Non-zero means this window has the input
 *				focus.
 * UPDATE_SCROLLBAR:		Non-zero means scrollbar should be updated
 *				during next redisplay operation.
 * GOT_SELECTION:		Non-zero means we've claimed the selection.
 * SPINBOX_DELETED:		This spinbox has been effectively destroyed.
 * VALIDATING:                  Non-zero means we are in a validateCmd
 * VALIDATE_VAR:                Non-zero means we are attempting to validate
 *                              the spinbox's textvariable with validateCmd
 * VALIDATE_ABORT:              Non-zero if validatecommand signals an abort
 *                              for current procedure and make no changes
 */

#define REDRAW_PENDING		1
#define BORDER_NEEDED		2
#define CURSOR_ON		4
#define GOT_FOCUS		8
#define UPDATE_SCROLLBAR	0x10
#define GOT_SELECTION		0x20
#define SPINBOX_DELETED		0x40
#define VALIDATING		0x80
#define VALIDATE_VAR		0x100
#define VALIDATE_ABORT		0x200

/*
 * The following macro defines how many extra pixels to leave on each
 * side of the text in the spinbox.
 */

#define XPAD 1
#define YPAD 1

/*
 * A comparison function for double values
 */
#define MIN_DBL_VAL		1E-9
#define DOUBLES_EQ(d1, d2)	(fabs((d1) - (d2)) < MIN_DBL_VAL)

/*
 * The following enum is used to define a type for the -state option
 * of the Spinbox widget.  These values are used as indices into the 
 * string table below.
 */

enum state {
    STATE_DISABLED, STATE_NORMAL, STATE_UNEDITABLE
};
static char *stateStrings[] = {
    "disabled", "normal", "uneditable", (char *) NULL
};

/*
 * Definitions for -validate option values:
 */

static char *validateStrings[] = {
    "all", "key", "focus", "focusin", "focusout", "none", (char *) NULL
};
enum validateType {
    VALIDATE_ALL, VALIDATE_KEY, VALIDATE_FOCUS,
    VALIDATE_FOCUSIN, VALIDATE_FOCUSOUT, VALIDATE_NONE,
    /*
     * These extra enums are for use with SpinboxValidateChange
     */
    VALIDATE_FORCED, VALIDATE_DELETE, VALIDATE_INSERT, VALIDATE_BUTTON
};

/*
 * Information used for argv parsing.
 */

static Tk_OptionSpec optionSpecs[] = {
    {TK_OPTION_BORDER, "-activebackground", "activeBackground", "Background",
	DEF_BUTTON_ACTIVE_BG_COLOR, -1, Tk_Offset(Spinbox, activeBorder),
	0, (ClientData) DEF_BUTTON_ACTIVE_BG_MONO, 0},
    {TK_OPTION_BORDER, "-background", "background", "Background",
	DEF_SPINBOX_BG_COLOR, -1, Tk_Offset(Spinbox, normalBorder),
	0, (ClientData) DEF_SPINBOX_BG_MONO, 0},
    {TK_OPTION_SYNONYM, "-bd", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, -1, 0, (ClientData) "-borderwidth", 0},
    {TK_OPTION_SYNONYM, "-bg", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, -1, 0, (ClientData) "-background", 0},
    {TK_OPTION_PIXELS, "-borderwidth", "borderWidth", "BorderWidth",
	DEF_SPINBOX_BORDER_WIDTH, -1, Tk_Offset(Spinbox, borderWidth), 
        0, 0, 0},
    {TK_OPTION_BORDER, "-buttonbackground", "background", "Background",
	DEF_BUTTON_BG_COLOR, -1, Tk_Offset(Spinbox, buttonBorder),
	0, (ClientData) DEF_BUTTON_BG_MONO, 0},
    {TK_OPTION_CURSOR, "-buttoncursor", "buttonCursor", "Cursor",
	DEF_BUTTON_CURSOR, -1, Tk_Offset(Spinbox, bCursor),
	TK_OPTION_NULL_OK, 0, 0},
    {TK_OPTION_RELIEF, "-buttondownrelief", "buttonRelief", "Relief",
	DEF_BUTTON_RELIEF, -1, Tk_Offset(Spinbox, bdRelief),
        0, 0, 0},
    {TK_OPTION_RELIEF, "-buttonuprelief", "buttonRelief", "Relief",
	DEF_BUTTON_RELIEF, -1, Tk_Offset(Spinbox, buRelief),
        0, 0, 0},
    {TK_OPTION_STRING, "-command", "command", "Command",
	DEF_SPINBOX_CMD, -1, Tk_Offset(Spinbox, command),
	TK_OPTION_NULL_OK, 0, 0},
    {TK_OPTION_CURSOR, "-cursor", "cursor", "Cursor",
	DEF_SPINBOX_CURSOR, -1, Tk_Offset(Spinbox, cursor),
	TK_OPTION_NULL_OK, 0, 0},
    {TK_OPTION_COLOR, "-disabledforeground", "disabledForeground",
	"DisabledForeground", DEF_SPINBOX_DISABLED_FG_COLOR, -1,
	Tk_Offset(Spinbox, disabledFgColorPtr),
	0, (ClientData) DEF_SPINBOX_DISABLED_FG_MONO, 0},
    {TK_OPTION_BOOLEAN, "-exportselection", "exportSelection",
        "ExportSelection", DEF_SPINBOX_EXPORT_SELECTION, -1, 
        Tk_Offset(Spinbox, exportSelection), 0, 0, 0},
    {TK_OPTION_SYNONYM, "-fg", "foreground", (char *) NULL,
	(char *) NULL, 0, -1, 0, (ClientData) "-foreground", 0},
    {TK_OPTION_FONT, "-font", "font", "Font",
	DEF_SPINBOX_FONT, -1, Tk_Offset(Spinbox, tkfont), 0, 0, 0},
    {TK_OPTION_COLOR, "-foreground", "foreground", "Foreground",
	DEF_SPINBOX_FG, -1, Tk_Offset(Spinbox, fgColorPtr), 0, 
        0, 0},
    {TK_OPTION_STRING, "-format", "format", "Format",
	DEF_SPINBOX_FORMAT, -1, Tk_Offset(Spinbox, reqFormat),
	TK_OPTION_NULL_OK, 0, 0},
    {TK_OPTION_DOUBLE, "-from", "from", "From",
	DEF_SPINBOX_FROM, -1, Tk_Offset(Spinbox, fromValue), 0, 0, 0},
    {TK_OPTION_COLOR, "-highlightbackground", "highlightBackground",
	"HighlightBackground", DEF_SPINBOX_HIGHLIGHT_BG,
	-1, Tk_Offset(Spinbox, highlightBgColorPtr), 
        0, 0, 0},
    {TK_OPTION_COLOR, "-highlightcolor", "highlightColor", "HighlightColor",
	DEF_SPINBOX_HIGHLIGHT, -1, Tk_Offset(Spinbox, highlightColorPtr),
	0, 0, 0},
    {TK_OPTION_PIXELS, "-highlightthickness", "highlightThickness",
	"HighlightThickness", DEF_SPINBOX_HIGHLIGHT_WIDTH, -1, 
	Tk_Offset(Spinbox, highlightWidth), 0, 0, 0},
    {TK_OPTION_DOUBLE, "-increment", "increment", "Increment",
	DEF_SPINBOX_INCREMENT, -1, Tk_Offset(Spinbox, increment), 0, 0, 0},
    {TK_OPTION_BORDER, "-insertbackground", "insertBackground", "Foreground",
	DEF_SPINBOX_INSERT_BG, -1, Tk_Offset(Spinbox, insertBorder), 
        0, 0, 0},
    {TK_OPTION_PIXELS, "-insertborderwidth", "insertBorderWidth", 
        "BorderWidth", DEF_SPINBOX_INSERT_BD_COLOR, -1, 
        Tk_Offset(Spinbox, insertBorderWidth), 0, 
        (ClientData) DEF_SPINBOX_INSERT_BD_MONO, 0},
    {TK_OPTION_INT, "-insertofftime", "insertOffTime", "OffTime",
        DEF_SPINBOX_INSERT_OFF_TIME, -1, Tk_Offset(Spinbox, insertOffTime), 
        0, 0, 0},
    {TK_OPTION_INT, "-insertontime", "insertOnTime", "OnTime",
        DEF_SPINBOX_INSERT_ON_TIME, -1, Tk_Offset(Spinbox, insertOnTime), 
        0, 0, 0},
    {TK_OPTION_PIXELS, "-insertwidth", "insertWidth", "InsertWidth",
	DEF_SPINBOX_INSERT_WIDTH, -1, Tk_Offset(Spinbox, insertWidth), 
        0, 0, 0},
    {TK_OPTION_STRING, "-invalidcommand", "invalidCommand", "InvalidCommand",
	DEF_SPINBOX_INVALIDCMD, -1, Tk_Offset(Spinbox, invalidCmd),
	TK_OPTION_NULL_OK, 0, 0},
    {TK_OPTION_SYNONYM, "-invcmd", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, -1, 0, (ClientData) "-invalidcommand", 0},
    {TK_OPTION_JUSTIFY, "-justify", "justify", "Justify",
	DEF_SPINBOX_JUSTIFY, -1, Tk_Offset(Spinbox, justify), 0, 0, 0},
    {TK_OPTION_RELIEF, "-relief", "relief", "Relief",
	DEF_SPINBOX_RELIEF, -1, Tk_Offset(Spinbox, relief), 
        0, 0, 0},
    {TK_OPTION_INT, "-repeatdelay", "repeatDelay", "RepeatDelay",
        DEF_SPINBOX_REPEAT_DELAY, -1, Tk_Offset(Spinbox, repeatDelay), 
        0, 0, 0},
    {TK_OPTION_INT, "-repeatinterval", "repeatInterval", "RepeatInterval",
        DEF_SPINBOX_REPEAT_INTERVAL, -1, Tk_Offset(Spinbox, repeatInterval), 
        0, 0, 0},
    {TK_OPTION_BORDER, "-selectbackground", "selectBackground", "Foreground",
        DEF_SPINBOX_SELECT_COLOR, -1, Tk_Offset(Spinbox, selBorder),
        0, (ClientData) DEF_SPINBOX_SELECT_MONO, 0},
    {TK_OPTION_PIXELS, "-selectborderwidth", "selectBorderWidth", 
        "BorderWidth", DEF_SPINBOX_SELECT_BD_COLOR, -1, 
        Tk_Offset(Spinbox, selBorderWidth), 
        0, (ClientData) DEF_SPINBOX_SELECT_BD_MONO, 0},
    {TK_OPTION_COLOR, "-selectforeground", "selectForeground", "Background",
	DEF_SPINBOX_SELECT_FG_COLOR, -1, Tk_Offset(Spinbox, selFgColorPtr),
	0, (ClientData) DEF_SPINBOX_SELECT_FG_MONO, 0},
    {TK_OPTION_COLOR, "-selectdisabledforeground", "selectDisabledForeground",
	"Background", DEF_SPINBOX_SELECT_DISABLED_FG_COLOR, -1,
	Tk_Offset(Spinbox, selFgColorPtr),
	0, (ClientData) DEF_SPINBOX_SELECT_DISABLED_FG_MONO, 0},
    {TK_OPTION_STRING_TABLE, "-state", "state", "State",
	DEF_SPINBOX_STATE, -1, Tk_Offset(Spinbox, state), 
        0, (ClientData) stateStrings, 0},
    {TK_OPTION_STRING, "-takefocus", "takeFocus", "TakeFocus",
	DEF_SPINBOX_TAKE_FOCUS, -1, Tk_Offset(Spinbox, takeFocus), 
        TK_CONFIG_NULL_OK, 0, 0},
    {TK_OPTION_STRING, "-textvariable", "textVariable", "Variable",
	DEF_SPINBOX_TEXT_VARIABLE, -1, Tk_Offset(Spinbox, textVarName),
	TK_CONFIG_NULL_OK, 0, 0},
    {TK_OPTION_DOUBLE, "-to", "to", "To",
	DEF_SPINBOX_TO, -1, Tk_Offset(Spinbox, toValue), 0, 0, 0},
    {TK_OPTION_STRING_TABLE, "-validate", "validate", "Validate",
       DEF_SPINBOX_VALIDATE, -1, Tk_Offset(Spinbox, validate),
       0, (ClientData) validateStrings, 0},
    {TK_OPTION_STRING, "-validatecommand", "validateCommand", "ValidateCommand",
       (char *) NULL, -1, Tk_Offset(Spinbox, validateCmd),
       TK_CONFIG_NULL_OK, 0, 0},
    {TK_OPTION_STRING, "-values", "values", "Values",
	 DEF_SPINBOX_VALUES, -1, Tk_Offset(Spinbox, valueStr),
	 TK_OPTION_NULL_OK, 0, 0},
    {TK_OPTION_SYNONYM, "-vcmd", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, -1, 0, (ClientData) "-validatecommand", 0},
    {TK_OPTION_INT, "-width", "width", "Width",
	DEF_SPINBOX_WIDTH, -1, Tk_Offset(Spinbox, prefWidth), 0, 0, 0},
    {TK_OPTION_BOOLEAN, "-wrap", "wrap", "Wrap",
	DEF_SPINBOX_WRAP, -1, Tk_Offset(Spinbox, wrap), 0, 0, 0},
    {TK_OPTION_STRING, "-xscrollcommand", "xScrollCommand", "ScrollCommand",
	DEF_SPINBOX_SCROLL_COMMAND, -1, Tk_Offset(Spinbox, scrollCmd),
	TK_CONFIG_NULL_OK, 0, 0},
    {TK_OPTION_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, -1, 0, 0, 0}
};

/*
 * Flags for GetSpinboxIndex procedure:
 */

#define ZERO_OK			1
#define LAST_PLUS_ONE_OK	2

/*
 * The following tables define the spinbox widget commands (and sub-
 * commands) and map the indexes into the string tables into 
 * enumerated types used to dispatch the spinbox widget command.
 */

static char *commandNames[] = {
    "bbox", "cget", "configure", "delete", "get", "icursor", "identify",
    "index", "insert", "invoke", "scan", "selection", "set",
    "validate", "xview", (char *) NULL
};

enum command {
    COMMAND_BBOX, COMMAND_CGET, COMMAND_CONFIGURE, COMMAND_DELETE, 
    COMMAND_GET, COMMAND_ICURSOR, COMMAND_IDENTIFY, COMMAND_INDEX,
    COMMAND_INSERT, COMMAND_INVOKE, COMMAND_SCAN, COMMAND_SELECTION,
    COMMAND_SET, COMMAND_VALIDATE, COMMAND_XVIEW
};

static char *selCommandNames[] = {
    "adjust", "clear", "element", "from", "present", "range", "to",
    (char *) NULL
};

enum selcommand {
    SELECTION_ADJUST, SELECTION_CLEAR, SELECTION_ELEMENT, SELECTION_FROM, 
    SELECTION_PRESENT, SELECTION_RANGE, SELECTION_TO
};

/*
 * Extra for selection of elements
 */

static char *selElementNames[] = {
    "none", "buttondown", "buttonup", (char *) NULL, "entry"
};
enum selelement {
    SEL_NONE, SEL_BUTTONDOWN, SEL_BUTTONUP, SEL_NULL, SEL_ENTRY
};

/*
 * Forward declarations for procedures defined later in this file:
 */

static int		ConfigureSpinbox _ANSI_ARGS_((Tcl_Interp *interp,
			    Spinbox *sbPtr, int objc, 
                            Tcl_Obj *CONST objv[], int flags));
static void		DeleteChars _ANSI_ARGS_((Spinbox *sbPtr, int index,
			    int count));
static void		DestroySpinbox _ANSI_ARGS_((char *memPtr));
static void		DisplaySpinbox _ANSI_ARGS_((ClientData clientData));
static void		SpinboxBlinkProc _ANSI_ARGS_((ClientData clientData));
static void		SpinboxCmdDeletedProc _ANSI_ARGS_((
			    ClientData clientData));
static void		SpinboxComputeGeometry _ANSI_ARGS_((Spinbox *sbPtr));
static void		SpinboxEventProc _ANSI_ARGS_((ClientData clientData,
			    XEvent *eventPtr));
static void		SpinboxFocusProc _ANSI_ARGS_ ((Spinbox *sbPtr,
			    int gotFocus));
static int		SpinboxFetchSelection _ANSI_ARGS_((ClientData clientData,
			    int offset, char *buffer, int maxBytes));
static void		SpinboxLostSelection _ANSI_ARGS_((
			    ClientData clientData));
static void		EventuallyRedraw _ANSI_ARGS_((Spinbox *sbPtr));
static void		SpinboxScanTo _ANSI_ARGS_((Spinbox *sbPtr, int y));
static void		SpinboxSetValue _ANSI_ARGS_((Spinbox *sbPtr,
			    char *value));
static void		SpinboxSelectTo _ANSI_ARGS_((
			    Spinbox *sbPtr, int index));
static char *		SpinboxTextVarProc _ANSI_ARGS_((ClientData clientData,
			    Tcl_Interp *interp, char *name1, char *name2,
			    int flags));
static void		SpinboxUpdateScrollbar _ANSI_ARGS_((Spinbox *sbPtr));
static int		SpinboxValidate _ANSI_ARGS_((Spinbox *sbPtr,
			    char *cmd));
static int		SpinboxValidateChange _ANSI_ARGS_((Spinbox *sbPtr,
			    char *change, char *new, int index, int type));
static void		ExpandPercents _ANSI_ARGS_((Spinbox *sbPtr,
			    char *before, char *change, char *new, int index,
			    int type, Tcl_DString *dsPtr));
static void		SpinboxValueChanged _ANSI_ARGS_((Spinbox *sbPtr,
			    char *input));
static void		SpinboxVisibleRange _ANSI_ARGS_((Spinbox *sbPtr,
			    double *firstPtr, double *lastPtr));
static int		SpinboxWidgetObjCmd _ANSI_ARGS_((ClientData clientData,
			    Tcl_Interp *interp, int objc, 
			    Tcl_Obj *CONST objv[]));
static void		SpinboxWorldChanged _ANSI_ARGS_((
			    ClientData instanceData));
static int		GetSpinboxIndex _ANSI_ARGS_((Tcl_Interp *interp,
			    Spinbox *sbPtr, char *string, int *indexPtr));
static int		GetSpinboxElement _ANSI_ARGS_((Spinbox *sbPtr,
			    int x, int y));
static void		InsertChars _ANSI_ARGS_((Spinbox *sbPtr, int index,
			    char *string));
static int		SpinboxInvoke _ANSI_ARGS_((Tcl_Interp *interp,
			    Spinbox *sbPtr, int element));
static int		ComputeFormat _ANSI_ARGS_((Spinbox *sbPtr));

/*
 * The structure below defines spinbox class behavior by means of procedures
 * that can be invoked from generic window code.
 */
#ifdef TkClassProcs
static TkClassProcs spinboxClass = {
    NULL,			/* createProc. */
    SpinboxWorldChanged,		/* geometryProc. */
    NULL			/* modalProc. */
};
#endif

/*
 *--------------------------------------------------------------
 *
 * Tk_SpinboxObjCmd --
 *
 *	This procedure is invoked to process the "spinbox" Tcl
 *	command.  See the user documentation for details on what
 *	it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

int
Tk_SpinboxObjCmd(clientData, interp, objc, objv)
    ClientData clientData;	/* Either NULL or pointer to option table. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int objc;			/* Number of arguments. */
    Tcl_Obj *CONST objv[];      /* Argument objects. */
{
    register Spinbox *sbPtr;
    Tk_OptionTable optionTable;
    Tk_Window tkwin;

    optionTable = (Tk_OptionTable) clientData;
    if (optionTable == NULL) {
	Tcl_CmdInfo info;
	char *name;

	/*
	 * We haven't created the option table for this widget class
	 * yet.  Do it now and save the table as the clientData for
	 * the command, so we'll have access to it in future
	 * invocations of the command.
	 */

	optionTable = Tk_CreateOptionTable(interp, optionSpecs);
	name = Tcl_GetString(objv[0]);
	Tcl_GetCommandInfo(interp, name, &info);
	info.objClientData = (ClientData) optionTable;
	Tcl_SetCommandInfo(interp, name, &info);
    }

    if (objc < 2) {
	Tcl_WrongNumArgs(interp, 1, objv, "pathName ?options?");
	return TCL_ERROR;
    }

    tkwin = Tk_CreateWindowFromPath(interp, Tk_MainWindow(interp),
            Tcl_GetString(objv[1]), (char *) NULL);
    if (tkwin == NULL) {
	return TCL_ERROR;
    }

    /*
     * Initialize the fields of the structure that won't be initialized
     * by ConfigureSpinbox, or that ConfigureSpinbox requires to be
     * initialized already (e.g. resource pointers).
     */

    sbPtr			= (Spinbox *) ckalloc(sizeof(Spinbox));
    sbPtr->tkwin		= tkwin;
    sbPtr->display		= Tk_Display(tkwin);
    sbPtr->interp		= interp;
    sbPtr->widgetCmd		= Tcl_CreateObjCommand(interp,
	    Tk_PathName(sbPtr->tkwin), SpinboxWidgetObjCmd,
	    (ClientData) sbPtr, SpinboxCmdDeletedProc);
    sbPtr->optionTable		= optionTable;
    sbPtr->string		= (char *) ckalloc(1);
    sbPtr->string[0]		= '\0';
    sbPtr->insertPos		= 0;
    sbPtr->selectFirst		= -1;
    sbPtr->selectLast		= -1;
    sbPtr->selectAnchor		= 0;
    sbPtr->scanMarkX		= 0;
    sbPtr->scanMarkIndex	= 0;

    sbPtr->activeBorder		= NULL;
    sbPtr->buttonBorder		= NULL;
    sbPtr->normalBorder		= NULL;
    sbPtr->borderWidth		= 0;
    sbPtr->cursor		= None;
    sbPtr->exportSelection	= 1;
    sbPtr->tkfont		= NULL;
    sbPtr->fgColorPtr		= NULL;
    sbPtr->disabledFgColorPtr	= NULL;
    sbPtr->highlightBgColorPtr	= NULL;
    sbPtr->highlightColorPtr	= NULL;
    sbPtr->highlightWidth	= 0;
    sbPtr->insertBorder		= NULL;
    sbPtr->insertBorderWidth	= 0;
    sbPtr->insertOffTime	= 0;
    sbPtr->insertOnTime		= 0;
    sbPtr->insertWidth		= 0;
    sbPtr->justify		= TK_JUSTIFY_LEFT;
    sbPtr->relief		= TK_RELIEF_FLAT;
    sbPtr->selBorder		= NULL;
    sbPtr->selBorderWidth	= 0;
    sbPtr->selFgColorPtr	= NULL;
    sbPtr->selDisFgColorPtr	= NULL;
    sbPtr->state		= STATE_NORMAL;
    sbPtr->textVarName		= NULL;
    sbPtr->takeFocus		= NULL;
    sbPtr->prefWidth		= 0;
    sbPtr->scrollCmd		= NULL;
    sbPtr->numBytes		= 0;
    sbPtr->numChars		= 0;
    sbPtr->displayString	= sbPtr->string;
    sbPtr->numDisplayBytes	= 0;
    sbPtr->inset		= XPAD;
    sbPtr->textLayout		= NULL;
    sbPtr->layoutX		= 0;
    sbPtr->layoutY		= 0;
    sbPtr->leftX		= 0;
    sbPtr->leftIndex		= 0;
    sbPtr->insertBlinkHandler	= (Tcl_TimerToken) NULL;
    sbPtr->textGC		= None;
    sbPtr->selTextGC		= None;
    sbPtr->highlightGC		= None;
    sbPtr->avgWidth		= 1;
    sbPtr->flags		= 0;
    sbPtr->validateCmd		= NULL;
    sbPtr->validate		= VALIDATE_NONE;
    sbPtr->invalidCmd		= NULL;
    sbPtr->selElement		= SEL_NONE;
    sbPtr->curElement		= SEL_NONE;
    sbPtr->command		= NULL;
    sbPtr->bCursor		= None;
    sbPtr->repeatDelay		= 400;
    sbPtr->repeatInterval	= 100;
    sbPtr->fromValue		= 0.0;
    sbPtr->toValue		= 100.0;
    sbPtr->increment		= 1.0;
    sbPtr->valueStr		= NULL;
    sbPtr->listObj		= NULL;
    sbPtr->nElements		= 0;
    sbPtr->eIndex		= 0;
    sbPtr->wrap			= 0;
    sbPtr->formatBuf		= (char *) ckalloc(TCL_DOUBLE_SPACE);
    sbPtr->reqFormat		= NULL;
    sbPtr->valueFormat		= NULL;
    sbPtr->bdRelief		= TK_RELIEF_FLAT;
    sbPtr->buRelief		= TK_RELIEF_FLAT;

    Tk_SetClass(sbPtr->tkwin, "Spinbox");
#ifdef TkClassProcs
    TkSetClassProcs(sbPtr->tkwin, &spinboxClass, (ClientData) sbPtr);
#endif
    Tk_CreateEventHandler(sbPtr->tkwin,
	    PointerMotionMask|ExposureMask|StructureNotifyMask|FocusChangeMask,
	    SpinboxEventProc, (ClientData) sbPtr);
    Tk_CreateSelHandler(sbPtr->tkwin, XA_PRIMARY, XA_STRING,
	    SpinboxFetchSelection, (ClientData) sbPtr, XA_STRING);

    if (Tk_InitOptions(interp, (char *) sbPtr, optionTable, tkwin)
	    != TCL_OK) {
	Tk_DestroyWindow(sbPtr->tkwin);
	return TCL_ERROR;
    }
    if (ConfigureSpinbox(interp, sbPtr, objc-2, objv+2, 0) != TCL_OK) {
	goto error;
    }
    
    Tcl_SetResult(interp, Tk_PathName(sbPtr->tkwin), TCL_STATIC);
    return TCL_OK;

    error:
    Tk_DestroyWindow(sbPtr->tkwin);
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * SpinboxWidgetObjCmd --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to a widget managed by this module.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

static int
SpinboxWidgetObjCmd(clientData, interp, objc, objv)
    ClientData clientData;	/* Information about spinbox widget. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int objc;			/* Number of arguments. */
    Tcl_Obj *CONST objv[];	/* Argument objects. */
{
    Spinbox *sbPtr = (Spinbox *) clientData;
    int cmdIndex, selIndex, result;
    Tcl_Obj *objPtr;

    if (objc < 2) {
	Tcl_WrongNumArgs(interp, 1, objv, "option ?arg arg ...?");
	return TCL_ERROR;
    }
    Tcl_Preserve((ClientData) sbPtr);

    /* 
     * Parse the widget command by looking up the second token in
     * the list of valid command names. 
     */

    result = Tcl_GetIndexFromObj(interp, objv[1], commandNames,
	    "option", 0, &cmdIndex);
    if (result != TCL_OK) {
	return result;
    }

    switch (cmdIndex) {
        case COMMAND_BBOX: {
	    int index, x, y, width, height;
	    char buf[TCL_INTEGER_SPACE * 4];

	    if (objc != 3) {
	        Tcl_WrongNumArgs(interp, 2, objv, "index");
		goto error;
	    }
	    if (GetSpinboxIndex(interp, sbPtr, Tcl_GetString(objv[2]), 
                    &index) != TCL_OK) {
	        goto error;
	    }
	    if ((index == sbPtr->numChars) && (index > 0)) {
	        index--;
	    }
	    Tk_CharBbox(sbPtr->textLayout, index, &x, &y, 
                    &width, &height);
	    sprintf(buf, "%d %d %d %d", x + sbPtr->layoutX,
		    y + sbPtr->layoutY, width, height);
	    Tcl_SetResult(interp, buf, TCL_VOLATILE);
	    break;
	} 
	
        case COMMAND_CGET: {
	    if (objc != 3) {
	        Tcl_WrongNumArgs(interp, 2, objv, "option");
		goto error;
	    }
	    
	    objPtr = Tk_GetOptionValue(interp, (char *) sbPtr,
		    sbPtr->optionTable, objv[2], sbPtr->tkwin);
	    if (objPtr == NULL) {
		 goto error;
	    } else {
		Tcl_SetObjResult(interp, objPtr);
	    }
	    break;
	}

        case COMMAND_CONFIGURE: {
	    if (objc <= 3) {
		objPtr = Tk_GetOptionInfo(interp, (char *) sbPtr,
			sbPtr->optionTable,
			(objc == 3) ? objv[2] : (Tcl_Obj *) NULL,
			sbPtr->tkwin);
		if (objPtr == NULL) {
		    goto error;
		} else {
		    Tcl_SetObjResult(interp, objPtr);
		}
	    } else {
		result = ConfigureSpinbox(interp, sbPtr, objc-2, objv+2, 0);
	    }
	    break;
	}

        case COMMAND_DELETE: {
	    int first, last;

	    if ((objc < 3) || (objc > 4)) {
	        Tcl_WrongNumArgs(interp, 2, objv, "firstIndex ?lastIndex?");
		goto error;
	    }
	    if (GetSpinboxIndex(interp, sbPtr, Tcl_GetString(objv[2]), 
                    &first) != TCL_OK) {
	        goto error;
	    }
	    if (objc == 3) {
	        last = first + 1;
	    } else {
	        if (GetSpinboxIndex(interp, sbPtr, Tcl_GetString(objv[3]), 
                        &last) != TCL_OK) {
		    goto error;
		}
	    }
	    if ((last >= first) && (sbPtr->state == STATE_NORMAL)) {
	        DeleteChars(sbPtr, first, last - first);
	    }
	    break;
	}

        case COMMAND_GET: {
	    if (objc != 2) {
	        Tcl_WrongNumArgs(interp, 2, objv, (char *) NULL);
		goto error;
	    }
	    Tcl_SetResult(interp, sbPtr->string, TCL_STATIC);
	    break;
	}

        case COMMAND_ICURSOR: {
	    if (objc != 3) {
	        Tcl_WrongNumArgs(interp, 2, objv, "pos");
		goto error;
	    }
	    if (GetSpinboxIndex(interp, sbPtr, Tcl_GetString(objv[2]),
                    &sbPtr->insertPos) != TCL_OK) {
	        goto error;
	    }
	    EventuallyRedraw(sbPtr);
	    break;
	}
	
        case COMMAND_IDENTIFY: {
	    int x, y, elem;

	    if (objc != 4) {
	        Tcl_WrongNumArgs(interp, 2, objv, "x y");
		goto error;
	    }
	    if ((Tcl_GetIntFromObj(interp, objv[2], &x) != TCL_OK) ||
		    (Tcl_GetIntFromObj(interp, objv[3], &y) != TCL_OK)) {
		goto error;
	    }
	    elem = GetSpinboxElement(sbPtr, x, y);
	    if (elem != SEL_NONE) {
		Tcl_SetResult(interp, selElementNames[elem], TCL_VOLATILE);
	    }
	    break;
	}
	
        case COMMAND_INDEX: {
	    int index;

	    if (objc != 3) {
	        Tcl_WrongNumArgs(interp, 2, objv, "string");
		goto error;
	    }
	    if (GetSpinboxIndex(interp, sbPtr, Tcl_GetString(objv[2]), 
                    &index) != TCL_OK) {
	        goto error;
	    }
	    Tcl_SetObjResult(interp, Tcl_NewIntObj(index));
	    break;
	}

        case COMMAND_INSERT: {
	    int index;

	    if (objc != 4) {
	        Tcl_WrongNumArgs(interp, 2, objv, "index text");
		goto error;
	    }
	    if (GetSpinboxIndex(interp, sbPtr, Tcl_GetString(objv[2]), 
                    &index) != TCL_OK) {
	        goto error;
	    }
	    if (sbPtr->state == STATE_NORMAL) {
	        InsertChars(sbPtr, index, Tcl_GetString(objv[3]));
	    }
	    break;
	}

        case COMMAND_INVOKE: {
	    if (objc != 3) {
	        Tcl_WrongNumArgs(interp, 2, objv, "elemName");
		goto error;
	    }
	    result = Tcl_GetIndexFromObj(interp, objv[2],
		    selElementNames, "element", 0, &cmdIndex);
	    if (result != TCL_OK) {
		goto error;
	    }
	    if (sbPtr->state != STATE_DISABLED) {
		if (SpinboxInvoke(interp, sbPtr, cmdIndex) != TCL_OK) {
		    goto error;
		}
	    }
	    break;
	}

        case COMMAND_SCAN: {
	    int x;
	    char *minorCmd;

	    if (objc != 4) {
	        Tcl_WrongNumArgs(interp, 2, objv, "mark|dragto x");
		goto error;
	    }
	    if (Tcl_GetIntFromObj(interp, objv[3], &x) != TCL_OK) {
	        goto error;
	    }

	    minorCmd = Tcl_GetString(objv[2]);
	    if (minorCmd[0] == 'm' 
                    && (strncmp(minorCmd, "mark", strlen(minorCmd)) == 0)) {
	        sbPtr->scanMarkX = x;
		sbPtr->scanMarkIndex = sbPtr->leftIndex;
	    } else if ((minorCmd[0] == 'd')
		&& (strncmp(minorCmd, "dragto", strlen(minorCmd)) == 0)) {
	        SpinboxScanTo(sbPtr, x);
	    } else {
	        Tcl_AppendResult(interp, "bad scan option \"", 
                        Tcl_GetString(objv[2]), "\": must be mark or dragto", 
                        (char *) NULL);
		goto error;
	    }
	    break;
	}
	    
	case COMMAND_SELECTION: {
	    int index, index2;

	    if (objc < 3) {
	        Tcl_WrongNumArgs(interp, 2, objv, "option ?index?");
		goto error;
	    }

	    /* 
	     * Parse the selection sub-command, using the command
	     * table "selCommandNames" defined above.
	     */
	    
	    result = Tcl_GetIndexFromObj(interp, objv[2], selCommandNames,
                    "selection option", 0, &selIndex);
	    if (result != TCL_OK) {
	        goto error;
	    }

	    switch(selIndex) {
	        case SELECTION_ADJUST: {
		    if (objc != 4) {
		        Tcl_WrongNumArgs(interp, 3, objv, "index");
			goto error;
		    }
		    if (GetSpinboxIndex(interp, sbPtr, 
                            Tcl_GetString(objv[3]), &index) != TCL_OK) {
		        goto error;
		    }
		    if (sbPtr->selectFirst >= 0) {
		        int half1, half2;
		
			half1 = (sbPtr->selectFirst 
			        + sbPtr->selectLast)/2;
			half2 = (sbPtr->selectFirst 
				+ sbPtr->selectLast + 1)/2;
			if (index < half1) {
			    sbPtr->selectAnchor = sbPtr->selectLast;
			} else if (index > half2) {
			    sbPtr->selectAnchor = sbPtr->selectFirst;
			} else {
			  /*
			   * We're at about the halfway point in the 
			   * selection; just keep the existing anchor.
			   */
			}
		    }
		    SpinboxSelectTo(sbPtr, index);
		    break;
		}

	        case SELECTION_CLEAR: {
		    if (objc != 3) {
		        Tcl_WrongNumArgs(interp, 3, objv, (char *) NULL);
			goto error;
		    }
		    if (sbPtr->selectFirst >= 0) {
		        sbPtr->selectFirst = -1;
			sbPtr->selectLast = -1;
			EventuallyRedraw(sbPtr);
		    }
		    goto done;
		}

	        case SELECTION_FROM: {
		    if (objc != 4) {
		        Tcl_WrongNumArgs(interp, 3, objv, "index");
			goto error;
		    }
		    if (GetSpinboxIndex(interp, sbPtr, 
                            Tcl_GetString(objv[3]), &index) != TCL_OK) {
		        goto error;
		    }
		    sbPtr->selectAnchor = index;
		    break;
		}

	        case SELECTION_PRESENT: {
		    if (objc != 3) {
		        Tcl_WrongNumArgs(interp, 3, objv, (char *) NULL);
			goto error;
		    }
		    Tcl_SetObjResult(interp,
			    Tcl_NewBooleanObj((sbPtr->selectFirst >= 0)));
		    goto done;
		}

	        case SELECTION_RANGE: {
		    if (objc != 5) {
		        Tcl_WrongNumArgs(interp, 3, objv, "start end");
			goto error;
		    }
		    if (GetSpinboxIndex(interp, sbPtr, 
                            Tcl_GetString(objv[3]), &index) != TCL_OK) {
		        goto error;
		    }
		    if (GetSpinboxIndex(interp, sbPtr, 
                            Tcl_GetString(objv[4]),& index2) != TCL_OK) {
		        goto error;
		    }
		    if (index >= index2) {
		        sbPtr->selectFirst = -1;
			sbPtr->selectLast = -1;
		    } else {
		        sbPtr->selectFirst = index;
			sbPtr->selectLast = index2;
		    }
		    if (!(sbPtr->flags & GOT_SELECTION)
			    && (sbPtr->exportSelection)) {
		        Tk_OwnSelection(sbPtr->tkwin, XA_PRIMARY, 
			        SpinboxLostSelection, (ClientData) sbPtr);
			sbPtr->flags |= GOT_SELECTION;
		    }
		    EventuallyRedraw(sbPtr);
		    break;
		}
		
	        case SELECTION_TO: {
		    if (objc != 4) {
		        Tcl_WrongNumArgs(interp, 3, objv, "index");
			goto error;
		    }
		    if (GetSpinboxIndex(interp, sbPtr, 
                            Tcl_GetString(objv[3]), &index) != TCL_OK) {
		        goto error;
		    }
		    SpinboxSelectTo(sbPtr, index);
		    break;
		}

	        case SELECTION_ELEMENT: {
		    if ((objc < 3) || (objc > 4)) {
		        Tcl_WrongNumArgs(interp, 3, objv, "?elemName?");
			goto error;
		    }
		    if (objc == 3) {
			Tcl_SetResult(interp,
				selElementNames[sbPtr->selElement],
				TCL_VOLATILE);
		    } else {
			int lastElement = sbPtr->selElement;

			result = Tcl_GetIndexFromObj(interp, objv[3],
				selElementNames, "selection element", 0,
				&(sbPtr->selElement));
			if (result != TCL_OK) {
			    goto error;
			}
			if (lastElement != sbPtr->selElement) {
			    EventuallyRedraw(sbPtr);
			}
		    }
		    break;
		}
	    }
	    break;
	}

        case COMMAND_SET: {
	    if (objc > 3) {
	        Tcl_WrongNumArgs(interp, 2, objv, "?string?");
		goto error;
	    }
	    if (objc == 3) {
		SpinboxValueChanged(sbPtr, Tcl_GetString(objv[2]));
	    }
	    Tcl_SetResult(interp, sbPtr->string, TCL_STATIC);
	    break;
	}

        case COMMAND_VALIDATE: {
	    int code;

	    if (objc != 2) {
		Tcl_WrongNumArgs(interp, 2, objv, (char *) NULL);
		goto error;
	    }
	    selIndex = sbPtr->validate;
	    sbPtr->validate = VALIDATE_ALL;
	    code = SpinboxValidateChange(sbPtr, (char *) NULL,
		    sbPtr->string, -1, VALIDATE_FORCED);
	    if (sbPtr->validate != VALIDATE_NONE) {
		sbPtr->validate = selIndex;
	    }
	    Tcl_SetObjResult(interp, Tcl_NewBooleanObj((code == TCL_OK)));
	    break;
	}

        case COMMAND_XVIEW: {
	    int index;

	    if (objc == 2) {
	        double first, last;
		char buf[TCL_DOUBLE_SPACE * 2];
	    
		SpinboxVisibleRange(sbPtr, &first, &last);
		sprintf(buf, "%g %g", first, last);
		Tcl_SetResult(interp, buf, TCL_VOLATILE);
		goto done;
	    } else if (objc == 3) {
	        if (GetSpinboxIndex(interp, sbPtr, Tcl_GetString(objv[2]), 
                        &index) != TCL_OK) {
		    goto error;
		}
	    } else {
	        double fraction;
		int count;

		index = sbPtr->leftIndex;
		switch (Tk_GetScrollInfoObj(interp, objc, objv, &fraction, 
                        &count)) {
		    case TK_SCROLL_ERROR: {
		        goto error;
		    }
		    case TK_SCROLL_MOVETO: {
		        index = (int) ((fraction * sbPtr->numChars) + 0.5);
			break;
		    }
		    case TK_SCROLL_PAGES: {
		        int charsPerPage;
		    
			charsPerPage = ((Tk_Width(sbPtr->tkwin)
    			        - 2 * sbPtr->inset - sbPtr->bWidth) 
                                / sbPtr->avgWidth) - 2;
			if (charsPerPage < 1) {
			    charsPerPage = 1;
			}
			index += count * charsPerPage;
			break;
		    }
		    case TK_SCROLL_UNITS: {
		        index += count;
			break;
		    }
		}
	    }
	    if (index >= sbPtr->numChars) {
	        index = sbPtr->numChars - 1;
	    }
	    if (index < 0) {
	        index = 0;
	    }
	    sbPtr->leftIndex = index;
	    sbPtr->flags |= UPDATE_SCROLLBAR;
	    SpinboxComputeGeometry(sbPtr);
	    EventuallyRedraw(sbPtr);
	    break;
	}
    }

    done:
    Tcl_Release((ClientData) sbPtr);
    return result;

    error:
    Tcl_Release((ClientData) sbPtr);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * DestroySpinbox --
 *
 *	This procedure is invoked by Tcl_EventuallyFree or Tcl_Release
 *	to clean up the internal structure of an spinbox at a safe time
 *	(when no-one is using it anymore).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the spinbox is freed up.
 *
 *----------------------------------------------------------------------
 */

static void
DestroySpinbox(memPtr)
    char *memPtr;		/* Info about spinbox widget. */
{
    Spinbox *sbPtr = (Spinbox *) memPtr;
    sbPtr->flags |= SPINBOX_DELETED;

    Tcl_DeleteCommandFromToken(sbPtr->interp, sbPtr->widgetCmd);
    if (sbPtr->flags & REDRAW_PENDING) {
        Tcl_CancelIdleCall(DisplaySpinbox, (ClientData) sbPtr);
    }

    /*
     * Free up all the stuff that requires special handling, then
     * let Tk_FreeOptions handle all the standard option-related
     * stuff.
     */

    ckfree(sbPtr->string);
    if (sbPtr->textVarName != NULL) {
	Tcl_UntraceVar(sbPtr->interp, sbPtr->textVarName,
		TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		SpinboxTextVarProc, (ClientData) sbPtr);
    }
    /* If we have an internal list object, free it */
    if (sbPtr->listObj != NULL) {
	Tcl_DecrRefCount(sbPtr->listObj);
	sbPtr->listObj = NULL;
    }
    if (sbPtr->textGC != None) {
	Tk_FreeGC(sbPtr->display, sbPtr->textGC);
    }
    if (sbPtr->selTextGC != None) {
	Tk_FreeGC(sbPtr->display, sbPtr->selTextGC);
    }
    Tcl_DeleteTimerHandler(sbPtr->insertBlinkHandler);
    if (sbPtr->displayString != sbPtr->string) {
	ckfree(sbPtr->displayString);
    }
    if (sbPtr->formatBuf) {
	ckfree(sbPtr->formatBuf);
    }
    Tk_FreeTextLayout(sbPtr->textLayout);
    Tk_FreeConfigOptions((char *) sbPtr, sbPtr->optionTable,
	    sbPtr->tkwin);
    sbPtr->tkwin = NULL;
    ckfree((char *) sbPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * ConfigureSpinbox --
 *
 *	This procedure is called to process an argv/argc list, plus
 *	the Tk option database, in order to configure (or reconfigure)
 *	an spinbox widget.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then the interp's result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as colors, border width,
 *	etc. get set for sbPtr;  old resources get freed,
 *	if there were any.
 *
 *----------------------------------------------------------------------
 */

static int
ConfigureSpinbox(interp, sbPtr, objc, objv, flags)
    Tcl_Interp *interp;		/* Used for error reporting. */
    Spinbox *sbPtr;		/* Information about widget; may or may not
				 * already have values for some fields. */
    int objc;			/* Number of valid entries in argv. */
    Tcl_Obj *CONST objv[];	/* Argument objects. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    Tk_SavedOptions savedOptions;
    Tcl_Obj *errorResult = NULL;
    char *oldValues, *oldFormat;
    int error, oldExport, valuesChanged = 0;
    double oldFrom, oldTo;

    /*
     * Eliminate any existing trace on a variable monitored by the spinbox.
     */

    if (sbPtr->textVarName != NULL) {
	Tcl_UntraceVar(interp, sbPtr->textVarName, 
		TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		SpinboxTextVarProc, (ClientData) sbPtr);
    }

    oldExport	= sbPtr->exportSelection;
    oldValues	= sbPtr->valueStr;
    oldFormat	= sbPtr->reqFormat;
    oldFrom	= sbPtr->fromValue;
    oldTo	= sbPtr->toValue;

    for (error = 0; error <= 1; error++) {
	if (!error) {
	    /*
	     * First pass: set options to new values.
	     */

	    if (Tk_SetOptions(interp, (char *) sbPtr,
		    sbPtr->optionTable, objc, objv,
		    sbPtr->tkwin, &savedOptions, (int *) NULL) != TCL_OK) {
		continue;
	    }
	} else {
	    /*
	     * Second pass: restore options to old values.
	     */

	    errorResult = Tcl_GetObjResult(interp);
	    Tcl_IncrRefCount(errorResult);
	    Tk_RestoreSavedOptions(&savedOptions);
	}

	/*
	 * A few other options also need special processing, such as parsing
	 * the geometry and setting the background from a 3-D border.
	 */

	Tk_SetBackgroundFromBorder(sbPtr->tkwin, sbPtr->normalBorder);

	if (sbPtr->insertWidth <= 0) {
	    sbPtr->insertWidth = 2;
	}
	if (sbPtr->insertBorderWidth > sbPtr->insertWidth/2) {
	    sbPtr->insertBorderWidth = sbPtr->insertWidth/2;
	}

	if (sbPtr->fromValue > sbPtr->toValue) {
	    Tcl_SetResult(interp, "-to value must be greater than -from value",
		    TCL_VOLATILE);
	    continue;
	}

	if (sbPtr->reqFormat && (oldFormat != sbPtr->reqFormat)) {
	    /*
	     * Make sure that the given format is somewhat correct, and
	     * calculate the minimum space we'll need for the values as
	     * strings.
	     */
	    int min, max;
	    size_t formatLen, formatSpace = TCL_DOUBLE_SPACE;
	    char fbuf[4], *fmt = sbPtr->reqFormat;

	    formatLen = strlen(fmt);
	    if ((fmt[0] != '%') || (fmt[formatLen-1] != 'f')) {
		badFormatOpt:
		Tcl_AppendResult(sbPtr->interp,
			"bad spinbox format specifier \"",
			sbPtr->reqFormat, "\"", (char *) NULL);
		continue;
	    }
	    if ((sscanf(fmt, "%%%d.%d%[f]", &min, &max, fbuf) == 3)
		    && (max >= 0)) {
		formatSpace = min + max + 1;
	    } else if (((sscanf(fmt, "%%.%d%[f]", &min, fbuf) == 2)
		    || (sscanf(fmt, "%%%d%[f]", &min, fbuf) == 2)
		    || (sscanf(fmt, "%%%d.%[f]", &min, fbuf) == 2))
		    && (min >= 0)) {
		formatSpace = min + 1;
	    } else {
		goto badFormatOpt;
	    }
	    if (formatSpace < TCL_DOUBLE_SPACE) {
		formatSpace = TCL_DOUBLE_SPACE;
	    }
	    sbPtr->formatBuf = ckrealloc(sbPtr->formatBuf, formatSpace);
	    /*
	     * We perturb the value of oldFrom to allow us to go into
	     * the branch below that will reformat the displayed value.
	     */
	    oldFrom = sbPtr->fromValue - 1;
	}

	/*
	 * See if we have to rearrange our listObj data
	 */
	if (oldValues != sbPtr->valueStr) {
	    if (sbPtr->listObj != NULL) {
		Tcl_DecrRefCount(sbPtr->listObj);
	    }
	    sbPtr->listObj = NULL;
	    if (sbPtr->valueStr != NULL) {
		Tcl_Obj *newObjPtr;
		int nelems;

		newObjPtr = Tcl_NewStringObj(sbPtr->valueStr, -1);
		if (Tcl_ListObjLength(interp, newObjPtr, &nelems) != TCL_OK) {
		    valuesChanged = -1;
		    continue;
		}
		sbPtr->listObj = newObjPtr;
		Tcl_IncrRefCount(sbPtr->listObj);
		sbPtr->nElements = nelems;
		sbPtr->eIndex = 0;
		valuesChanged++;
	    }
	}

	/*
	 * Restart the cursor timing sequence in case the on-time or 
	 * off-time just changed.  Set validate temporarily to none,
	 * so the configure doesn't cause it to be triggered.
	 */

	if (sbPtr->flags & GOT_FOCUS) {
	    int validate = sbPtr->validate;
	    sbPtr->validate = VALIDATE_NONE;
	    SpinboxFocusProc(sbPtr, 1);
	    sbPtr->validate = validate;
	}

	/*
	 * Claim the selection if we've suddenly started exporting it.
	 */

	if (sbPtr->exportSelection && (!oldExport)
	        && (sbPtr->selectFirst != -1)
	        && !(sbPtr->flags & GOT_SELECTION)) {
	    Tk_OwnSelection(sbPtr->tkwin, XA_PRIMARY, SpinboxLostSelection,
		    (ClientData) sbPtr);
	    sbPtr->flags |= GOT_SELECTION;
	}

	/*
	 * Recompute the window's geometry and arrange for it to be
	 * redisplayed.
	 */

	Tk_SetInternalBorder(sbPtr->tkwin,
	        sbPtr->borderWidth + sbPtr->highlightWidth);
	if (sbPtr->highlightWidth <= 0) {
	    sbPtr->highlightWidth = 0;
	}
	sbPtr->inset = sbPtr->highlightWidth 
	        + sbPtr->borderWidth + XPAD;
	break;
    }
    if (!error) {
	Tk_FreeSavedOptions(&savedOptions);
    }

    ComputeFormat(sbPtr);

    if (valuesChanged > 0) {
	Tcl_Obj *objPtr;

	/*
	 * No check for error return, because there shouldn't be one
	 * given the check for valid list above
	 */
	Tcl_ListObjIndex(interp, sbPtr->listObj, 0, &objPtr);
	SpinboxValueChanged(sbPtr, Tcl_GetString(objPtr));
    } else if ((sbPtr->valueStr == NULL)
	    && !DOUBLES_EQ(sbPtr->fromValue, sbPtr->toValue)
	    && (!DOUBLES_EQ(sbPtr->fromValue, oldFrom)
		    || !DOUBLES_EQ(sbPtr->toValue, oldTo))
	) {
	/*
	 * If the valueStr is empty and from && to are specified, check
	 * to see if the current string is within the range.  If not,
	 * it 
	 */
	int code;
	double dvalue;

	code = Tcl_GetDouble(NULL, sbPtr->string, &dvalue);
	if (code != TCL_OK) {
	    dvalue = sbPtr->fromValue;
	} else {
	    if (dvalue > sbPtr->toValue) {
		dvalue = sbPtr->toValue;
	    } else if (dvalue < sbPtr->fromValue) {
		dvalue = sbPtr->fromValue;
	    }
	}
	sprintf(sbPtr->formatBuf, sbPtr->valueFormat, dvalue);
	SpinboxValueChanged(sbPtr, sbPtr->formatBuf);
    }

    /*
     * If the spinbox is tied to the value of a variable, then set up
     * a trace on the variable's value, create the variable if it doesn't
     * exist, and set the spinbox's value from the variable's value.
     */

    if (sbPtr->textVarName != NULL) {
	char *value;

	value = Tcl_GetVar(interp, sbPtr->textVarName, TCL_GLOBAL_ONLY);
	if (value == NULL) {
	    SpinboxValueChanged(sbPtr, NULL);
	} else {
	    SpinboxSetValue(sbPtr, value);
	}
	Tcl_TraceVar(interp, sbPtr->textVarName,
		TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		SpinboxTextVarProc, (ClientData) sbPtr);
    }

    SpinboxWorldChanged((ClientData) sbPtr);
    if (error) {
        Tcl_SetObjResult(interp, errorResult);
	Tcl_DecrRefCount(errorResult);
	return TCL_ERROR;
    } else {
        return TCL_OK;
    }
}

/*
 *---------------------------------------------------------------------------
 *
 * SpinboxWorldChanged --
 *
 *      This procedure is called when the world has changed in some
 *      way and the widget needs to recompute all its graphics contexts
 *	and determine its new geometry.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Spinbox will be relayed out and redisplayed.
 *
 *---------------------------------------------------------------------------
 */
 
static void
SpinboxWorldChanged(instanceData)
    ClientData instanceData;	/* Information about widget. */
{
    XGCValues gcValues;
    GC gc = None;
    unsigned long mask;
    Spinbox *sbPtr;

    sbPtr = (Spinbox *) instanceData;

    sbPtr->avgWidth = Tk_TextWidth(sbPtr->tkfont, "0", 1);
    if (sbPtr->avgWidth == 0) {
	sbPtr->avgWidth = 1;
    }

    sbPtr->bWidth = sbPtr->avgWidth + 2 * (1+XPAD);
    if (sbPtr->bWidth < 11) {
	sbPtr->bWidth = 11; /* we want a min visible size */
    }

    if (sbPtr->normalBorder != NULL) {
	Tk_SetBackgroundFromBorder(sbPtr->tkwin, sbPtr->normalBorder);
    }

    if (sbPtr->state == STATE_DISABLED) {
	gcValues.foreground = sbPtr->disabledFgColorPtr->pixel;
    } else {
	gcValues.foreground = sbPtr->fgColorPtr->pixel;
    }
    gcValues.font = Tk_FontId(sbPtr->tkfont);
    gcValues.graphics_exposures = False;
    mask = GCForeground | GCFont | GCGraphicsExposures;
    gc = Tk_GetGC(sbPtr->tkwin, mask, &gcValues);
    if (sbPtr->textGC != None) {
	Tk_FreeGC(sbPtr->display, sbPtr->textGC);
    }
    sbPtr->textGC = gc;

    if ((sbPtr->state == STATE_DISABLED) &&
	    (sbPtr->selDisFgColorPtr != NULL)){
	gcValues.foreground = sbPtr->selDisFgColorPtr->pixel;
    } else {
	gcValues.foreground = sbPtr->selFgColorPtr->pixel;
    }
    gcValues.font = Tk_FontId(sbPtr->tkfont);
    mask = GCForeground | GCFont;
    gc = Tk_GetGC(sbPtr->tkwin, mask, &gcValues);
    if (sbPtr->selTextGC != None) {
	Tk_FreeGC(sbPtr->display, sbPtr->selTextGC);
    }
    sbPtr->selTextGC = gc;

    /*
     * Recompute the window's geometry and arrange for it to be
     * redisplayed.
     */

    SpinboxComputeGeometry(sbPtr);
    sbPtr->flags |= UPDATE_SCROLLBAR;
    EventuallyRedraw(sbPtr);
}

/*
 *--------------------------------------------------------------
 *
 * DisplaySpinbox --
 *
 *	This procedure redraws the contents of an spinbox window.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information appears on the screen.
 *
 *--------------------------------------------------------------
 */

static void
DisplaySpinbox(clientData)
    ClientData clientData;	/* Information about window. */
{
    Spinbox *sbPtr = (Spinbox *) clientData;
    Tk_Window tkwin = sbPtr->tkwin;
    int baseY, selStartX, selEndX, cursorX, bWidth;
    int xBound;
    Tk_FontMetrics fm;
    Pixmap pixmap;
    int showSelection;
    int startx, height, inset, pad, tHeight;

    sbPtr->flags &= ~REDRAW_PENDING;
    if ((sbPtr->tkwin == NULL) || !Tk_IsMapped(tkwin)) {
	return;
    }

    Tk_GetFontMetrics(sbPtr->tkfont, &fm);

    /*
     * Update the scrollbar if that's needed.
     */

    if (sbPtr->flags & UPDATE_SCROLLBAR) {
	sbPtr->flags &= ~UPDATE_SCROLLBAR;
	SpinboxUpdateScrollbar(sbPtr);
    }

    /*
     * In order to avoid screen flashes, this procedure redraws the
     * textual area of the spinbox into off-screen memory, then copies
     * it back on-screen in a single operation.  This means there's
     * no point in time where the on-screen image has been cleared.
     */

    pixmap = Tk_GetPixmap(sbPtr->display, Tk_WindowId(tkwin),
	    Tk_Width(tkwin), Tk_Height(tkwin), Tk_Depth(tkwin));

    /*
     * Compute x-coordinate of the pixel just after last visible
     * one, plus vertical position of baseline of text.
     */

    xBound = Tk_Width(tkwin) - sbPtr->inset - sbPtr->bWidth;
    baseY = (Tk_Height(tkwin) + fm.ascent - fm.descent) / 2;

    /*
     * On Windows and Mac, we need to hide the selection whenever we
     * don't have the focus.
     */

#ifdef ALWAYS_SHOW_SELECTION
    showSelection = 1;
#else
    showSelection = (sbPtr->flags & GOT_FOCUS);
#endif

    /*
     * Draw the background in three layers.  From bottom to top the
     * layers are:  normal background, selection background, and
     * insertion cursor background.
     */

    Tk_Fill3DRectangle(tkwin, pixmap, sbPtr->normalBorder,
		0, 0, Tk_Width(tkwin), Tk_Height(tkwin), 0, TK_RELIEF_FLAT);

    if (showSelection
	    && (sbPtr->selectLast > sbPtr->leftIndex)) {
	if (sbPtr->selectFirst <= sbPtr->leftIndex) {
	    selStartX = sbPtr->leftX;
	} else {
	    Tk_CharBbox(sbPtr->textLayout, sbPtr->selectFirst,
		    &selStartX, NULL, NULL, NULL);
	    selStartX += sbPtr->layoutX;
	}
	if ((selStartX - sbPtr->selBorderWidth) < xBound) {
	    Tk_CharBbox(sbPtr->textLayout, sbPtr->selectLast,
		    &selEndX, NULL, NULL, NULL);
	    selEndX += sbPtr->layoutX;
	    Tk_Fill3DRectangle(tkwin, pixmap, sbPtr->selBorder,
		    selStartX - sbPtr->selBorderWidth,
		    baseY - fm.ascent - sbPtr->selBorderWidth,
		    (selEndX - selStartX) + 2*sbPtr->selBorderWidth,
		    (fm.ascent + fm.descent) + 2*sbPtr->selBorderWidth,
		    sbPtr->selBorderWidth, TK_RELIEF_RAISED);
	} 
    }

    /*
     * Draw a special background for the insertion cursor, overriding
     * even the selection background.  As a special hack to keep the
     * cursor visible when the insertion cursor color is the same as
     * the color for selected text (e.g., on mono displays), write
     * background in the cursor area (instead of nothing) when the
     * cursor isn't on.  Otherwise the selection would hide the cursor.
     */

    if ((sbPtr->insertPos >= sbPtr->leftIndex)
	    && (sbPtr->state == STATE_NORMAL)
	    && (sbPtr->flags & GOT_FOCUS)) {
	Tk_CharBbox(sbPtr->textLayout, sbPtr->insertPos, &cursorX, NULL,
		NULL, NULL);
	cursorX += sbPtr->layoutX;
	cursorX -= (sbPtr->insertWidth)/2;
	if (cursorX < xBound) {
	    if (sbPtr->flags & CURSOR_ON) {
		Tk_Fill3DRectangle(tkwin, pixmap, sbPtr->insertBorder,
			cursorX, baseY - fm.ascent, sbPtr->insertWidth,
			fm.ascent + fm.descent, sbPtr->insertBorderWidth,
			TK_RELIEF_RAISED);
	    } else if (sbPtr->insertBorder == sbPtr->selBorder) {
		Tk_Fill3DRectangle(tkwin, pixmap, sbPtr->normalBorder,
			cursorX, baseY - fm.ascent, sbPtr->insertWidth,
			fm.ascent + fm.descent, 0, TK_RELIEF_FLAT);
	    }
	}
    }

    /*
     * Draw the text in two pieces:  first the unselected portion, then the
     * selected portion on top of it.
     */

    Tk_DrawTextLayout(sbPtr->display, pixmap, sbPtr->textGC,
	    sbPtr->textLayout, sbPtr->layoutX, sbPtr->layoutY,
	    sbPtr->leftIndex, sbPtr->numChars);

    if (showSelection
	    && (sbPtr->selTextGC != sbPtr->textGC)
	    && (sbPtr->selectFirst < sbPtr->selectLast)) {
	int selFirst;

	if (sbPtr->selectFirst < sbPtr->leftIndex) {
	    selFirst = sbPtr->leftIndex;
	} else {
	    selFirst = sbPtr->selectFirst;
	}
	Tk_DrawTextLayout(sbPtr->display, pixmap, sbPtr->selTextGC,
		sbPtr->textLayout, sbPtr->layoutX, sbPtr->layoutY,
		selFirst, sbPtr->selectLast);
    }

    /*
     * Draw the spin button controls.
     */
    bWidth = sbPtr->bWidth;
    pad    = XPAD + 1;
    inset  = sbPtr->inset - XPAD;
    startx = Tk_Width(tkwin) - (bWidth + inset);
    height = (Tk_Height(tkwin) - 2*inset)/2;
#if 0
    Tk_Fill3DRectangle(tkwin, pixmap, sbPtr->buttonBorder,
	    startx, inset, bWidth, height, 1, sbPtr->buRelief);
    Tk_Fill3DRectangle(tkwin, pixmap, sbPtr->buttonBorder,
	    startx, inset+height, bWidth, height, 1, sbPtr->bdRelief);
#else
    Tk_Fill3DRectangle(tkwin, pixmap, sbPtr->buttonBorder,
	    startx, inset, bWidth, height, 1,
	    (sbPtr->selElement == SEL_BUTTONUP) ?
	    TK_RELIEF_SUNKEN : TK_RELIEF_RAISED);
    Tk_Fill3DRectangle(tkwin, pixmap, sbPtr->buttonBorder,
	    startx, inset+height, bWidth, height, 1,
	    (sbPtr->selElement == SEL_BUTTONDOWN) ?
	    TK_RELIEF_SUNKEN : TK_RELIEF_RAISED);
#endif
    
    bWidth -= 2*pad;
    if ((bWidth > 1)) {
	XPoint points[3];
	int starty, space, offset;

	space   = height - 2*pad;
	/*
	 * Ensure width of triangle is odd to guarantee a sharp tip
	 */
	if (!(bWidth % 2)) {
	    bWidth++;
	}
	tHeight = (bWidth + 1) / 2;
	if (tHeight > space) {
	    tHeight = space;
	}
	space   = (space - tHeight) / 2;
	startx += pad;
	starty  = inset + height - pad - space;
	offset  = (sbPtr->selElement == SEL_BUTTONUP);
	/*
	 * The points are slightly different for the up and down arrows
	 * because (for *.x), we need to account for a bug in the way
	 * XFillPolygon draws triangles, and we want to shift
	 * the arrows differently when allowing for depressed behavior.
	 */
	points[0].x = startx + offset;
	points[0].y = starty + (offset ? 0 : -1);
	points[1].x = startx + bWidth/2 + offset;
	points[1].y = starty - tHeight + (offset ? 0 : -1);
	points[2].x = startx + bWidth + offset;
	points[2].y = points[0].y;
	XFillPolygon(sbPtr->display, pixmap, sbPtr->textGC,
		points, 3, Convex, CoordModeOrigin);

	starty = inset + height + pad + space;
	offset = (sbPtr->selElement == SEL_BUTTONDOWN);
	points[0].x = startx + 1 + offset;
	points[0].y = starty + (offset ? 1 : 0);
	points[1].x = startx + bWidth/2 + offset;
	points[1].y = starty + tHeight + (offset ? 0 : -1);
	points[2].x = startx - 1 + bWidth + offset;
	points[2].y = points[0].y;
	XFillPolygon(sbPtr->display, pixmap, sbPtr->textGC,
		points, 3, Convex, CoordModeOrigin);
    }

    /*
     * Draw the border and focus highlight last, so they will overwrite
     * any text that extends past the viewable part of the window.
     */

    if (sbPtr->relief != TK_RELIEF_FLAT) {
	inset = sbPtr->highlightWidth;
	Tk_Draw3DRectangle(tkwin, pixmap, sbPtr->normalBorder, inset, inset,
		Tk_Width(tkwin) - 2 * inset, Tk_Height(tkwin) - 2 * inset,
		sbPtr->borderWidth, sbPtr->relief);
    }
    if (sbPtr->highlightWidth > 0) {
#ifdef TkpDrawHighlightBorder
	GC fgGC, bgGC;

	bgGC = Tk_GCForColor(sbPtr->highlightBgColorPtr, pixmap);
	if (sbPtr->flags & GOT_FOCUS) {
	    fgGC = Tk_GCForColor(sbPtr->highlightColorPtr, pixmap);
	    TkpDrawHighlightBorder(tkwin, fgGC, bgGC, 
	            sbPtr->highlightWidth, pixmap);
	} else {
	    TkpDrawHighlightBorder(tkwin, bgGC, bgGC, 
	            sbPtr->highlightWidth, pixmap);
	}
#else
	GC gc = Tk_GCForColor((sbPtr->flags & GOT_FOCUS)
		? (sbPtr->highlightColorPtr) : (sbPtr->highlightBgColorPtr),
		Tk_WindowId(tkwin));
	Tk_DrawFocusHighlight(tkwin, gc, sbPtr->highlightWidth, pixmap);
#endif
    }
    
    /*
     * Everything's been redisplayed;  now copy the pixmap onto the screen
     * and free up the pixmap.
     */

    XCopyArea(sbPtr->display, pixmap, Tk_WindowId(tkwin), sbPtr->textGC,
	    0, 0, (unsigned) Tk_Width(tkwin), (unsigned) Tk_Height(tkwin),
	    0, 0);
    Tk_FreePixmap(sbPtr->display, pixmap);
    sbPtr->flags &= ~BORDER_NEEDED;
}

/*
 *----------------------------------------------------------------------
 *
 * SpinboxComputeGeometry --
 *
 *	This procedure is invoked to recompute information about where
 *	in its window an spinbox's string will be displayed.  It also
 *	computes the requested size for the window.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The leftX and tabOrigin fields are recomputed for sbPtr,
 *	and leftIndex may be adjusted.  Tk_GeometryRequest is called
 *	to register the desired dimensions for the window.
 *
 *----------------------------------------------------------------------
 */

static void
SpinboxComputeGeometry(sbPtr)
    Spinbox *sbPtr;		/* Widget record for spinbox. */
{
    int totalLength, overflow, maxOffScreen, rightX;
    int height, width;
    Tk_FontMetrics fm;

    if (sbPtr->displayString != sbPtr->string) {
	ckfree(sbPtr->displayString);
	sbPtr->displayString = sbPtr->string;
	sbPtr->numDisplayBytes = sbPtr->numBytes;
    }

    Tk_FreeTextLayout(sbPtr->textLayout);
    sbPtr->textLayout = Tk_ComputeTextLayout(sbPtr->tkfont,
	    sbPtr->displayString, sbPtr->numChars, 0,
	    sbPtr->justify, TK_IGNORE_NEWLINES, &totalLength, &height);

    sbPtr->layoutY = (Tk_Height(sbPtr->tkwin) - height) / 2;

    /*
     * Recompute where the leftmost character on the display will
     * be drawn (sbPtr->leftX) and adjust leftIndex if necessary
     * so that we don't let characters hang off the edge of the
     * window unless the entire window is full.
     */

    overflow = totalLength -
	(Tk_Width(sbPtr->tkwin) - 2*sbPtr->inset - sbPtr->bWidth);
    if (overflow <= 0) {
	sbPtr->leftIndex = 0;
	if (sbPtr->justify == TK_JUSTIFY_LEFT) {
	    sbPtr->leftX = sbPtr->inset;
	} else if (sbPtr->justify == TK_JUSTIFY_RIGHT) {
	    sbPtr->leftX = Tk_Width(sbPtr->tkwin) - sbPtr->inset
		- sbPtr->bWidth - totalLength;
	} else {
	    sbPtr->leftX = (Tk_Width(sbPtr->tkwin) - sbPtr->bWidth
		    - totalLength)/2;
	}
	sbPtr->layoutX = sbPtr->leftX;
    } else {
	/*
	 * The whole string can't fit in the window.  Compute the
	 * maximum number of characters that may be off-screen to
	 * the left without leaving empty space on the right of the
	 * window, then don't let leftIndex be any greater than that.
	 */

	maxOffScreen = Tk_PointToChar(sbPtr->textLayout, overflow, 0);
	Tk_CharBbox(sbPtr->textLayout, maxOffScreen,
		&rightX, NULL, NULL, NULL);
	if (rightX < overflow) {
	    maxOffScreen++;
	}
	if (sbPtr->leftIndex > maxOffScreen) {
	    sbPtr->leftIndex = maxOffScreen;
	}
	Tk_CharBbox(sbPtr->textLayout, sbPtr->leftIndex, &rightX,
		NULL, NULL, NULL);
	sbPtr->leftX = sbPtr->inset;
	sbPtr->layoutX = sbPtr->leftX - rightX;
    }

    Tk_GetFontMetrics(sbPtr->tkfont, &fm);
    height = fm.linespace + 2*sbPtr->inset + 2*(YPAD-XPAD);
    if (sbPtr->prefWidth > 0) {
	width = sbPtr->prefWidth*sbPtr->avgWidth + 2*sbPtr->inset;
    } else {
	if (totalLength == 0) {
	    width = sbPtr->avgWidth + 2*sbPtr->inset;
	} else {
	    width = totalLength + 2*sbPtr->inset;
	}
    }

    /*
     * Add one extra length for the spin buttons
     */
    width += sbPtr->bWidth;

    Tk_GeometryRequest(sbPtr->tkwin, width, height);
}

/*
 *----------------------------------------------------------------------
 *
 * InsertChars --
 *
 *	Add new characters to an spinbox widget.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	New information gets added to sbPtr;  it will be redisplayed
 *	soon, but not necessarily immediately.
 *
 *----------------------------------------------------------------------
 */

static void
InsertChars(sbPtr, index, value)
    Spinbox *sbPtr;		/* Spinbox that is to get the new elements. */
    int index;			/* Add the new elements before this
				 * character index. */
    char *value;		/* New characters to add (NULL-terminated
				 * string). */
{
    int byteIndex, byteCount, oldChars, charsAdded, newByteCount;
    char *new, *string;

    string = sbPtr->string;
    byteIndex = Tcl_UtfAtIndex(string, index) - string;
    byteCount = strlen(value);
    if (byteCount == 0) {
	return;
    }

    newByteCount = sbPtr->numBytes + byteCount + 1;
    new = (char *) ckalloc((unsigned) newByteCount);
    memcpy(new, string, (size_t) byteIndex);
    strcpy(new + byteIndex, value);
    strcpy(new + byteIndex + byteCount, string + byteIndex);

    if ((sbPtr->validate == VALIDATE_KEY ||
	 sbPtr->validate == VALIDATE_ALL) &&
	SpinboxValidateChange(sbPtr, value, new, index,
		VALIDATE_INSERT) != TCL_OK) {
	ckfree(new);
	return;
    }

    ckfree(string);
    sbPtr->string = new;

    /*
     * The following construction is used because inserting improperly
     * formed UTF-8 sequences between other improperly formed UTF-8
     * sequences could result in actually forming valid UTF-8 sequences;
     * the number of characters added may not be Tcl_NumUtfChars(string, -1),
     * because of context.  The actual number of characters added is how
     * many characters are in the string now minus the number that
     * used to be there.
     */

    oldChars = sbPtr->numChars;
    sbPtr->numChars = Tcl_NumUtfChars(new, -1);
    charsAdded = sbPtr->numChars - oldChars;
    sbPtr->numBytes += byteCount;

    if (sbPtr->displayString == string) {
	sbPtr->displayString = new;
	sbPtr->numDisplayBytes = sbPtr->numBytes;
    }

    /*
     * Inserting characters invalidates all indexes into the string.
     * Touch up the indexes so that they still refer to the same
     * characters (at new positions).  When updating the selection
     * end-points, don't include the new text in the selection unless
     * it was completely surrounded by the selection.
     */

    if (sbPtr->selectFirst >= index) {
	sbPtr->selectFirst += charsAdded;
    }
    if (sbPtr->selectLast > index) {
	sbPtr->selectLast += charsAdded;
    }
    if ((sbPtr->selectAnchor > index)
	    || (sbPtr->selectFirst >= index)) {
	sbPtr->selectAnchor += charsAdded;
    }
    if (sbPtr->leftIndex > index) {
	sbPtr->leftIndex += charsAdded;
    }
    if (sbPtr->insertPos >= index) {
	sbPtr->insertPos += charsAdded;
    }
    SpinboxValueChanged(sbPtr, NULL);
}

/*
 *----------------------------------------------------------------------
 *
 * DeleteChars --
 *
 *	Remove one or more characters from an spinbox widget.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Memory gets freed, the spinbox gets modified and (eventually)
 *	redisplayed.
 *
 *----------------------------------------------------------------------
 */

static void
DeleteChars(sbPtr, index, count)
    Spinbox *sbPtr;		/* Spinbox widget to modify. */
    int index;			/* Index of first character to delete. */
    int count;			/* How many characters to delete. */
{
    int byteIndex, byteCount, newByteCount;
    char *new, *string;
    char *todelete;

    if ((index + count) > sbPtr->numChars) {
	count = sbPtr->numChars - index;
    }
    if (count <= 0) {
	return;
    }

    string = sbPtr->string;
    byteIndex = Tcl_UtfAtIndex(string, index) - string;
    byteCount = Tcl_UtfAtIndex(string + byteIndex, count) - (string + byteIndex);

    newByteCount = sbPtr->numBytes + 1 - byteCount;
    new = (char *) ckalloc((unsigned) newByteCount);
    memcpy(new, string, (size_t) byteIndex);
    strcpy(new + byteIndex, string + byteIndex + byteCount);

    todelete = (char *) ckalloc((unsigned) (byteCount + 1));
    memcpy(todelete, string + byteIndex, (size_t) byteCount);
    todelete[byteCount] = '\0';

    if ((sbPtr->validate == VALIDATE_KEY ||
	 sbPtr->validate == VALIDATE_ALL) &&
	SpinboxValidateChange(sbPtr, todelete, new, index,
		VALIDATE_DELETE) != TCL_OK) {
	ckfree(new);
	ckfree(todelete);
	return;
    }

    ckfree(todelete);
    ckfree(sbPtr->string);
    sbPtr->string = new;
    sbPtr->numChars -= count;
    sbPtr->numBytes -= byteCount;

    if (sbPtr->displayString == string) {
	sbPtr->displayString = new;
	sbPtr->numDisplayBytes = sbPtr->numBytes;
    }

    /*
     * Deleting characters results in the remaining characters being
     * renumbered.  Update the various indexes into the string to reflect
     * this change.
     */

    if (sbPtr->selectFirst >= index) {
	if (sbPtr->selectFirst >= (index + count)) {
	    sbPtr->selectFirst -= count;
	} else {
	    sbPtr->selectFirst = index;
	}
    }
    if (sbPtr->selectLast >= index) {
	if (sbPtr->selectLast >= (index + count)) {
	    sbPtr->selectLast -= count;
	} else {
	    sbPtr->selectLast = index;
	}
    }
    if (sbPtr->selectLast <= sbPtr->selectFirst) {
	sbPtr->selectFirst = -1;
	sbPtr->selectLast = -1;
    }
    if (sbPtr->selectAnchor >= index) {
	if (sbPtr->selectAnchor >= (index+count)) {
	    sbPtr->selectAnchor -= count;
	} else {
	    sbPtr->selectAnchor = index;
	}
    }
    if (sbPtr->leftIndex > index) {
	if (sbPtr->leftIndex >= (index + count)) {
	    sbPtr->leftIndex -= count;
	} else {
	    sbPtr->leftIndex = index;
	}
    }
    if (sbPtr->insertPos >= index) {
	if (sbPtr->insertPos >= (index + count)) {
	    sbPtr->insertPos -= count;
	} else {
	    sbPtr->insertPos = index;
	}
    }
    SpinboxValueChanged(sbPtr, NULL);
}

/*
 *----------------------------------------------------------------------
 *
 * SpinboxValueChanged --
 *
 *	This procedure is invoked when characters are inserted into
 *	an spinbox or deleted from it.  It updates the spinbox's associated
 *	variable, if there is one, and does other bookkeeping such
 *	as arranging for redisplay.  If input is NULL, it uses the
 *	current spinbox value.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
SpinboxValueChanged(sbPtr, newValue)
    Spinbox *sbPtr;		/* Spinbox whose value just changed. */
    char *newValue;
{

    if (newValue != NULL) {
	SpinboxSetValue(sbPtr, newValue);
    }
    if (sbPtr->textVarName == NULL) {
	newValue = NULL;
    } else {
	newValue = Tcl_SetVar(sbPtr->interp, sbPtr->textVarName,
		sbPtr->string, TCL_GLOBAL_ONLY);
    }

    if ((newValue != NULL) && (strcmp(newValue, sbPtr->string) != 0)) {
	/*
	 * The value of the variable is different than what we asked for.
	 * This means that a trace on the variable modified it.  In this
	 * case our trace procedure wasn't invoked since the modification
	 * came while a trace was already active on the variable.  So,
	 * update our value to reflect the variable's latest value.
	 */

	SpinboxSetValue(sbPtr, newValue);
    } else {
	/*
	 * Arrange for redisplay.
	 */

	sbPtr->flags |= UPDATE_SCROLLBAR;
	SpinboxComputeGeometry(sbPtr);
	EventuallyRedraw(sbPtr);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * SpinboxSetValue --
 *
 *	Replace the contents of a text spinbox with a given value.  This
 *	procedure is invoked when updating the spinbox from the spinbox's
 *	associated variable.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The string displayed in the spinbox will change.  The selection,
 *	insertion point, and view may have to be adjusted to keep them
 *	within the bounds of the new string.  Note: this procedure does
 *	*not* update the spinbox's associated variable, since that could
 *	result in an infinite loop.
 *
 *----------------------------------------------------------------------
 */

static void
SpinboxSetValue(sbPtr, value)
    Spinbox *sbPtr;		/* Spinbox whose value is to be changed. */
    char *value;		/* New text to display in spinbox. */
{
    char *oldSource;
    int code, valueLen, malloced = 0;

    if (strcmp(value, sbPtr->string) == 0) {
	return;
    }
    valueLen = strlen(value);

    if (sbPtr->flags & VALIDATE_VAR) {
	sbPtr->flags |= VALIDATE_ABORT;
    } else {
	/*
	 * If we validate, we create a copy of the value, as it may
	 * point to volatile memory, like the value of the -textvar
	 * which may get freed during validation
	 */
	oldSource = (char *) ckalloc((unsigned) (valueLen + 1));
	strcpy(oldSource, value);
	value = oldSource;
	malloced = 1;

	sbPtr->flags |= VALIDATE_VAR;
	code = SpinboxValidateChange(sbPtr, (char *) NULL, value, -1,
		VALIDATE_FORCED);
	sbPtr->flags &= ~VALIDATE_VAR;
	/*
	 * If VALIDATE_ABORT has been set, then this operation should be
	 * aborted because the validatecommand did something else instead
	 */
	if (sbPtr->flags & VALIDATE_ABORT) {
	    sbPtr->flags &= ~VALIDATE_ABORT;
	    ckfree(value);
	    return;
	}
    }

    oldSource = sbPtr->string;
    ckfree(sbPtr->string);

    if (malloced) {
	sbPtr->string = value;
    } else {
	sbPtr->string = (char *) ckalloc((unsigned) (valueLen + 1));
	strcpy(sbPtr->string, value);
    }
    sbPtr->numBytes = valueLen;
    sbPtr->numChars = Tcl_NumUtfChars(value, valueLen);

    if (sbPtr->displayString == oldSource) {
	sbPtr->displayString = sbPtr->string;
	sbPtr->numDisplayBytes = sbPtr->numBytes;
    }

    if (sbPtr->selectFirst >= 0) {
	if (sbPtr->selectFirst >= sbPtr->numChars) {
	    sbPtr->selectFirst = -1;
	    sbPtr->selectLast = -1;
	} else if (sbPtr->selectLast > sbPtr->numChars) {
	    sbPtr->selectLast = sbPtr->numChars;
	}
    }
    if (sbPtr->leftIndex >= sbPtr->numChars) {
	if (sbPtr->numChars > 0) {
	    sbPtr->leftIndex = sbPtr->numChars - 1;
	} else {
	    sbPtr->leftIndex = 0;
	}
    }
    if (sbPtr->insertPos > sbPtr->numChars) {
	sbPtr->insertPos = sbPtr->numChars;
    }

    sbPtr->flags |= UPDATE_SCROLLBAR;
    SpinboxComputeGeometry(sbPtr);
    EventuallyRedraw(sbPtr);
}

/*
 *--------------------------------------------------------------
 *
 * SpinboxEventProc --
 *
 *	This procedure is invoked by the Tk dispatcher for various
 *	events on spinboxes.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	When the window gets deleted, internal structures get
 *	cleaned up.  When it gets exposed, it is redisplayed.
 *
 *--------------------------------------------------------------
 */

static void
SpinboxEventProc(clientData, eventPtr)
    ClientData clientData;	/* Information about window. */
    XEvent *eventPtr;		/* Information about event. */
{
    Spinbox *sbPtr = (Spinbox *) clientData;

    switch (eventPtr->type) {
	case MotionNotify: {
	    int elem;

	    elem = GetSpinboxElement(sbPtr, eventPtr->xmotion.x,
		    eventPtr->xmotion.y);
	    if (elem != sbPtr->curElement) {
		Tk_Cursor cursor;

		sbPtr->curElement = elem;
		if (elem == SEL_ENTRY) {
		    cursor = sbPtr->cursor;
		} else if ((elem == SEL_BUTTONDOWN)
			|| (elem == SEL_BUTTONUP)) {
		    cursor = sbPtr->bCursor;
		} else {
		    cursor = None;
		}
		if (cursor != None) {
		    Tk_DefineCursor(sbPtr->tkwin, cursor);
		} else {
		    Tk_UndefineCursor(sbPtr->tkwin);
		}
	    }
	    break;
	}
	case Expose:
	    EventuallyRedraw(sbPtr);
	    sbPtr->flags |= BORDER_NEEDED;
	    break;
	case DestroyNotify:
	    DestroySpinbox((char *) clientData);
	    break;
	case ConfigureNotify:
	    Tcl_Preserve((ClientData) sbPtr);
	    sbPtr->flags |= UPDATE_SCROLLBAR;
	    SpinboxComputeGeometry(sbPtr);
	    EventuallyRedraw(sbPtr);
	    Tcl_Release((ClientData) sbPtr);
	    break;
	case FocusIn:
	case FocusOut:
	    if (eventPtr->xfocus.detail != NotifyInferior) {
		SpinboxFocusProc(sbPtr, (eventPtr->type == FocusIn));
	    }
	    break;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * SpinboxCmdDeletedProc --
 *
 *	This procedure is invoked when a widget command is deleted.  If
 *	the widget isn't already in the process of being destroyed,
 *	this command destroys it.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The widget is destroyed.
 *
 *----------------------------------------------------------------------
 */

static void
SpinboxCmdDeletedProc(clientData)
    ClientData clientData;	/* Pointer to widget record for widget. */
{
    Spinbox *sbPtr = (Spinbox *) clientData;

    /*
     * This procedure could be invoked either because the window was
     * destroyed and the command was then deleted (in which case tkwin
     * is NULL) or because the command was deleted, and then this procedure
     * destroys the widget.
     */

    if (!(sbPtr->flags & SPINBOX_DELETED)) {
        Tk_DestroyWindow(sbPtr->tkwin);
    }
}

/*
 *---------------------------------------------------------------------------
 *
 * GetSpinboxIndex --
 *
 *	Parse an index into an spinbox and return either its value
 *	or an error.
 *
 * Results:
 *	A standard Tcl result.  If all went well, then *indexPtr is
 *	filled in with the character index (into sbPtr) corresponding to
 *	string.  The index value is guaranteed to lie between 0 and
 *	the number of characters in the string, inclusive.  If an
 *	error occurs then an error message is left in the interp's result.
 *
 * Side effects:
 *	None.
 *
 *---------------------------------------------------------------------------
 */

static int
GetSpinboxIndex(interp, sbPtr, string, indexPtr)
    Tcl_Interp *interp;		/* For error messages. */
    Spinbox *sbPtr;		/* Spinbox for which the index is being
				 * specified. */
    char *string;		/* Specifies character in sbPtr. */
    int *indexPtr;		/* Where to store converted character
				 * index. */
{
    size_t length;

    length = strlen(string);

    if (string[0] == 'a') {
	if (strncmp(string, "anchor", length) == 0) {
	    *indexPtr = sbPtr->selectAnchor;
	} else {
	    badIndex:

	    /*
	     * Some of the paths here leave messages in the interp's result,
	     * so we have to clear it out before storing our own message.
	     */

	    Tcl_SetResult(interp, (char *) NULL, TCL_STATIC);
	    Tcl_AppendResult(interp, "bad spinbox index \"", string,
		    "\"", (char *) NULL);
	    return TCL_ERROR;
	}
    } else if (string[0] == 'e') {
	if (strncmp(string, "end", length) == 0) {
	    *indexPtr = sbPtr->numChars;
	} else {
	    goto badIndex;
	}
    } else if (string[0] == 'i') {
	if (strncmp(string, "insert", length) == 0) {
	    *indexPtr = sbPtr->insertPos;
	} else {
	    goto badIndex;
	}
    } else if (string[0] == 's') {
	if (sbPtr->selectFirst < 0) {
	    Tcl_SetResult(interp, "selection isn't in spinbox", TCL_STATIC);
	    return TCL_ERROR;
	}
	if (length < 5) {
	    goto badIndex;
	}
	if (strncmp(string, "sel.first", length) == 0) {
	    *indexPtr = sbPtr->selectFirst;
	} else if (strncmp(string, "sel.last", length) == 0) {
	    *indexPtr = sbPtr->selectLast;
	} else {
	    goto badIndex;
	}
    } else if (string[0] == '@') {
	int x, roundUp;

	if (Tcl_GetInt(interp, string + 1, &x) != TCL_OK) {
	    goto badIndex;
	}
	if (x < sbPtr->inset) {
	    x = sbPtr->inset;
	}
	roundUp = 0;
	if (x >= (Tk_Width(sbPtr->tkwin) - sbPtr->inset - sbPtr->bWidth)) {
	    x = Tk_Width(sbPtr->tkwin) - sbPtr->inset - sbPtr->bWidth - 1;
	    roundUp = 1;
	}
	*indexPtr = Tk_PointToChar(sbPtr->textLayout,
		x - sbPtr->layoutX, 0);

	/*
	 * Special trick:  if the x-position was off-screen to the right,
	 * round the index up to refer to the character just after the
	 * last visible one on the screen.  This is needed to enable the
	 * last character to be selected, for example.
	 */

	if (roundUp && (*indexPtr < sbPtr->numChars)) {
	    *indexPtr += 1;
	}
    } else {
	if (Tcl_GetInt(interp, string, indexPtr) != TCL_OK) {
	    goto badIndex;
	}
	if (*indexPtr < 0){
	    *indexPtr = 0;
	} else if (*indexPtr > sbPtr->numChars) {
	    *indexPtr = sbPtr->numChars;
	} 
    }
    return TCL_OK;
}

/*
 *---------------------------------------------------------------------------
 *
 * GetSpinboxElement --
 *
 *	Return the element associated with an x,y coord.
 *
 * Results:
 *	Element type as enum selelement.
 *
 * Side effects:
 *	None.
 *
 *---------------------------------------------------------------------------
 */

static int
GetSpinboxElement(sbPtr, x, y)
    Spinbox *sbPtr;		/* Spinbox for which the index is being
				 * specified. */
    int x;			/* x coord */
    int y;			/* y coord */
{
    if ((x < 0) || (y < 0) || (y > Tk_Height(sbPtr->tkwin))
	    || (x > Tk_Width(sbPtr->tkwin))) {
	return SEL_NONE;
    }

    if (x > (Tk_Width(sbPtr->tkwin) - sbPtr->inset - sbPtr->bWidth)) {
	if (y > (Tk_Height(sbPtr->tkwin) / 2)) {
	    return SEL_BUTTONDOWN;
	} else {
	    return SEL_BUTTONUP;
	}
    } else {
	return SEL_ENTRY;
    }
    return SEL_ENTRY;
}

/*
 *----------------------------------------------------------------------
 *
 * SpinboxScanTo --
 *
 *	Given a y-coordinate (presumably of the curent mouse location)
 *	drag the view in the window to implement the scan operation.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The view in the window may change.
 *
 *----------------------------------------------------------------------
 */

static void
SpinboxScanTo(sbPtr, x)
    Spinbox *sbPtr;		/* Information about widget. */
    int x;			/* X-coordinate to use for scan operation. */
{
    int newLeftIndex;

    /*
     * Compute new leftIndex for spinbox by amplifying the difference
     * between the current position and the place where the scan
     * started (the "mark" position).  If we run off the left or right
     * side of the spinbox, then reset the mark point so that the current
     * position continues to correspond to the edge of the window.
     * This means that the picture will start dragging as soon as the
     * mouse reverses direction (without this reset, might have to slide
     * mouse a long ways back before the picture starts moving again).
     */

    newLeftIndex = sbPtr->scanMarkIndex
	    - (10 * (x - sbPtr->scanMarkX)) / sbPtr->avgWidth;
    if (newLeftIndex >= sbPtr->numChars) {
	newLeftIndex = sbPtr->scanMarkIndex = sbPtr->numChars - 1;
	sbPtr->scanMarkX = x;
    }
    if (newLeftIndex < 0) {
	newLeftIndex = sbPtr->scanMarkIndex = 0;
	sbPtr->scanMarkX = x;
    } 

    if (newLeftIndex != sbPtr->leftIndex) {
	sbPtr->leftIndex = newLeftIndex;
	sbPtr->flags |= UPDATE_SCROLLBAR;
	SpinboxComputeGeometry(sbPtr);
	if (newLeftIndex != sbPtr->leftIndex) {
	    sbPtr->scanMarkIndex = sbPtr->leftIndex;
	    sbPtr->scanMarkX = x;
	}
	EventuallyRedraw(sbPtr);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * SpinboxSelectTo --
 *
 *	Modify the selection by moving its un-anchored end.  This could
 *	make the selection either larger or smaller.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The selection changes.
 *
 *----------------------------------------------------------------------
 */

static void
SpinboxSelectTo(sbPtr, index)
    Spinbox *sbPtr;		/* Information about widget. */
    int index;			/* Character index of element that is to
				 * become the "other" end of the selection. */
{
    int newFirst, newLast;

    /*
     * Grab the selection if we don't own it already.
     */

    if (!(sbPtr->flags & GOT_SELECTION) && (sbPtr->exportSelection)) {
	Tk_OwnSelection(sbPtr->tkwin, XA_PRIMARY, SpinboxLostSelection,
		(ClientData) sbPtr);
	sbPtr->flags |= GOT_SELECTION;
    }

    /*
     * Pick new starting and ending points for the selection.
     */

    if (sbPtr->selectAnchor > sbPtr->numChars) {
	sbPtr->selectAnchor = sbPtr->numChars;
    }
    if (sbPtr->selectAnchor <= index) {
	newFirst = sbPtr->selectAnchor;
	newLast = index;
    } else {
	newFirst = index;
	newLast = sbPtr->selectAnchor;
	if (newLast < 0) {
	    newFirst = newLast = -1;
	}
    }
    if ((sbPtr->selectFirst == newFirst)
	    && (sbPtr->selectLast == newLast)) {
	return;
    }
    sbPtr->selectFirst = newFirst;
    sbPtr->selectLast = newLast;
    EventuallyRedraw(sbPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * SpinboxFetchSelection --
 *
 *	This procedure is called back by Tk when the selection is
 *	requested by someone.  It returns part or all of the selection
 *	in a buffer provided by the caller.
 *
 * Results:
 *	The return value is the number of non-NULL bytes stored
 *	at buffer.  Buffer is filled (or partially filled) with a
 *	NULL-terminated string containing part or all of the selection,
 *	as given by offset and maxBytes.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
SpinboxFetchSelection(clientData, offset, buffer, maxBytes)
    ClientData clientData;	/* Information about spinbox widget. */
    int offset;			/* Byte offset within selection of first
				 * character to be returned. */
    char *buffer;		/* Location in which to place selection. */
    int maxBytes;		/* Maximum number of bytes to place at
				 * buffer, not including terminating NULL
				 * character. */
{
    Spinbox *sbPtr = (Spinbox *) clientData;
    int byteCount;
    char *string, *selStart, *selEnd;

    if ((sbPtr->selectFirst < 0) || !(sbPtr->exportSelection)) {
	return -1;
    }
    string = sbPtr->displayString;
    selStart = Tcl_UtfAtIndex(string, sbPtr->selectFirst);
    selEnd = Tcl_UtfAtIndex(selStart,
	    sbPtr->selectLast - sbPtr->selectFirst);
    byteCount = selEnd - selStart - offset;
    if (byteCount > maxBytes) {
	byteCount = maxBytes;
    }
    if (byteCount <= 0) {
	return 0;
    }
    memcpy(buffer, selStart + offset, (size_t) byteCount);
    buffer[byteCount] = '\0';
    return byteCount;
}

/*
 *----------------------------------------------------------------------
 *
 * SpinboxLostSelection --
 *
 *	This procedure is called back by Tk when the selection is
 *	grabbed away from an spinbox widget.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The existing selection is unhighlighted, and the window is
 *	marked as not containing a selection.
 *
 *----------------------------------------------------------------------
 */

static void
SpinboxLostSelection(clientData)
    ClientData clientData;	/* Information about spinbox widget. */
{
    Spinbox *sbPtr = (Spinbox *) clientData;

    sbPtr->flags &= ~GOT_SELECTION;

    /*
     * On Windows and Mac systems, we want to remember the selection
     * for the next time the focus enters the window.  On Unix, we need
     * to clear the selection since it is always visible.
     */

#ifdef ALWAYS_SHOW_SELECTION
    if ((sbPtr->selectFirst >= 0) && sbPtr->exportSelection) {
	sbPtr->selectFirst = -1;
	sbPtr->selectLast = -1;
	EventuallyRedraw(sbPtr);
    }
#endif
}

/*
 *----------------------------------------------------------------------
 *
 * EventuallyRedraw --
 *
 *	Ensure that an spinbox is eventually redrawn on the display.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information gets redisplayed.  Right now we don't do selective
 *	redisplays:  the whole window will be redrawn.  This doesn't
 *	seem to hurt performance noticeably, but if it does then this
 *	could be changed.
 *
 *----------------------------------------------------------------------
 */

static void
EventuallyRedraw(sbPtr)
    Spinbox *sbPtr;		/* Information about widget. */
{
    if ((sbPtr->tkwin == NULL) || !Tk_IsMapped(sbPtr->tkwin)) {
	return;
    }

    /*
     * Right now we don't do selective redisplays:  the whole window
     * will be redrawn.  This doesn't seem to hurt performance noticeably,
     * but if it does then this could be changed.
     */

    if (!(sbPtr->flags & REDRAW_PENDING)) {
	sbPtr->flags |= REDRAW_PENDING;
	Tcl_DoWhenIdle(DisplaySpinbox, (ClientData) sbPtr);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * SpinboxVisibleRange --
 *
 *	Return information about the range of the spinbox that is
 *	currently visible.
 *
 * Results:
 *	*firstPtr and *lastPtr are modified to hold fractions between
 *	0 and 1 identifying the range of characters visible in the
 *	spinbox.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
SpinboxVisibleRange(sbPtr, firstPtr, lastPtr)
    Spinbox *sbPtr;		/* Information about widget. */
    double *firstPtr;		/* Return position of first visible
				 * character in widget. */
    double *lastPtr;		/* Return position of char just after last
				 * visible one. */
{
    int charsInWindow;

    if (sbPtr->numChars == 0) {
	*firstPtr = 0.0;
	*lastPtr = 1.0;
    } else {
	charsInWindow = Tk_PointToChar(sbPtr->textLayout,
		Tk_Width(sbPtr->tkwin) - sbPtr->inset - sbPtr->bWidth
			- sbPtr->layoutX - 1, 0);
	if (charsInWindow < sbPtr->numChars) {
	    charsInWindow++;
	}
	charsInWindow -= sbPtr->leftIndex;
	if (charsInWindow == 0) {
	    charsInWindow = 1;
	}

	*firstPtr = (double) sbPtr->leftIndex / sbPtr->numChars;
	*lastPtr = (double) (sbPtr->leftIndex + charsInWindow)
		/ sbPtr->numChars;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * SpinboxUpdateScrollbar --
 *
 *	This procedure is invoked whenever information has changed in
 *	an spinbox in a way that would invalidate a scrollbar display.
 *	If there is an associated scrollbar, then this procedure updates
 *	it by invoking a Tcl command.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	A Tcl command is invoked, and an additional command may be
 *	invoked to process errors in the command.
 *
 *----------------------------------------------------------------------
 */

static void
SpinboxUpdateScrollbar(sbPtr)
    Spinbox *sbPtr;			/* Information about widget. */
{
    char args[TCL_DOUBLE_SPACE * 2];
    int code;
    double first, last;
    Tcl_Interp *interp;

    if (sbPtr->scrollCmd == NULL) {
	return;
    }

    interp = sbPtr->interp;
    Tcl_Preserve((ClientData) interp);
    SpinboxVisibleRange(sbPtr, &first, &last);
    sprintf(args, " %g %g", first, last);
    code = Tcl_VarEval(interp, sbPtr->scrollCmd, args, (char *) NULL);
    if (code != TCL_OK) {
	Tcl_AddErrorInfo(interp,
		"\n    (horizontal scrolling command executed by spinbox)");
	Tcl_BackgroundError(interp);
    }
    Tcl_SetResult(interp, (char *) NULL, TCL_STATIC);
    Tcl_Release((ClientData) interp);
}

/*
 *----------------------------------------------------------------------
 *
 * SpinboxBlinkProc --
 *
 *	This procedure is called as a timer handler to blink the
 *	insertion cursor off and on.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The cursor gets turned on or off, redisplay gets invoked,
 *	and this procedure reschedules itself.
 *
 *----------------------------------------------------------------------
 */

static void
SpinboxBlinkProc(clientData)
    ClientData clientData;	/* Pointer to record describing spinbox. */
{
    Spinbox *sbPtr = (Spinbox *) clientData;

    if ((sbPtr->state != STATE_NORMAL) ||
	    !(sbPtr->flags & GOT_FOCUS) ||
	    (sbPtr->insertOffTime == 0)) {
	return;
    }
    if (sbPtr->flags & CURSOR_ON) {
	sbPtr->flags &= ~CURSOR_ON;
	sbPtr->insertBlinkHandler = Tcl_CreateTimerHandler(
	    sbPtr->insertOffTime, SpinboxBlinkProc, (ClientData) sbPtr);
    } else {
	sbPtr->flags |= CURSOR_ON;
	sbPtr->insertBlinkHandler = Tcl_CreateTimerHandler(
	    sbPtr->insertOnTime, SpinboxBlinkProc, (ClientData) sbPtr);
    }
    EventuallyRedraw(sbPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * SpinboxFocusProc --
 *
 *	This procedure is called whenever the spinbox gets or loses the
 *	input focus.  It's also called whenever the window is reconfigured
 *	while it has the focus.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The cursor gets turned on or off.
 *
 *----------------------------------------------------------------------
 */

static void
SpinboxFocusProc(sbPtr, gotFocus)
    Spinbox *sbPtr;		/* Spinbox that got or lost focus. */
    int gotFocus;		/* 1 means window is getting focus, 0 means
				 * it's losing it. */
{
    Tcl_DeleteTimerHandler(sbPtr->insertBlinkHandler);
    if (gotFocus) {
	sbPtr->flags |= GOT_FOCUS | CURSOR_ON;
	if (sbPtr->insertOffTime != 0) {
	    sbPtr->insertBlinkHandler = Tcl_CreateTimerHandler(
		    sbPtr->insertOnTime, SpinboxBlinkProc,
		    (ClientData) sbPtr);
	}
	if (sbPtr->validate == VALIDATE_ALL ||
	    sbPtr->validate == VALIDATE_FOCUS ||
	    sbPtr->validate == VALIDATE_FOCUSIN) {
	    SpinboxValidateChange(sbPtr, (char *) NULL,
		    sbPtr->string, -1, VALIDATE_FOCUSIN);
	}
    } else {
	sbPtr->flags &= ~(GOT_FOCUS | CURSOR_ON);
	sbPtr->insertBlinkHandler = (Tcl_TimerToken) NULL;
	if (sbPtr->validate == VALIDATE_ALL ||
	    sbPtr->validate == VALIDATE_FOCUS ||
	    sbPtr->validate == VALIDATE_FOCUSOUT) {
	    SpinboxValidateChange(sbPtr, (char *) NULL,
		    sbPtr->string, -1, VALIDATE_FOCUSOUT);
	}
    }
    EventuallyRedraw(sbPtr);
}

/*
 *--------------------------------------------------------------
 *
 * SpinboxTextVarProc --
 *
 *	This procedure is invoked when someone changes the variable
 *	whose contents are to be displayed in an spinbox.
 *
 * Results:
 *	NULL is always returned.
 *
 * Side effects:
 *	The text displayed in the spinbox will change to match the
 *	variable.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static char *
SpinboxTextVarProc(clientData, interp, name1, name2, flags)
    ClientData clientData;	/* Information about spinbox. */
    Tcl_Interp *interp;		/* Interpreter containing variable. */
    char *name1;		/* Not used. */
    char *name2;		/* Not used. */
    int flags;			/* Information about what happened. */
{
    Spinbox *sbPtr = (Spinbox *) clientData;
    char *value;

    /*
     * If the variable is unset, then immediately recreate it unless
     * the whole interpreter is going away.
     */

    if (flags & TCL_TRACE_UNSETS) {
	if ((flags & TCL_TRACE_DESTROYED) && !(flags & TCL_INTERP_DESTROYED)) {
	    Tcl_SetVar(interp, sbPtr->textVarName, sbPtr->string,
		    TCL_GLOBAL_ONLY);
	    Tcl_TraceVar(interp, sbPtr->textVarName,
		    TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		    SpinboxTextVarProc, clientData);
	}
	return (char *) NULL;
    }

    /*
     * Update the spinbox's text with the value of the variable, unless
     * the spinbox already has that value (this happens when the variable
     * changes value because we changed it because someone typed in
     * the spinbox).
     */

    value = Tcl_GetVar(interp, sbPtr->textVarName, TCL_GLOBAL_ONLY);
    if (value == NULL) {
	value = "";
    }
    SpinboxSetValue(sbPtr, value);
    return (char *) NULL;
}

/*
 *--------------------------------------------------------------
 *
 * SpinboxInvoke --
 *
 *	This procedure is invoked when ...
 *
 * Results:
 *	TCL_OK if everything evals ok.
 *      TCL_ERROR if an error occurred while modifying the value.
 *
 * Side effects:
 *      An error condition may arise
 *
 *--------------------------------------------------------------
 */

static int
SpinboxInvoke(interp, sbPtr, element)
    register Tcl_Interp *interp;
    register Spinbox *sbPtr;
    int element;
{
    char *type;
    int code, up;
    Tcl_DString script;

    switch (element) {
	case SEL_BUTTONUP:
	    type = "up";
	    up = 1;
	    break;
	case SEL_BUTTONDOWN:
	    type = "down";
	    up = 0;
	    break;
	default:
	    return TCL_OK;
    }

    if (fabs(sbPtr->increment) > MIN_DBL_VAL) {
	if (sbPtr->listObj != NULL) {
	    Tcl_Obj *objPtr;

	    Tcl_ListObjIndex(interp, sbPtr->listObj, sbPtr->eIndex, &objPtr);
	    if (strcmp(Tcl_GetString(objPtr), sbPtr->string)) {
		/*
		 * Somehow the string changed from what we expected,
		 * so let's do a search on the list to see if the current
		 * value is there.  If not, move to the first element of
		 * the list.
		 */
		int i, listc, elemLen;
		size_t length = sbPtr->numChars;
		char *bytes;
		Tcl_Obj **listv;

		Tcl_ListObjGetElements(interp, sbPtr->listObj, &listc, &listv);
		for (i = 0; i < listc; i++) {
		    bytes = Tcl_GetStringFromObj(listv[i], &elemLen);
		    if ((length == elemLen) &&
			    (memcmp(bytes, sbPtr->string, length) == 0)) {
			sbPtr->eIndex = i;
			break;
		    }
		}
	    }
	    if (up) {
		if (++sbPtr->eIndex >= sbPtr->nElements) {
		    if (sbPtr->wrap) {
			sbPtr->eIndex = 0;
		    } else {
			sbPtr->eIndex = sbPtr->nElements-1;
		    }
		}
	    } else {
		if (--sbPtr->eIndex < 0) {
		    if (sbPtr->wrap) {
			sbPtr->eIndex = sbPtr->nElements-1;
		    } else {
			sbPtr->eIndex = 0;
		    }
		}
	    }
	    Tcl_ListObjIndex(interp, sbPtr->listObj, sbPtr->eIndex, &objPtr);
	    SpinboxValueChanged(sbPtr, Tcl_GetString(objPtr));
	} else if (!DOUBLES_EQ(sbPtr->fromValue, sbPtr->toValue)) {
	    double dvalue;

	    if (Tcl_GetDouble(NULL, sbPtr->string, &dvalue) != TCL_OK) {
		/*
		 * If the string is empty, or isn't a valid double value,
		 * just use the -from value
		 */
		dvalue = sbPtr->fromValue;
	    } else {
		if (up) {
		    dvalue += sbPtr->increment;
		    if (dvalue > sbPtr->toValue) {
			if (sbPtr->wrap) {
			    dvalue = sbPtr->fromValue;
			} else {
			    dvalue = sbPtr->toValue;
			}
		    } else if (dvalue < sbPtr->fromValue) {
			/*
			 * It's possible that when pressing up, we are
			 * still less than the fromValue, because the
			 * user may have manipulated the value by hand.
			 */
			dvalue = sbPtr->fromValue;
		    }
		} else {
		    dvalue -= sbPtr->increment;
		    if (dvalue < sbPtr->fromValue) {
			if (sbPtr->wrap) {
			    dvalue = sbPtr->toValue;
			} else {
			    dvalue = sbPtr->fromValue;
			}
		    } else if (dvalue > sbPtr->toValue) {
			/*
			 * It's possible that when pressing down, we are
			 * still greater than the toValue, because the
			 * user may have manipulated the value by hand.
			 */
			dvalue = sbPtr->toValue;
		    }
		}
	    }
	    sprintf(sbPtr->formatBuf, sbPtr->valueFormat, dvalue);
	    SpinboxValueChanged(sbPtr, sbPtr->formatBuf);
	}
    }

    if (sbPtr->command != NULL) {
	Tcl_DStringInit(&script);
	ExpandPercents(sbPtr, sbPtr->command, type, "", 0,
		VALIDATE_BUTTON, &script);
	Tcl_DStringAppend(&script, "", 1);

	code = Tcl_EvalEx(interp, Tcl_DStringValue(&script), -1,
		TCL_EVAL_GLOBAL | TCL_EVAL_DIRECT);
	Tcl_DStringFree(&script);

	if (code != TCL_OK) {
	    Tcl_AddErrorInfo(interp, "\n\t(in command executed by spinbox)");
	    Tcl_BackgroundError(interp);
	    /*
	     * Yes, it's an error, but a bg one, so we return OK
	     */
	    return TCL_OK;
	}

	Tcl_SetResult(interp, NULL, 0);
    }

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * SpinboxValidate --
 *
 *	This procedure is invoked when any character is added or
 *	removed from the spinbox widget, or a focus has trigerred validation.
 *
 * Results:
 *	TCL_OK if the validatecommand passes the new string.
 *      TCL_BREAK if the vcmd executed OK, but rejects the string.
 *      TCL_ERROR if an error occurred while executing the vcmd
 *      or a valid Tcl_Bool is not returned.
 *
 * Side effects:
 *      An error condition may arise
 *
 *--------------------------------------------------------------
 */

static int
SpinboxValidate(sbPtr, cmd)
     register Spinbox *sbPtr;	/* Spinbox that needs validation. */
     register char *cmd;	/* Validation command (NULL-terminated
				 * string). */
{
    register Tcl_Interp *interp = sbPtr->interp;
    int code, bool;

    code = Tcl_EvalEx(interp, cmd, -1, TCL_EVAL_GLOBAL | TCL_EVAL_DIRECT);

    if (code != TCL_OK && code != TCL_RETURN) {
	Tcl_AddErrorInfo(interp,
		"\n\t(in validation command executed by spinbox)");
	Tcl_BackgroundError(interp);
	return TCL_ERROR;
    }

    if (Tcl_GetBooleanFromObj(interp, Tcl_GetObjResult(interp),
			      &bool) != TCL_OK) {
	Tcl_AddErrorInfo(interp,
		 "\nvalid boolean not returned by validation command");
	Tcl_BackgroundError(interp);
	Tcl_SetResult(interp, NULL, 0);
	return TCL_ERROR;
    }

    Tcl_SetResult(interp, NULL, 0);
    return (bool ? TCL_OK : TCL_BREAK);
}

/*
 *--------------------------------------------------------------
 *
 * SpinboxValidateChange --
 *
 *	This procedure is invoked when any character is added or
 *	removed from the spinbox widget, or a focus has trigerred validation.
 *
 * Results:
 *	TCL_OK if the validatecommand accepts the new string,
 *      TCL_ERROR if any problems occured with validatecommand.
 *
 * Side effects:
 *      The insertion/deletion may be aborted, and the
 *      validatecommand might turn itself off (if an error
 *      or loop condition arises).
 *
 *--------------------------------------------------------------
 */

static int
SpinboxValidateChange(sbPtr, change, new, index, type)
     register Spinbox *sbPtr;	/* Spinbox that needs validation. */
     char *change;		/* Characters to be added/deleted
				 * (NULL-terminated string). */
     char *new;                 /* Potential new value of spinbox string */
     int index;                 /* index of insert/delete, -1 otherwise */
     int type;                  /* forced, delete, insert,
				 * focusin or focusout */
{
    int code, varValidate = (sbPtr->flags & VALIDATE_VAR);
    char *p;
    Tcl_DString script;

    if (sbPtr->validateCmd == NULL ||
	sbPtr->validate == VALIDATE_NONE) {
	return (varValidate ? TCL_ERROR : TCL_OK);
    }

    /*
     * If we're already validating, then we're hitting a loop condition
     * Return and set validate to 0 to disallow further validations
     * and prevent current validation from finishing
     */
    if (sbPtr->flags & VALIDATING) {
	sbPtr->validate = VALIDATE_NONE;
	return (varValidate ? TCL_ERROR : TCL_OK);
    }

    sbPtr->flags |= VALIDATING;

    /*
     * Now form command string and run through the -validatecommand
     */

    Tcl_DStringInit(&script);
    ExpandPercents(sbPtr, sbPtr->validateCmd,
	    change, new, index, type, &script);
    Tcl_DStringAppend(&script, "", 1);

    p = Tcl_DStringValue(&script);
    code = SpinboxValidate(sbPtr, p);
    Tcl_DStringFree(&script);

    /*
     * If e->validate has become VALIDATE_NONE during the validation, or
     * we now have VALIDATE_VAR set (from SpinboxSetValue) and didn't before,
     * it means that a loop condition almost occured.  Do not allow
     * this validation result to finish.
     */
    if (sbPtr->validate == VALIDATE_NONE
	    || (!varValidate && (sbPtr->flags & VALIDATE_VAR))) {
	code = TCL_ERROR;
    }
    /*
     * If validate will return ERROR, then disallow further validations
     * Otherwise, if it didn't accept the new string (returned TCL_BREAK)
     * then eval the invalidCmd (if it's set)
     */
    if (code == TCL_ERROR) {
	sbPtr->validate = VALIDATE_NONE;
    } else if (code == TCL_BREAK) {
	/*
	 * If we were doing forced validation (like via a variable
	 * trace) and the command returned 0, the we turn off validation
	 * because we assume that textvariables have precedence in
	 * managing the value.  We also don't call the invcmd, as it
	 * may want to do spinbox manipulation which the setting of the
	 * var will later wipe anyway.
	 */
	if (varValidate) {
	    sbPtr->validate = VALIDATE_NONE;
	} else if (sbPtr->invalidCmd != NULL) {
	    Tcl_DStringInit(&script);
	    ExpandPercents(sbPtr, sbPtr->invalidCmd,
			   change, new, index, type, &script);
	    Tcl_DStringAppend(&script, "", 1);
	    p = Tcl_DStringValue(&script);
	    if (Tcl_GlobalEval(sbPtr->interp, p) != TCL_OK) {
		Tcl_AddErrorInfo(sbPtr->interp,
			"\n\t(in invalidcommand executed by spinbox)");
		Tcl_BackgroundError(sbPtr->interp);
		code = TCL_ERROR;
		sbPtr->validate = VALIDATE_NONE;
	    }
	    Tcl_DStringFree(&script);
	}
    }

    sbPtr->flags &= ~VALIDATING;

    return code;
}

/*
 *--------------------------------------------------------------
 *
 * ExpandPercents --
 *
 *	Given a command and an event, produce a new command
 *	by replacing % constructs in the original command
 *	with information from the X event.  See user documentation
 *	for details of the substitutions.
 *
 * Results:
 *	The new expanded command is appended to the dynamic string
 *	given by dsPtr.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static void
ExpandPercents(sbPtr, before, change, new, index, type, dsPtr)
     register Spinbox *sbPtr;	/* Spinbox that needs validation. */
     register char *before;	/* Command containing percent
				 * expressions to be replaced. */
     char *change; 		/* Characters to added/deleted
				 * (NULL-terminated string). */
     char *new;                 /* Potential new value of spinbox string */
     int index;                 /* index of insert/delete */
     int type;                  /* INSERT or DELETE */
     Tcl_DString *dsPtr;        /* Dynamic string in which to append
				 * new command. */
{
    int spaceNeeded, cvtFlags;	/* Used to substitute string as proper Tcl
				 * list element. */
    int number, length;
    register char *string;
    Tcl_UniChar ch;
    char numStorage[2*TCL_INTEGER_SPACE];

    while (1) {
	if (*before == '\0') {
	    break;
	}
	/*
	 * Find everything up to the next % character and append it
	 * to the result string.
	 */

	string = before;
	/*
	 * No need to convert '%', as it is in ascii range
	 */
	string = Tcl_UtfFindFirst(before, '%');
	if (string == (char *) NULL) {
	    Tcl_DStringAppend(dsPtr, before, -1);
	    break;
	} else if (string != before) {
	    Tcl_DStringAppend(dsPtr, before, string-before);
	    before = string;
	}

	/*
	 * There's a percent sequence here.  Process it.
	 */

	before++; /* skip over % */
	if (*before != '\0') {
	    before += Tcl_UtfToUniChar(before, &ch);
	} else {
	    ch = '%';
	}
	if (type == VALIDATE_BUTTON) {
	    /*
	     * -command %-substitution
	     */
	    switch (ch) {
		case 's': /* Current string value of spinbox */
		    string = sbPtr->string;
		    break;
		case 'd': /* direction, up or down */
		    string = change;
		    break;
		case 'W': /* widget name */
		    string = Tk_PathName(sbPtr->tkwin);
		    break;
		default:
		    length = Tcl_UniCharToUtf(ch, numStorage);
		    numStorage[length] = '\0';
		    string = numStorage;
		    break;
	    }
	} else {
	    /*
	     * -validatecommand / -invalidcommand %-substitution
	     */
	    switch (ch) {
		case 'd': /* Type of call that caused validation */
		    switch (type) {
			case VALIDATE_INSERT:
			    number = 1;
			    break;
			case VALIDATE_DELETE:
			    number = 0;
			    break;
			default:
			    number = -1;
			    break;
		    }
		    sprintf(numStorage, "%d", number);
		    string = numStorage;
		    break;
		case 'i': /* index of insert/delete */
		    sprintf(numStorage, "%d", index);
		    string = numStorage;
		    break;
		case 'P': /* 'Peeked' new value of the string */
		    string = new;
		    break;
		case 's': /* Current string value of spinbox */
		    string = sbPtr->string;
		    break;
		case 'S': /* string to be inserted/deleted, if any */
		    string = change;
		    break;
		case 'v': /* type of validation currently set */
		    string = validateStrings[sbPtr->validate];
		    break;
		case 'V': /* type of validation in effect */
		    switch (type) {
			case VALIDATE_INSERT:
			case VALIDATE_DELETE:
			    string = validateStrings[VALIDATE_KEY];
			    break;
			case VALIDATE_FORCED:
			    string = "forced";
			    break;
			default:
			    string = validateStrings[type];
			    break;
		    }
		    break;
		case 'W': /* widget name */
		    string = Tk_PathName(sbPtr->tkwin);
		    break;
		default:
		    length = Tcl_UniCharToUtf(ch, numStorage);
		    numStorage[length] = '\0';
		    string = numStorage;
		    break;
	    }
	}

	spaceNeeded = Tcl_ScanCountedElement(string, -1, &cvtFlags);
	length = Tcl_DStringLength(dsPtr);
	Tcl_DStringSetLength(dsPtr, length + spaceNeeded);
	spaceNeeded = Tcl_ConvertCountedElement(string, -1,
		Tcl_DStringValue(dsPtr) + length,
		cvtFlags | TCL_DONT_USE_BRACES);
	Tcl_DStringSetLength(dsPtr, length + spaceNeeded);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ComputeFormat --
 *
 *	This procedure is invoked to recompute the "format" fields
 *	of a spinbox's widget record, which determines how the value
 *	of the dial is converted to a string.
 *
 * Results:
 *	Tcl result code.
 *
 * Side effects:
 *	The format fields of the spinbox are modified.
 *
 *----------------------------------------------------------------------
 */
static int
ComputeFormat(sbPtr)
     Spinbox *sbPtr;			/* Information about dial widget. */
{
    double maxValue, x;
    int mostSigDigit, numDigits, leastSigDigit, afterDecimal;
    int eDigits, fDigits;

    /*
     * Compute the displacement from the decimal of the most significant
     * digit required for any number in the dial's range.
     */

    if (sbPtr->reqFormat) {
	sbPtr->valueFormat = sbPtr->reqFormat;
	return TCL_OK;
    }

    maxValue = fabs(sbPtr->fromValue);
    x = fabs(sbPtr->toValue);
    if (x > maxValue) {
	maxValue = x;
    }
    if (maxValue == 0) {
	maxValue = 1;
    }
    mostSigDigit = (int) floor(log10(maxValue));

    if (fabs(sbPtr->increment) > MIN_DBL_VAL) {
	/*
	 * A increment was specified, so use it.
	 */
	leastSigDigit = (int) floor(log10(sbPtr->increment));
    } else {
	leastSigDigit = 0;
    }
    numDigits = mostSigDigit - leastSigDigit + 1;
    if (numDigits < 1) {
	numDigits = 1;
    }

    /*
     * Compute the number of characters required using "e" format and
     * "f" format, and then choose whichever one takes fewer characters.
     */

    eDigits = numDigits + 4;
    if (numDigits > 1) {
	eDigits++;			/* Decimal point. */
    }
    afterDecimal = numDigits - mostSigDigit - 1;
    if (afterDecimal < 0) {
	afterDecimal = 0;
    }
    fDigits = (mostSigDigit >= 0) ? mostSigDigit + afterDecimal : afterDecimal;
    if (afterDecimal > 0) {
	fDigits++;			/* Decimal point. */
    }
    if (mostSigDigit < 0) {
	fDigits++;			/* Zero to left of decimal point. */
    }
    if (fDigits <= eDigits) {
	sprintf(sbPtr->digitFormat, "%%.%df", afterDecimal);
    } else {
	sprintf(sbPtr->digitFormat, "%%.%de", numDigits-1);
    }
    sbPtr->valueFormat = sbPtr->digitFormat;
    return TCL_OK;
}
