/* 
 * tkStripchart.c --
 *
 *	This module implements "Stripchart" widgets for the Tk
 *	toolkit.
 *
 * Copyright 1992 Regents of the University of Victoria, Wellington, NZ.
 * This code is derived from the tkScale widget.
 * Copyright 1990 Regents of the University of California.
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.	 The University of California
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

/*
 *  BUGS:
 *   sometimes the callback procedure doesn't work (????)
 *    (it can often be made to work by setting a (different) value and
 *     restarting the callback procedure.)
 *
 *  CHANGES:
 *   would be nicer having tick interval instead of numticks.
 *
 */

#include "pkg.h"

/* Default stripchart configuration values */

#define DEF_STRIPCHART_BG_COLOR		NORMAL_BG
#define DEF_STRIPCHART_STRIP_COLOR	"red"
#define DEF_STRIPCHART_TEXT_COLOR	"black"
#define DEF_STRIPCHART_TICK_COLOR	"blue2"

#define ALT_STRIPCHART_BG_COLOR		"red"
#define ALT_STRIPCHART_STRIP_COLOR	"green"
#define ALT_STRIPCHART_TEXT_COLOR	"white"
#define ALT_STRIPCHART_TICK_COLOR	"gray70"

/* for monochrome displays */

#define DEF_STRIPCHART_BG_MONO		"white"
#define DEF_STRIPCHART_STRIP_MONO	"black"
#define DEF_STRIPCHART_TEXT_MONO	"black"
#define DEF_STRIPCHART_TICK_MONO	"black"

#define ALT_STRIPCHART_BG_MONO		"black"
#define ALT_STRIPCHART_STRIP_MONO	"white"
#define ALT_STRIPCHART_TEXT_MONO	"white"
#define ALT_STRIPCHART_TICK_MONO	"white"

#define DEF_STRIPCHART_NUMSTRIPS	"40"
#define DEF_STRIPCHART_NUMTICKS		"11"
#define DEF_STRIPCHART_BORDER_WIDTH	"0"
#define DEF_STRIPCHART_FONT		DEF_FONT
#define DEF_STRIPCHART_MINVALUE		"0"
#define DEF_STRIPCHART_MAXVALUE		"100"
#define DEF_STRIPCHART_CALLBACK_INTERVAL "500"
#define DEF_STRIPCHART_TITLE		(char *)NULL
#define DEF_STRIPCHART_HEIGHT		"80"
#define DEF_STRIPCHART_RELIEF		"raised"
#define DEF_STRIPCHART_TICK_INTERVAL	"50"
#define DEF_STRIPCHART_WIDTH		"2"
#define DEF_STRIPCHART_STRIPBORDERWIDTH	"0"
#define DEF_STRIPCHART_STRIPRELIEF	"raised"
#define DEF_STRIPCHART_SHOWTICKS	"true"
#define DEF_USERBITS			"0"
#define DEF_USERDATA			(char *) NULL
#define DEF_GUARANTEE_DRAW		"false"

#define MAX_STRIPS	100
#define MAX_TICKS	40
#define PADDING		2

#define hasatitle(d)	((d)->title != NULL)

/*
 * A data structure of the following type is kept for each
 * Stripchart that currently exists for this process:
 */

typedef struct {
  Tk_Window tkwin;		/* Window that embodies the Stripchart.	 NULL
				 * means that the window has been destroyed
				 * but the data structures haven't yet been
				 * cleaned up.*/
  Display *display;
  Tcl_Interp *interp;		/* Interpreter associated with
				 * widget.  Used to delete widget
				 * command.  */
  Tk_Uid screenName;		/* If this window isn't a toplevel window
				 * then this is NULL;  otherwise it gives
				 * the name of the screen on which window
				 * is displayed. */
  Tk_3DBorder border;		/* Structure used to draw 3-D border and
				 * background. */
  int borderWidth;		/* Width of 3-D border (if any). */
  int relief;			/* 3-d effect: TK_RELIEF_RAISED etc. */
  Cursor cursor;		/* Cursor for widget */
    
  /* strip stuff */
  Tk_3DBorder stripBorder;	/* Structure used to draw the strips */
  int stripBorderWidth;
  int stripRelief;
  int strip_width;		/* width of a strip */
  int max_height;		/* maximum height of a strip */
  int min_value;
  int max_value;		/* maximum value of a strip */
  int num_strips;		/* the number of strips */
  int num_ticks;		/* the number of ticks to display */
  int value[MAX_STRIPS];	/* the data to be displayed in strip form. */

  int stripstodisplay;		/* how many of the num_strips should be
				 * displayed. */
  int scrollrequired;
  int guarentee_draw;

  int grow_up;
  Tk_Font tkfont;		/* Information about text font, or NULL. */
  XColor *textColorPtr;		/* Color for drawing text. */
  GC textGC;			/* GC for drawing text. */
  XColor *tickColorPtr;		/* Color for drawing ticks. */
  GC tickGC;			/* GC for drawing ticks. */
  int showticks;

  /* call back stuff */
  char *command;		/* Command used for callback. Malloc'ed. */
  int continue_callback;	/* boolean flag used to terminate the
				 * callback */
  unsigned long interval;	/* interval (mS) between callbacks */
  char *title;			/* Label to display above of stripchart;
				 * NULL means don't display a
				 * title. Malloc'ed. */

  int displaybits;		/* Various flags; see below for definitions */
  int userbits;
  char *userdata;
  int cb_key;			/* callback data key */
  int cb_loc;			/* callback data location */
  Tcl_TimerToken cb_timer;	/* Token for callback timer. */
    
  /* Alternative colour scheme */

  Tk_3DBorder altborder;	  /* Structure used to draw border  */
  Tk_3DBorder altstripBorder;	 /* structure used to draw strip  */
  XColor *a_textColor;	  /* alternative text color	    */
  XColor *a_tickColor;	  /* alternative tick color	    */

} Stripchart;

/*
 * displaybits for Stripcharts:
 *
 * REDRAW_PENDING:		Non-zero means a DoWhenIdle handler
 *				has already been queued to redraw
 *				this window.
 * CLEAR_NEEDED:		Need to clear the window when redrawing.
 * DISPLAY_TITLE:		Draw the title, if there is one.
 * DISPLAY_STRIPS:		Draw the strips.
 * DISPLAY_BORDER:		Draw the border.
 * DISPLAY_TICKS:		Draw the ticks.
 * DISPLAY_ALL:			Display everything.
 */

#define	  REDRAW_PENDING		1
#define	  CLEAR_NEEDED			2
#define	  DISPLAY_TITLE			4
#define	  DISPLAY_STRIPS		8
#define	  DISPLAY_BORDER		16
#define	  DISPLAY_TICKS			32
#define	  DISPLAY_ALL			(DISPLAY_TITLE | DISPLAY_STRIPS | DISPLAY_BORDER | DISPLAY_TICKS)

static Tk_ConfigSpec configSpecs[] = {
  {TK_CONFIG_BORDER, "-background", "background", "Background",
   DEF_STRIPCHART_BG_COLOR, Tk_Offset(Stripchart, border),
   TK_CONFIG_COLOR_ONLY},
  {TK_CONFIG_BORDER, "-background", "background", "Background",
   DEF_STRIPCHART_BG_MONO, Tk_Offset(Stripchart, border),
   TK_CONFIG_MONO_ONLY},
  {TK_CONFIG_BORDER, "-stripcolor", "stripcolor", "Stripcolor",
   DEF_STRIPCHART_STRIP_COLOR, Tk_Offset(Stripchart, stripBorder),
   TK_CONFIG_COLOR_ONLY},
  {TK_CONFIG_BORDER, "-stripcolor", "stripcolor", "Stripcolor",
   DEF_STRIPCHART_STRIP_MONO, Tk_Offset(Stripchart, stripBorder),
   TK_CONFIG_MONO_ONLY},
  {TK_CONFIG_SYNONYM, "-bd", "borderWidth", (char *) NULL,
   (char *) NULL, 0, 0},
  {TK_CONFIG_SYNONYM, "-bg", "background", (char *) NULL,
   (char *) NULL, 0, 0},
  {TK_CONFIG_SYNONYM, "-fg", "stripcolor", (char *) NULL,
   (char *) NULL, 0, 0},
  {TK_CONFIG_SYNONYM, "-stripcolour", "stripcolor", (char *) NULL,
   (char *) NULL, 0, 0},
  {TK_CONFIG_SYNONYM, "-textcolour", "textcolor", (char *) NULL,
   (char *) NULL, 0, 0},
  {TK_CONFIG_INT, "-stripborderwidth", "stripborderwidth",
   "Stripborderwidth", DEF_STRIPCHART_STRIPBORDERWIDTH,
   Tk_Offset(Stripchart, stripBorderWidth), 0},
  {TK_CONFIG_STRING, "-command", "command", "Command",
   (char *) NULL, Tk_Offset(Stripchart, command), 0},
  {TK_CONFIG_INT, "-interval", "interval", "Interval",
   DEF_STRIPCHART_CALLBACK_INTERVAL, Tk_Offset(Stripchart, interval), 0},
  {TK_CONFIG_BOOLEAN, "-up", "up", "Up", "true",
   Tk_Offset(Stripchart, grow_up), 0},
  {TK_CONFIG_FONT, "-font", "font", "Font",
   DEF_STRIPCHART_FONT, Tk_Offset(Stripchart, tkfont), 0},
  {TK_CONFIG_COLOR, "-textcolor", "textcolor", "Textcolor",
   DEF_STRIPCHART_TEXT_COLOR, Tk_Offset(Stripchart, textColorPtr),
   TK_CONFIG_COLOR_ONLY},
  {TK_CONFIG_COLOR, "-textcolor", "textcolor", "Textcolor",
   DEF_STRIPCHART_TEXT_MONO, Tk_Offset(Stripchart, textColorPtr),
   TK_CONFIG_MONO_ONLY},
  {TK_CONFIG_COLOR, "-tickcolor", "tickcolor", "Tickcolor",
   DEF_STRIPCHART_TICK_COLOR, Tk_Offset(Stripchart, tickColorPtr),
   TK_CONFIG_COLOR_ONLY},
  {TK_CONFIG_COLOR, "-tickcolor", "tickcolor", "Tickcolor",
   DEF_STRIPCHART_TICK_MONO, Tk_Offset(Stripchart, tickColorPtr),
   TK_CONFIG_MONO_ONLY},

  {TK_CONFIG_ACTIVE_CURSOR, "-cursor", "cursor", "Cursor",
   (char *)NULL, Tk_Offset(Stripchart, cursor), TK_CONFIG_NULL_OK},

  {TK_CONFIG_BORDER, "-altbackground", "altbackground", "Background",
   ALT_STRIPCHART_BG_COLOR, Tk_Offset(Stripchart, altborder),
   TK_CONFIG_COLOR_ONLY},
  {TK_CONFIG_BORDER, "-altbackground", "altbackground", "Background",
   ALT_STRIPCHART_BG_MONO, Tk_Offset(Stripchart, altborder),
   TK_CONFIG_MONO_ONLY},
  {TK_CONFIG_BORDER, "-altstripcolor", "altstripcolor",
   "Foreground", ALT_STRIPCHART_STRIP_COLOR,
   Tk_Offset(Stripchart, altstripBorder), TK_CONFIG_COLOR_ONLY},
  {TK_CONFIG_BORDER, "-altstripcolor", "altstripcolor",
   "Foreground", ALT_STRIPCHART_STRIP_MONO,
   Tk_Offset(Stripchart, altstripBorder), TK_CONFIG_MONO_ONLY},
  {TK_CONFIG_COLOR, "-alttickcolor", "tickcolor", "Foreground",
   ALT_STRIPCHART_TICK_COLOR, Tk_Offset(Stripchart, a_tickColor),
   TK_CONFIG_COLOR_ONLY},
  {TK_CONFIG_COLOR, "-alttickcolor", "tickcolor", "Foreground",
   ALT_STRIPCHART_TICK_MONO, Tk_Offset(Stripchart, a_tickColor),
   TK_CONFIG_MONO_ONLY},
  {TK_CONFIG_COLOR, "-alttextcolor", "textcolor", "Foreground",
   ALT_STRIPCHART_TEXT_COLOR, Tk_Offset(Stripchart, a_textColor),
   TK_CONFIG_COLOR_ONLY},
  {TK_CONFIG_COLOR, "-alttextcolor", "textcolor", "Foreground",
   ALT_STRIPCHART_TEXT_MONO, Tk_Offset(Stripchart, a_textColor),
   TK_CONFIG_MONO_ONLY},
  {TK_CONFIG_SYNONYM, "-alttickcolour", "alttickcolor", (char *) NULL,
   (char *) NULL, 0, 0},
  {TK_CONFIG_SYNONYM, "-alttc", "alttickcolor", (char *)NULL,
   (char *)NULL, 0, 0},
  {TK_CONFIG_SYNONYM, "-altstripcolour", "altstripcolor",
   (char *) NULL, (char *) NULL, 0, 0},
  {TK_CONFIG_SYNONYM, "-altsc", "altstripcolor", (char *)NULL,
   (char *)NULL, 0, 0},
  {TK_CONFIG_SYNONYM, "-altbg", "altbackground", (char *)NULL,
   (char *)NULL, 0, 0},
  {TK_CONFIG_SYNONYM, "-altfg", "altstripcolor", (char *)NULL,
   (char *)NULL, 0, 0},
  {TK_CONFIG_SYNONYM, "-alttextcolour", "alttextcolor", (char *)NULL,
   (char *)NULL, 0, 0},

  {TK_CONFIG_BOOLEAN, "-showticks", "showticks", "Showticks",
   DEF_STRIPCHART_SHOWTICKS, Tk_Offset(Stripchart, showticks), 0},
  {TK_CONFIG_BOOLEAN, "-guaranteedrawing", "guaranteedrawing",
   "Guaranteedrawing", DEF_GUARANTEE_DRAW,
   Tk_Offset(Stripchart, guarentee_draw), 0},
  {TK_CONFIG_INT, "-min", "min", "Min",
   DEF_STRIPCHART_MINVALUE, Tk_Offset(Stripchart, min_value), 0},
  {TK_CONFIG_INT, "-max", "max", "Max",
   DEF_STRIPCHART_MAXVALUE, Tk_Offset(Stripchart, max_value), 0},
  {TK_CONFIG_INT, "-numstrips", "numstrips", "Numstrips",
   DEF_STRIPCHART_NUMSTRIPS, Tk_Offset(Stripchart, num_strips), 0},
  {TK_CONFIG_STRING, "-title", "title", "Title",
   DEF_STRIPCHART_TITLE, Tk_Offset(Stripchart, title), TK_CONFIG_NULL_OK},
  {TK_CONFIG_INT, "-width", "width", "Width", DEF_STRIPCHART_WIDTH,
   Tk_Offset(Stripchart, strip_width), 0},
  {TK_CONFIG_INT, "-height", "height", "Height", DEF_STRIPCHART_HEIGHT,
   Tk_Offset(Stripchart, max_height), 0},
  {TK_CONFIG_INT, "-borderwidth", "borderWidth", "BorderWidth",
   DEF_STRIPCHART_BORDER_WIDTH, Tk_Offset(Stripchart, borderWidth), 0},
  {TK_CONFIG_INT, "-stripwidth", "stripwidth", "Stripwidth",
   DEF_STRIPCHART_WIDTH, Tk_Offset(Stripchart, strip_width), 0},
  {TK_CONFIG_INT, "-numticks", "numticks", "Numticks",
   DEF_STRIPCHART_NUMTICKS, Tk_Offset(Stripchart, num_ticks), 0},
  {TK_CONFIG_RELIEF, "-relief", "relief", "Relief",
   DEF_STRIPCHART_RELIEF, Tk_Offset(Stripchart, relief), 0},
  {TK_CONFIG_RELIEF, "-striprelief", "striprelief", "Striprelief",
   DEF_STRIPCHART_STRIPRELIEF, Tk_Offset(Stripchart, stripRelief), 0},
  {TK_CONFIG_STRING, "-data", "data", "Data",
   DEF_USERDATA, Tk_Offset(Stripchart, userdata), 0},
  {TK_CONFIG_INT, "-userbits", "userbits", "Userbits",
   DEF_USERBITS, Tk_Offset(Stripchart, userbits), 0},
  {TK_CONFIG_INT, "-cbloc","cbloc","Cbloc",
   "-1", Tk_Offset(Stripchart, cb_loc),0},
  {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
   (char *) NULL, 0, 0}
};

/*
 * Forward declarations for procedures defined later in this file:
 */

static void	Callback _ANSI_ARGS_((Stripchart *scPtr));
static void	ComputeStripchartGeometry _ANSI_ARGS_((Stripchart *scPtr));
static int	ConfigureStripchart _ANSI_ARGS_((Tcl_Interp *interp,
						 Stripchart *scPtr, int argc, char **argv,
						 int flags));
static void	DestroyStripchart _ANSI_ARGS_((ClientData clientData));
static void	DisplayStripchart _ANSI_ARGS_((ClientData clientData));
static void	DrawStripi _ANSI_ARGS_((Stripchart *scPtr, int i));
static void	EventuallyRedrawStripchart _ANSI_ARGS_((Stripchart *scPtr, int displaybits));
static void	ReplaceColours _ANSI_ARGS_((Stripchart *scPtr,
					    int argc, char *argv[]));
static void	SetStripchartValue _ANSI_ARGS_((Stripchart *scPtr,
						int value));
static void	ScrollStrips _ANSI_ARGS_((Stripchart *scPtr));
static void	StripchartEventProc _ANSI_ARGS_((ClientData clientData,
						 XEvent *eventPtr));
static int	StripchartWidgetCmd _ANSI_ARGS_((ClientData clientData,
						 Tcl_Interp *interp, int argc, char **argv));
static void	SetStripchartValue _ANSI_ARGS_((Stripchart *scPtr,
						int value));
static void	SwapColours _ANSI_ARGS_((Stripchart *scPtr));

/*
 *--------------------------------------------------------------
 *
 * Tk_StripchartCmd --
 *
 *	This procedure is invoked to process the "Stripchart" and
 *	"toplevel" Tcl commands.  See the user documentation for
 *	details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

int
Tk_StripchartCmd(clientData, interp, argc, argv)
     ClientData clientData;	/* Main window associated with
				 * interpreter. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int argc;			/* Number of arguments. */
     char **argv;		/* Argument strings. */
{
  Tk_Window tkwin = (Tk_Window) clientData;
  Tk_Window new;
  register Stripchart *scPtr;


  if (argc < 2) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
		     argv[0], " pathName ?options?\"", (char *) NULL);
    return TCL_ERROR;
  }

  /*
   * Create the window.
   */

  new = Tk_CreateWindowFromPath(interp, tkwin, argv[1], (char *)NULL);
  if (new == NULL) {
    return TCL_ERROR;
  }

  scPtr = (Stripchart *) ckalloc(sizeof(Stripchart));
  scPtr->tkwin = new;
  scPtr->display = Tk_Display(new);
  scPtr->interp = interp;
  scPtr->border = NULL;
  scPtr->stripBorder = NULL;
  scPtr->tkfont = NULL;
  scPtr->textColorPtr = NULL;
  scPtr->textGC = None;
  scPtr->tickColorPtr = NULL;
  scPtr->tickGC = None;
  scPtr->cursor = None;
    
  scPtr->altborder = NULL;
  scPtr->altstripBorder = NULL;
  scPtr->a_textColor = NULL;
  scPtr->a_tickColor = NULL;

  scPtr->command = NULL;
  scPtr->userdata = NULL;
  scPtr->cb_loc = -1;
  scPtr->cb_key = 0;
  scPtr->cb_timer = NULL;
  scPtr->continue_callback = FALSE;
    
    
  scPtr->title = NULL;
  scPtr->stripstodisplay = 0;
  scPtr->scrollrequired = 0;
  scPtr->displaybits = 0;

  Tk_SetClass(new, "Stripchart");
  Tk_CreateEventHandler(scPtr->tkwin,
			ExposureMask|StructureNotifyMask,
			StripchartEventProc, (ClientData) scPtr);
  Tcl_CreateCommand(interp, Tk_PathName(scPtr->tkwin), StripchartWidgetCmd,
		    (ClientData) scPtr, (Tcl_CmdDeleteProc *) NULL);

  if (ConfigureStripchart(interp, scPtr, argc-2, argv+2, 0) != TCL_OK) {
    Tk_DestroyWindow(new);
    return TCL_ERROR;
  }

  Tcl_SetResult(interp, Tk_PathName(scPtr->tkwin), TCL_STATIC);
  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * StripchartWidgetCmd --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to a Stripchart widget.  See the user
 *	documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

static int
StripchartWidgetCmd(clientData, interp, argc, argv)
     ClientData clientData;	/* Information about Stripchart widget. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int argc;			/* Number of arguments. */
     char **argv;		/* Argument strings. */
{
  register Stripchart *scPtr = (Stripchart *) clientData;
  int result = TCL_OK;
  int length;
  char c;
  
  if (argc < 2) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
		     argv[0], " option ?arg arg ...?\"", (char *) NULL);
    return TCL_ERROR;
  }
  Tcl_Preserve((ClientData) scPtr);
  c = argv[1][0];
  length = strlen(argv[1]);
  if ((c == 'c') && (strncmp(argv[1], "cget", length) == 0)
      && (length >= 2)) {
    if (argc != 3) {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " cget option\"", (char *) NULL);
      goto error;
    }
    result = Tk_ConfigureValue(interp, scPtr->tkwin, configSpecs,
			       (char *) scPtr, argv[2], 0);
  } else if ((c == 'c') && (strncmp(argv[1], "configure", length) == 0)
      && (length >= 2)) {
    if (argc == 2) {
      result = Tk_ConfigureInfo(interp, scPtr->tkwin, configSpecs,
				(char *) scPtr, (char *) NULL, 0);
    } else if (argc == 3) {
      result = Tk_ConfigureInfo(interp, scPtr->tkwin, configSpecs,
				(char *) scPtr, argv[2], 0);
    } else {
      result = ConfigureStripchart(interp, scPtr, argc-2, argv+2,
				   TK_CONFIG_ARGV_ONLY);
    }
  } else if ((c == 'g') && (strncmp(argv[1], "get", length) == 0)) {
    if (argc != 2) {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " get\"", (char *) NULL);
      goto error;
    }
    sprintf(interp->result, "%d",
	    scPtr->value[scPtr->stripstodisplay-1]);
  } else if ((c == 's') && (strncmp(argv[1], "set", length) == 0)) {
    int value;

    if (argc != 3) {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " set value\"", (char *) NULL);
      goto error;
    }
    if (Tcl_GetInt(interp, argv[2], &value) != TCL_OK) {
      goto error;
    }
    SetStripchartValue(scPtr, value);
  } else if ((c == 's') && (strncmp(argv[1], "start", length) == 0)) {
    if (! scPtr->continue_callback ) {
      scPtr->continue_callback = TRUE;
      Callback(scPtr);
    }
  } else if ((c == 's') && (strncmp(argv[1], "stop", length) == 0)) {
    scPtr->continue_callback = FALSE;
    if (scPtr->cb_timer) {
      Tk_DeleteTimerHandler(scPtr->cb_timer);
      scPtr->cb_timer = NULL;
    }

  } else if ((c == 'r') && (strncmp(argv[1], "reset", length) == 0)) {
    scPtr->stripstodisplay = 0;
    scPtr->scrollrequired = FALSE;
    EventuallyRedrawStripchart(scPtr, DISPLAY_ALL | CLEAR_NEEDED);
  } else if ((c == 's') && (strncmp(argv[1], "swapcolours", length) == 0)) {
    SwapColours(scPtr);
  } else if ((c == 'r') && (strncmp(argv[1],"replacecolours",length) == 0)) {
    ReplaceColours(scPtr, argc-2, argv+2);
  } else {
    Tcl_AppendResult(interp, "bad option \"", argv[1],
		     "\":  must be configure, get, set, reset, start or stop",
		     (char *) NULL);
    goto error;
  }

  Tcl_Release((ClientData) scPtr);
  return result;

error:
  Tcl_Release((ClientData) scPtr);
  return TCL_ERROR;
}

/*
 *------------------------------------------------------------------------
 *
 *  Callback --
 *	Execute a Tcl command repeatedly until told to stop.  Involked
 *	with the start command and stopped with the stop command.
 *
 *  Results:
 *	None.
 *
 *  Side Effects:
 *	Timer queue is changed with the addition of a command to be
 *	executed periodically.
 *
 *------------------------------------------------------------------------
 */

static void Callback(scPtr)
     Stripchart *scPtr;
{
  int no_errors = TRUE;
  int result;

  if ( scPtr->command != NULL && scPtr->command[0] != '\0' ) {
    result = Tcl_Eval(scPtr->interp, scPtr->command);
    if ( result != TCL_OK ) {
      Tk_BackgroundError(scPtr->interp);
      no_errors = FALSE;
    }
  }

#ifdef CR
  SetStripchartValue(scPtr, cbval(scPtr->userdata, &(scPtr->cb_loc)));
#endif

  if ( scPtr->continue_callback && no_errors )
    scPtr->cb_timer =
      Tk_CreateTimerHandler(scPtr->interval,
			    (Tcl_TimerProc *)Callback,
			    (ClientData)scPtr );

  return;
}

/*
 *----------------------------------------------------------------------
 *
 * DestroyStripchart --
 *
 *	This procedure is invoked by Tcl_EventuallyFree or Tcl_Release
 *	to clean up the internal structure of a Stripchart at a safe time
 *	(when no-one is using it anymore).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the Stripchart is freed up.
 *
 *----------------------------------------------------------------------
 */

static void
DestroyStripchart(clientData)
     ClientData clientData;	/* Info about Stripchart widget. */
{
  register Stripchart *scPtr = (Stripchart *) clientData;

  if (scPtr->value != NULL)
    free(scPtr->value);

  if ((scPtr->textGC != None) && scPtr->tkwin)
    Tk_FreeGC(scPtr->display, scPtr->textGC);

  if ((scPtr->tickGC != None) && scPtr->tkwin)
    Tk_FreeGC(scPtr->display, scPtr->tickGC);

  if (scPtr->cb_timer) 
    Tk_DeleteTimerHandler(scPtr->cb_timer);

  scPtr->cb_timer = NULL;
  scPtr->continue_callback = FALSE;

  /* free the configuration options in the widget */
  Tk_FreeOptions(configSpecs, (char *) scPtr, scPtr->display, 0);
    
  ckfree((char *) scPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * ConfigureStripchart --
 *
 *	This procedure is called to process an argv/argc list, plus
 *	the Tk option database, in order to configure (or
 *	reconfigure) a Stripchart widget.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as text string, colors, font,
 *	etc. get set for scPtr;	 old resources get freed, if there
 *	were any.
 *
 *----------------------------------------------------------------------
 */

static int
ConfigureStripchart(interp, scPtr, argc, argv, flags)
     Tcl_Interp *interp;		/* Used for error reporting. */
     register Stripchart *scPtr;  /* Information about widget. */
     int argc;			/* Number of valid entries in argv. */
     char **argv;		/* Arguments. */
     int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
  XGCValues gcValues;
  GC newGC;

  if (Tk_ConfigureWidget(interp, scPtr->tkwin, configSpecs,
			 argc, argv, (char *) scPtr, flags) != TCL_OK)
    return TCL_ERROR;

  Tk_SetBackgroundFromBorder(scPtr->tkwin, scPtr->border);

  gcValues.font = Tk_FontId(scPtr->tkfont);
  gcValues.foreground = scPtr->textColorPtr->pixel;
  newGC = Tk_GetGC(scPtr->tkwin, GCForeground|GCFont, &gcValues);
  if (scPtr->textGC != None) {
    Tk_FreeGC(scPtr->display,scPtr->textGC);
  }
  scPtr->textGC = newGC;

  gcValues.foreground = scPtr->tickColorPtr->pixel;
  newGC = Tk_GetGC(scPtr->tkwin, GCForeground, &gcValues);
  if (scPtr->tickGC != None) {
    Tk_FreeGC(scPtr->display,scPtr->tickGC);
  }
  scPtr->tickGC = newGC;

  if (scPtr->num_strips > MAX_STRIPS)
    scPtr->num_strips = MAX_STRIPS;

  if (scPtr->num_ticks > MAX_TICKS)
    scPtr->num_ticks = MAX_TICKS;

  if (scPtr->min_value > scPtr->max_value) {
    int temp = scPtr->min_value;
    scPtr->min_value = scPtr->max_value;
    scPtr->max_value = temp;
  }

  /*
   * Recompute display-related information, and let the geometry
   * manager know how much space is needed now.
   */
  ComputeStripchartGeometry(scPtr);

  EventuallyRedrawStripchart(scPtr, DISPLAY_ALL | CLEAR_NEEDED);

  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ComputeStripchartGeometry --
 *
 *	This procedure is called to compute various geometrical
 *	information for a stripchart, such as where various things get
 *	displayed.  It's called when the window is reconfigured.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
ComputeStripchartGeometry(scPtr)
     register Stripchart *scPtr;	     /* Information about widget. */
{
  int tt = hasatitle(scPtr);
  int bd = scPtr->borderWidth;
  int lineheight;
  Tk_FontMetrics fm;

  Tk_GetFontMetrics(scPtr->tkfont, &fm);
  lineheight = fm.ascent + fm.descent;

  Tk_GeometryRequest(scPtr->tkwin, /* width */
		     2*(bd + PADDING) + scPtr->num_strips * scPtr->strip_width,
		     /* height */
		     2*(bd + PADDING) + tt*(lineheight + PADDING) +
		     scPtr->max_height );

  Tk_SetInternalBorder(scPtr->tkwin, scPtr->borderWidth);
}

/*
 *----------------------------------------------------------------------
 *
 * DisplayStripchart --
 *
 *	This procedure is invoked to display a Stripchart widget.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Commands are output to X to display the Stripchart in its
 *	current mode.
 *
 *----------------------------------------------------------------------
 */

static void
DisplayStripchart(clientData)
     ClientData clientData;	/* Information about widget. */
{
  register Stripchart *scPtr = (Stripchart *) clientData;
  register Tk_Window tkwin = scPtr->tkwin;
  int bd = scPtr->borderWidth;
  int i, tt = hasatitle(scPtr);

  /* Variable declarations used in the title drawing routines */
  int x;
  int lineheight;
  Tk_FontMetrics fm;

  Tk_GetFontMetrics(scPtr->tkfont, &fm);
  lineheight = fm.ascent + fm.descent;

  scPtr->displaybits &= ~REDRAW_PENDING;
  if ((scPtr->tkwin == NULL) || !Tk_IsMapped(tkwin))
    return;

  /* Clear the window if necessary */
  if ( scPtr->displaybits & CLEAR_NEEDED )
    XClearWindow( Tk_Display(tkwin), Tk_WindowId(tkwin) );

  /*
   *  Display the title, centered in the window if there is enough space.
   *  Otherwise left justified and clipped on the right.
   */
  if (tt && scPtr->displaybits & DISPLAY_TITLE ) {
    int width = Tk_TextWidth(scPtr->tkfont, scPtr->title,
			     strlen(scPtr->title));
    if (width < Tk_Width(tkwin) - 2*bd )
      x = (Tk_Width(tkwin) - width) / 2;
    else
      x = bd + PADDING;

#ifndef WIN32
    XClearArea(Tk_Display(tkwin), Tk_WindowId(tkwin), bd, bd,
	       Tk_Width(tkwin) - 2*bd, lineheight + PADDING, False);
#else
    Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin),
		       scPtr->border, 0, 0,
		       Tk_Width(tkwin) - 2*bd, lineheight + PADDING,
		       0, TK_RELIEF_FLAT);
#endif
    Tk_DrawChars(Tk_Display(tkwin), Tk_WindowId(tkwin),
		 scPtr->textGC, scPtr->tkfont,
		 scPtr->title, strlen(scPtr->title), x, fm.ascent + bd);
  }

  /* draw the strips */
  if (scPtr->displaybits & CLEAR_NEEDED) {
    scPtr->displaybits &= ~CLEAR_NEEDED;
    for(i=1; i<=scPtr->stripstodisplay; i++)
      DrawStripi(scPtr, i);
  }
  else {
    if (scPtr->stripstodisplay == scPtr->num_strips) {
      if (! scPtr->scrollrequired)
	scPtr->scrollrequired = TRUE;
      else
	ScrollStrips(scPtr);
    }
    DrawStripi(scPtr, scPtr->stripstodisplay);
  }

  /* Display the border */
  if ( scPtr->displaybits & DISPLAY_BORDER ) {
    Tk_Draw3DRectangle(tkwin, Tk_WindowId(tkwin),
		       scPtr->border, 0, 0, Tk_Width(tkwin),
		       Tk_Height(tkwin), scPtr->borderWidth,
		       scPtr->relief);
  }

}

/*
 *--------------------------------------------------------------
 *
 * StripchartEventProc --
 *
 *	This procedure is invoked by the Tk dispatcher on
 *	structure changes to a Stripchart.  For Stripcharts with 3D
 *	borders, this procedure is also invoked for exposures.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	When the window gets deleted, internal structures get
 *	cleaned up.  When it gets exposed, it is redisplayed.
 *
 *--------------------------------------------------------------
 */

static void
StripchartEventProc(clientData, eventPtr)
     ClientData clientData;	/* Information about window. */
     register XEvent *eventPtr;	/* Information about event. */
{
  register Stripchart *scPtr = (Stripchart *) clientData;

  if ((eventPtr->type == Expose) && (eventPtr->xexpose.count == 0)) {
    if ((scPtr->relief != TK_RELIEF_FLAT) &&
	(scPtr->tkwin != NULL) &&
	!(scPtr->displaybits & REDRAW_PENDING)) {
      EventuallyRedrawStripchart(scPtr, DISPLAY_ALL | CLEAR_NEEDED);
      /*  A clear isn't technically needed as the bitmap is clear when the
       *  window is exposed.  This flag is used to indicate that all strips
       *  need to be drawn, not just the most recent one.
       */
    }
  } else if (eventPtr->type == DestroyNotify) {
    Tcl_DeleteCommand(scPtr->interp,
		      Tk_PathName(scPtr->tkwin));
    scPtr->tkwin = NULL;
    if (scPtr->cb_timer)
      Tk_DeleteTimerHandler(scPtr->cb_timer);
    scPtr->cb_timer = NULL;
    scPtr->continue_callback = 0;
    if (scPtr->displaybits & REDRAW_PENDING) {
      Tk_CancelIdleCall(DisplayStripchart, (ClientData) scPtr);
    }
    Tcl_EventuallyFree((ClientData) scPtr, (Tcl_FreeProc *) DestroyStripchart);
  }
}

/*
 *--------------------------------------------------------------
 *
 * SetStripchartValue --
 *
 *	This procedure changes the value of a Stripchart and invokes
 *	a Tcl command to reflect the current position of a Stripchart
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	(Re) Sets the scrollrequired flag.
 *
 *--------------------------------------------------------------
 */

static void
SetStripchartValue(scPtr, value)
     register Stripchart *scPtr;	    /* Info about widget. */
     int value;			/* New value for Stripchart.  Gets
				 * adjusted if it's off the Stripchart. */
{
  int i;
  int std = scPtr->stripstodisplay;
  int maxvalues = scPtr->num_strips;

  if (std != maxvalues) {
    scPtr->value[std] = value;
    scPtr->stripstodisplay++;
  }
  else {
    for (i=0; i<maxvalues-1; i++ )
      scPtr->value[i] = scPtr->value[i+1];
    scPtr->value[maxvalues-1] = value;
  }

  EventuallyRedrawStripchart(scPtr, DISPLAY_STRIPS);
}

/*
 *--------------------------------------------------------------
 *
 * EventuallyRedrawStripchart --
 *
 *	Arrange for part or all of a stripchart widget to redrawn at
 *	the next convenient time in the future.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static void
EventuallyRedrawStripchart(scPtr, displaybits)
     register Stripchart *scPtr;	   /* Information about widget. */
     int displaybits;
{

  if ( (scPtr->tkwin == NULL) || !Tk_IsMapped(scPtr->tkwin)
       || (scPtr->displaybits & REDRAW_PENDING) ) {
    scPtr->displaybits |= displaybits;
    return;
  }

  scPtr->displaybits = displaybits | REDRAW_PENDING;
  if (scPtr->guarentee_draw)
    DisplayStripchart((ClientData)scPtr);
  else
    Tk_DoWhenIdle(DisplayStripchart, (ClientData) scPtr);
}
/*
 *--------------------------------------------------------------
 *
 * SwapColours --
 *
 *     Save the current colour scheme so that it may be
 *     restored at some later stage.
 *
 * Results:
 *     None.
 *
 * Side effects:
 *     Colour scheme is swapped and Stripchart is redisplayed with
 *     the new colour scheme.
 *
 *--------------------------------------------------------------
 */

static void SwapColours(Stripchart *scPtr)
{
  Tk_3DBorder tempb;
  XColor     *tempc;

  /*
   *  Swap a and b using c as the temporary variable.
   *  NOTE:  This only works because the reference counts to the XColors are
   *	   unchanged because the pointers are swapped consistently.
   */
#define SWAP(a,b,c) { (c) = (a); (a) = (b); (b) = (c); }

  SWAP(scPtr->altborder, scPtr->border, tempb);
  SWAP(scPtr->altstripBorder, scPtr->stripBorder, tempb);

  SWAP(scPtr->a_textColor, scPtr->textColorPtr, tempc);
  SWAP(scPtr->a_tickColor, scPtr->tickColorPtr, tempc);

#undef SWAP

  ConfigureStripchart(scPtr->interp, scPtr, 0, NULL,
		      TK_CONFIG_ARGV_ONLY);
}

/*
 *--------------------------------------------------------------
 *
 * ReplaceColours --
 *
 *     Store the current colour scheme and replace it with
 *     the new one.
 *
 * Results:
 *     None.
 *
 * Side effects:
 *     Stripchart is displayed with the new colour scheme.
 *
 *--------------------------------------------------------------
 */

static void ReplaceColours(Stripchart *scPtr, int argc, char *argv[])
{
  scPtr->altborder = Tk_Get3DBorder(scPtr->interp,
				    scPtr->tkwin,
				    Tk_NameOf3DBorder(scPtr->border));

  scPtr->altstripBorder = Tk_Get3DBorder(scPtr->interp,
					 scPtr->tkwin,
					 Tk_NameOf3DBorder(scPtr->stripBorder));

  scPtr->a_textColor = Tk_GetColorByValue(scPtr->tkwin,
					  scPtr->textColorPtr);
  
  scPtr->a_tickColor = Tk_GetColorByValue(scPtr->tkwin,
					  scPtr->tickColorPtr);
  

  ConfigureStripchart(scPtr->interp, scPtr, argc, argv,
		      TK_CONFIG_ARGV_ONLY);
}

/*
 *--------------------------------------------------------------
 *
 * DrawStripi --
 *
 *     Draw the i-th strip of the stripchart.
 *
 * Results:
 *     None.
 *
 * Side effects:
 *     A new strip is drawn.
 *
 *--------------------------------------------------------------
 */
void DrawStripi(Stripchart *scPtr, int i)
{
  register Tk_Window tkwin = scPtr->tkwin;
  int x = scPtr->borderWidth + PADDING + (i-1)*scPtr->strip_width;
  int y;
  int w = scPtr->strip_width;
  int h;
  int maxv = scPtr->max_value;
  int minv = scPtr->min_value;
  XSegment ticks[MAX_TICKS];
  int lineheight;
  Tk_FontMetrics fm;

  Tk_GetFontMetrics(scPtr->tkfont, &fm);
  lineheight = fm.ascent + fm.descent;

  y = scPtr->borderWidth + PADDING + hasatitle(scPtr) * (lineheight + PADDING);

  if (i < 1 || i > scPtr->num_strips)
    return;

  /* Clear any strip that might be below this one */
#ifndef WIN32
  XClearArea(Tk_Display(tkwin), Tk_WindowId(tkwin), x, y, w,
	     scPtr->max_height, FALSE);
#else
  Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin),
		     scPtr->border, x, y, w, scPtr->max_height,
		     0, TK_RELIEF_FLAT);
#endif

  /* Calculate the height of the bar */
  h = scPtr->max_height * (scPtr->value[i-1] - minv) / (maxv - minv);
  if ( h > scPtr->max_height )
    h = scPtr->max_height;
  if ( h == 0 )
    h = 1;

  /* Adject the origin of the bars rectangle depending on its origin */
  if (scPtr->grow_up)
    y += (scPtr->max_height - h);

  /* Draw the sucker */
  Tk_Fill3DRectangle(tkwin, Tk_WindowId(tkwin),
		     scPtr->stripBorder, x, y, w, h,
		     scPtr->stripBorderWidth,
		     scPtr->stripRelief);

  /* Draw the ticks */
  if (scPtr->showticks) {
    for ( i=0; i<scPtr->num_ticks; i++ ) {
      ticks[i].x1 = x;
      ticks[i].x2 = x + scPtr->strip_width - 1;
      ticks[i].y1 = ticks[i].y2 = scPtr->borderWidth + PADDING +
	hasatitle(scPtr)*(lineheight + PADDING) +
	i*scPtr->max_height /(scPtr->num_ticks-1);
    }
  }
  XDrawSegments(Tk_Display(tkwin), Tk_WindowId(tkwin), scPtr->tickGC, ticks,
		scPtr->num_ticks);
}

/*
 *--------------------------------------------------------------
 *
 * ScrollStrips --
 *
 *     Scroll the strips in the stripchart region to the left
 *     by the width of a strip pixels.
 *
 * Results:
 *     None.
 *
 * Side effects:
 *     Display changes.
 *
 *--------------------------------------------------------------
 */
void ScrollStrips(Stripchart *scPtr)
{
  register Tk_Window tkwin = scPtr->tkwin;
  int src_x = scPtr->borderWidth + PADDING + scPtr->strip_width;
  int src_y;
  int dest_x = src_x - scPtr->strip_width;
  int dest_y;
  int w = (scPtr->num_strips - 1) * scPtr->strip_width;
  int h = scPtr->max_height;
  int lineheight;
  Tk_FontMetrics fm;

  Tk_GetFontMetrics(scPtr->tkfont, &fm);
  lineheight = fm.ascent + fm.descent;

  src_y = scPtr->borderWidth + PADDING +
    hasatitle(scPtr) * (lineheight + PADDING);
  dest_y = src_y;

  XCopyArea(Tk_Display(tkwin), Tk_WindowId(tkwin), Tk_WindowId(tkwin),
	    /*	  drawable src	 ,    drawable dest    */
	    Tk_GetGC(tkwin, 0, NULL), src_x, src_y, w, h, dest_x, dest_y);
}
