# combobox.tcl --
#
# This file defines the default bindings for Tk combobox widgets and provides
# procedures that help in implementing those bindings.
#
# RCS: @(#) $Id: combobox.tcl,v 1.1.1.1 2000/07/27 00:06:46 de Exp $
#
# Copyright (c) 1992-1994 The Regents of the University of California.
# Copyright (c) 1994-1997 Sun Microsystems, Inc.
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#

#-------------------------------------------------------------------------
# Elements of tkPriv that are used in this file:
#
# afterId -		If non-null, it means that auto-scanning is underway
#			and it gives the "after" id for the next auto-scan
#			command to be executed.
# mouseMoved -		Non-zero means the mouse has moved a significant
#			amount since the button went down (so, for example,
#			start dragging out a selection).
# pressX -		X-coordinate at which the mouse button was pressed.
# selectMode -		The style of selection currently underway:
#			char, word, or line.
# x, y -		Last known mouse coordinates for scanning
#			and auto-scanning.
# data -		Used for Cut and Copy
#-------------------------------------------------------------------------

#-------------------------------------------------------------------------
# The code below creates the default class bindings for entries.
#-------------------------------------------------------------------------
bind Combobox <<Cut>> {
    if {![catch {tkComboboxGetSelection %W} tkPriv(data)]} {
	clipboard clear -displayof %W
	clipboard append -displayof %W $tkPriv(data)
	%W delete sel.first sel.last
	unset tkPriv(data)
    }
}
bind Combobox <<Copy>> {
    if {![catch {tkComboboxGetSelection %W} tkPriv(data)]} {
	clipboard clear -displayof %W
	clipboard append -displayof %W $tkPriv(data)
	unset tkPriv(data)
    }
}
bind Combobox <<Paste>> {
    global tcl_platform
    catch {
	if {[string compare $tcl_platform(platform) "unix"]} {
	    catch {
		%W delete sel.first sel.last
	    }
	}
	%W insert insert [selection get -displayof %W -selection CLIPBOARD]
	tkComboboxSeeInsert %W
    }
}
bind Combobox <<Clear>> {
    %W delete sel.first sel.last
}
bind Combobox <<PasteSelection>> {
    if {!$tkPriv(mouseMoved) || $tk_strictMotif} {
	tkComboboxPaste %W %x
    }
}

# Standard Motif bindings:

bind Combobox <1> {
    tkComboboxButton1 %W %x %y
}
bind Combobox <B1-Motion> {
    set tkPriv(x) %x
    tkComboboxMouseSelect %W %x ignore
}
bind Combobox <Double-1> {
    set tkPriv(selectMode) word
    tkComboboxMouseSelect %W %x sel.first
}
bind Combobox <Triple-1> {
    set tkPriv(selectMode) line
    tkComboboxMouseSelect %W %x 0
}
bind Combobox <Shift-1> {
    set tkPriv(selectMode) char
    %W selection adjust @%x
}
bind Combobox <Double-Shift-1> {
    set tkPriv(selectMode) word
    tkComboboxMouseSelect %W %x
}
bind Combobox <Triple-Shift-1> {
    set tkPriv(selectMode) line
    tkComboboxMouseSelect %W %x
}
bind Combobox <B1-Leave> {
    set tkPriv(x) %x
    tkComboboxAutoScan %W
}
bind Combobox <B1-Enter> {
    tkCancelRepeat
}
bind Combobox <ButtonRelease-1> {
    tkCancelRepeat
    if {[info exists tkPriv(repeated)] && !$tkPriv(repeated)} {
	%W invoke [%W selection element]
    }
    %W selection element none
}
bind Combobox <Control-1> {
    %W icursor @%x
}

bind Combobox <Left> {
    tkComboboxSetCursor %W [expr {[%W index insert] - 1}]
}
bind Combobox <Right> {
    tkComboboxSetCursor %W [expr {[%W index insert] + 1}]
}
bind Combobox <Shift-Left> {
    tkComboboxKeySelect %W [expr {[%W index insert] - 1}]
    tkComboboxSeeInsert %W
}
bind Combobox <Shift-Right> {
    tkComboboxKeySelect %W [expr {[%W index insert] + 1}]
    tkComboboxSeeInsert %W
}
bind Combobox <Control-Left> {
    tkComboboxSetCursor %W [tkComboboxPreviousWord %W insert]
}
bind Combobox <Control-Right> {
    tkComboboxSetCursor %W [tkComboboxNextWord %W insert]
}
bind Combobox <Shift-Control-Left> {
    tkComboboxKeySelect %W [tkComboboxPreviousWord %W insert]
    tkComboboxSeeInsert %W
}
bind Combobox <Shift-Control-Right> {
    tkComboboxKeySelect %W [tkComboboxNextWord %W insert]
    tkComboboxSeeInsert %W
}
bind Combobox <Home> {
    tkComboboxSetCursor %W 0
}
bind Combobox <Shift-Home> {
    tkComboboxKeySelect %W 0
    tkComboboxSeeInsert %W
}
bind Combobox <End> {
    tkComboboxSetCursor %W end
}
bind Combobox <Shift-End> {
    tkComboboxKeySelect %W end
    tkComboboxSeeInsert %W
}

bind Combobox <Delete> {
    if {[%W selection present]} {
	%W delete sel.first sel.last
    } else {
	%W delete insert
    }
}
bind Combobox <BackSpace> {
    tkComboboxBackspace %W
}

bind Combobox <Control-space> {
    %W selection from insert
}
bind Combobox <Select> {
    %W selection from insert
}
bind Combobox <Control-Shift-space> {
    %W selection adjust insert
}
bind Combobox <Shift-Select> {
    %W selection adjust insert
}
bind Combobox <Control-slash> {
    %W selection range 0 end
}
bind Combobox <Control-backslash> {
    %W selection clear
}
bind Combobox <KeyPress> {
    tkComboboxInsert %W %A
}

# Ignore all Alt, Meta, and Control keypresses unless explicitly bound.
# Otherwise, if a widget binding for one of these is defined, the
# <KeyPress> class binding will also fire and insert the character,
# which is wrong.  Ditto for Escape, Return, and Tab.

bind Combobox <Alt-KeyPress> {# nothing}
bind Combobox <Meta-KeyPress> {# nothing}
bind Combobox <Control-KeyPress> {# nothing}
bind Combobox <Escape> {# nothing}
bind Combobox <Return> {# nothing}
bind Combobox <KP_Enter> {# nothing}
bind Combobox <Tab> {# nothing}
if {[string equal $tcl_platform(platform) "macintosh"]} {
	bind Combobox <Command-KeyPress> {# nothing}
}

# On Windows, paste is done using Shift-Insert.  Shift-Insert already
# generates the <<Paste>> event, so we don't need to do anything here.
if {[string compare $tcl_platform(platform) "windows"]} {
    bind Combobox <Insert> {
	catch {tkComboboxInsert %W [selection get -displayof %W]}
    }
}

# Additional emacs-like bindings:

bind Combobox <Control-a> {
    if {!$tk_strictMotif} {
	tkComboboxSetCursor %W 0
    }
}
bind Combobox <Control-b> {
    if {!$tk_strictMotif} {
	tkComboboxSetCursor %W [expr {[%W index insert] - 1}]
    }
}
bind Combobox <Control-d> {
    if {!$tk_strictMotif} {
	%W delete insert
    }
}
bind Combobox <Control-e> {
    if {!$tk_strictMotif} {
	tkComboboxSetCursor %W end
    }
}
bind Combobox <Control-f> {
    if {!$tk_strictMotif} {
	tkComboboxSetCursor %W [expr {[%W index insert] + 1}]
    }
}
bind Combobox <Control-h> {
    if {!$tk_strictMotif} {
	tkComboboxBackspace %W
    }
}
bind Combobox <Control-k> {
    if {!$tk_strictMotif} {
	%W delete insert end
    }
}
bind Combobox <Control-t> {
    if {!$tk_strictMotif} {
	tkComboboxTranspose %W
    }
}
bind Combobox <Meta-b> {
    if {!$tk_strictMotif} {
	tkComboboxSetCursor %W [tkComboboxPreviousWord %W insert]
    }
}
bind Combobox <Meta-d> {
    if {!$tk_strictMotif} {
	%W delete insert [tkComboboxNextWord %W insert]
    }
}
bind Combobox <Meta-f> {
    if {!$tk_strictMotif} {
	tkComboboxSetCursor %W [tkComboboxNextWord %W insert]
    }
}
bind Combobox <Meta-BackSpace> {
    if {!$tk_strictMotif} {
	%W delete [tkComboboxPreviousWord %W insert] insert
    }
}
bind Combobox <Meta-Delete> {
    if {!$tk_strictMotif} {
	%W delete [tkComboboxPreviousWord %W insert] insert
    }
}

# A few additional bindings of my own.

bind Combobox <2> {
    if {!$tk_strictMotif} {
	%W scan mark %x
	set tkPriv(x) %x
	set tkPriv(y) %y
	set tkPriv(mouseMoved) 0
    }
}
bind Combobox <B2-Motion> {
    if {!$tk_strictMotif} {
	if {abs(%x-$tkPriv(x)) > 2} {
	    set tkPriv(mouseMoved) 1
	}
	%W scan dragto %x
    }
}

# tkComboboxInvoke --
# Invoke an element of the combobox
#
# Arguments:
# w -		The combobox window.
# elem -	Element to invoke

proc tkComboboxInvoke {w elem} {
    global tkPriv

    $w invoke $elem
    incr tkPriv(repeated)
    set delay [$w cget -repeatinterval]
    if {$delay > 0} {
	set tkPriv(afterId) [after $delay [list tkComboboxInvoke $w $elem]]
    }
}

# tkComboboxClosestGap --
# Given x and y coordinates, this procedure finds the closest boundary
# between characters to the given coordinates and returns the index
# of the character just after the boundary.
#
# Arguments:
# w -		The combobox window.
# x -		X-coordinate within the window.

proc tkComboboxClosestGap {w x} {
    set pos [$w index @$x]
    set bbox [$w bbox $pos]
    if {($x - [lindex $bbox 0]) < ([lindex $bbox 2]/2)} {
	return $pos
    }
    incr pos
}

# tkComboboxButton1 --
# This procedure is invoked to handle button-1 presses in combobox
# widgets.  It moves the insertion cursor, sets the selection anchor,
# and claims the input focus.
#
# Arguments:
# w -		The combobox window in which the button was pressed.
# x -		The x-coordinate of the button press.

proc tkComboboxButton1 {w x y} {
    global tkPriv

    set tkPriv(element) [$w identify $x $y]
    switch -exact $tkPriv(element) {
	"spinup" - "spindown" -	"drop" {
	    if {[string compare "disabled" [$w cget -state]]} {
		$w selection element $tkPriv(element)
		set tkPriv(repeated) 0
		after cancel $tkPriv(afterId)
		set delay [$w cget -repeatdelay]
		if {$delay > 0} {
		    set tkPriv(afterId) [after $delay \
			    [list tkComboboxInvoke $w $tkPriv(element)]]
		}
	    }
	}
	"entry" {
	    set tkPriv(selectMode) char
	    set tkPriv(mouseMoved) 0
	    set tkPriv(pressX) $x
	    $w icursor [tkComboboxClosestGap $w $x]
	    $w selection from insert
	    if {[string equal [$w cget -state] "normal"]} {focus $w}
	    $w selection clear
	}
	default {
	    return -code error "unknown combobox element \"[$w identify $x $y]\""
	}
    }
}

# tkComboboxMouseSelect --
# This procedure is invoked when dragging out a selection with
# the mouse.  Depending on the selection mode (character, word,
# line) it selects in different-sized units.  This procedure
# ignores mouse motions initially until the mouse has moved from
# one character to another or until there have been multiple clicks.
#
# Arguments:
# w -		The combobox window in which the button was pressed.
# x -		The x-coordinate of the mouse.
# cursor -	optional place to set cursor.

proc tkComboboxMouseSelect {w x {cursor {}}} {
    global tkPriv

    if {[string compare "entry" $tkPriv(element)]} {
	if {[string compare "none" $tkPriv(element)] && \
		[string compare "ignore" $cursor]} {
	    $w selection element $tkPriv(element)
	    $w invoke $tkPriv(element)
	    $w selection element none
	}
	return
    }
    set cur [tkComboboxClosestGap $w $x]
    set anchor [$w index anchor]
    if {($cur != $anchor) || (abs($tkPriv(pressX) - $x) >= 3)} {
	set tkPriv(mouseMoved) 1
    }
    switch $tkPriv(selectMode) {
	char {
	    if {$tkPriv(mouseMoved)} {
		if {$cur < $anchor} {
		    $w selection range $cur $anchor
		} elseif {$cur > $anchor} {
		    $w selection range $anchor $cur
		} else {
		    $w selection clear
		}
	    }
	}
	word {
	    if {$cur < [$w index anchor]} {
		set before [tcl_wordBreakBefore [$w get] $cur]
		set after [tcl_wordBreakAfter [$w get] [expr {$anchor-1}]]
	    } else {
		set before [tcl_wordBreakBefore [$w get] $anchor]
		set after [tcl_wordBreakAfter [$w get] [expr {$cur - 1}]]
	    }
	    if {$before < 0} {
		set before 0
	    }
	    if {$after < 0} {
		set after end
	    }
	    $w selection range $before $after
	}
	line {
	    $w selection range 0 end
	}
    }
    if {[string compare $cursor {}] && \
	    [string compare $cursor "ignore"]} {
	catch {$w icursor $cursor}
    }
    update idletasks
}

# tkComboboxPaste --
# This procedure sets the insertion cursor to the current mouse position,
# pastes the selection there, and sets the focus to the window.
#
# Arguments:
# w -		The combobox window.
# x -		X position of the mouse.

proc tkComboboxPaste {w x} {
    global tkPriv

    $w icursor [tkComboboxClosestGap $w $x]
    catch {$w insert insert [selection get -displayof $w]}
    if {[string equal [$w cget -state] "normal"]} {focus $w}
}

# tkComboboxAutoScan --
# This procedure is invoked when the mouse leaves an combobox window
# with button 1 down.  It scrolls the window left or right,
# depending on where the mouse is, and reschedules itself as an
# "after" command so that the window continues to scroll until the
# mouse moves back into the window or the mouse button is released.
#
# Arguments:
# w -		The combobox window.

proc tkComboboxAutoScan {w} {
    global tkPriv
    set x $tkPriv(x)
    if {![winfo exists $w]} return
    if {$x >= [winfo width $w]} {
	$w xview scroll 2 units
	tkComboboxMouseSelect $w $x ignore
    } elseif {$x < 0} {
	$w xview scroll -2 units
	tkComboboxMouseSelect $w $x ignore
    }
    set tkPriv(afterId) [after 50 [list tkComboboxAutoScan $w]]
}

# tkComboboxKeySelect --
# This procedure is invoked when stroking out selections using the
# keyboard.  It moves the cursor to a new position, then extends
# the selection to that position.
#
# Arguments:
# w -		The combobox window.
# new -		A new position for the insertion cursor (the cursor hasn't
#		actually been moved to this position yet).

proc tkComboboxKeySelect {w new} {
    if {![$w selection present]} {
	$w selection from insert
	$w selection to $new
    } else {
	$w selection adjust $new
    }
    $w icursor $new
}

# tkComboboxInsert --
# Insert a string into an combobox at the point of the insertion cursor.
# If there is a selection in the combobox, and it covers the point of the
# insertion cursor, then delete the selection before inserting.
#
# Arguments:
# w -		The combobox window in which to insert the string
# s -		The string to insert (usually just a single character)

proc tkComboboxInsert {w s} {
    if {[string equal $s ""]} {
	return
    }
    catch {
	set insert [$w index insert]
	if {([$w index sel.first] <= $insert)
		&& ([$w index sel.last] >= $insert)} {
	    $w delete sel.first sel.last
	}
    }
    $w insert insert $s
    tkComboboxSeeInsert $w
}

# tkComboboxBackspace --
# Backspace over the character just before the insertion cursor.
# If backspacing would move the cursor off the left edge of the
# window, reposition the cursor at about the middle of the window.
#
# Arguments:
# w -		The combobox window in which to backspace.

proc tkComboboxBackspace w {
    if {[$w selection present]} {
	$w delete sel.first sel.last
    } else {
	set x [expr {[$w index insert] - 1}]
	if {$x >= 0} {$w delete $x}
	if {[$w index @0] >= [$w index insert]} {
	    set range [$w xview]
	    set left [lindex $range 0]
	    set right [lindex $range 1]
	    $w xview moveto [expr {$left - ($right - $left)/2.0}]
	}
    }
}

# tkComboboxSeeInsert --
# Make sure that the insertion cursor is visible in the combobox window.
# If not, adjust the view so that it is.
#
# Arguments:
# w -		The combobox window.

proc tkComboboxSeeInsert w {
    set c [$w index insert]
    if {($c < [$w index @0]) || ($c > [$w index @[winfo width $w]])} {
	$w xview $c
    }
}

# tkComboboxSetCursor -
# Move the insertion cursor to a given position in an combobox.  Also
# clears the selection, if there is one in the combobox, and makes sure
# that the insertion cursor is visible.
#
# Arguments:
# w -		The combobox window.
# pos -		The desired new position for the cursor in the window.

proc tkComboboxSetCursor {w pos} {
    $w icursor $pos
    $w selection clear
    tkComboboxSeeInsert $w
}

# tkComboboxTranspose -
# This procedure implements the "transpose" function for combobox widgets.
# It tranposes the characters on either side of the insertion cursor,
# unless the cursor is at the end of the line.  In this case it
# transposes the two characters to the left of the cursor.  In either
# case, the cursor ends up to the right of the transposed characters.
#
# Arguments:
# w -		The combobox window.

proc tkComboboxTranspose w {
    set i [$w index insert]
    if {$i < [$w index end]} {
	incr i
    }
    set first [expr {$i-2}]
    if {$first < 0} {
	return
    }
    set new [string index [$w get] [expr {$i-1}]][string index [$w get] $first]
    $w delete $first $i
    $w insert insert $new
    tkComboboxSeeInsert $w
}

# tkComboboxNextWord --
# Returns the index of the next word position after a given position in the
# combobox.  The next word is platform dependent and may be either the next
# end-of-word position or the next start-of-word position after the next
# end-of-word position.
#
# Arguments:
# w -		The combobox window in which the cursor is to move.
# start -	Position at which to start search.

if {[string equal $tcl_platform(platform) "windows"]}  {
    proc tkComboboxNextWord {w start} {
	set pos [tcl_endOfWord [$w get] [$w index $start]]
	if {$pos >= 0} {
	    set pos [tcl_startOfNextWord [$w get] $pos]
	}
	if {$pos < 0} {
	    return end
	}
	return $pos
    }
} else {
    proc tkComboboxNextWord {w start} {
	set pos [tcl_endOfWord [$w get] [$w index $start]]
	if {$pos < 0} {
	    return end
	}
	return $pos
    }
}

# tkComboboxPreviousWord --
#
# Returns the index of the previous word position before a given
# position in the combobox.
#
# Arguments:
# w -		The combobox window in which the cursor is to move.
# start -	Position at which to start search.

proc tkComboboxPreviousWord {w start} {
    set pos [tcl_startOfPreviousWord [$w get] [$w index $start]]
    if {$pos < 0} {
	return 0
    }
    return $pos
}

# tkComboboxGetSelection --
#
# Returns the selected text of the combobox with respect to the -show option.
#
# Arguments:
# w -         The combobox window from which the text to get

proc tkComboboxGetSelection {w} {
    set comboboxString [string range [$w get] [$w index sel.first] \
	    [expr {[$w index sel.last] - 1}]]
    if {[string compare [$w cget -show] ""]} {
	regsub -all . $comboboxString [string index [$w cget -show] 0] comboboxString
    }
    return $comboboxString
}

# tkComboboxSpinCmd --
#
# Returns the selected text of the combobox with respect to the -show option.
#
# Arguments:
# w -         The combobox window from which the text to get

proc tkComboboxSpinCmd {w which} {
    if {[string equal "" [$w cget -mode]]} {
	return
    }
    if {[string compare "range" [$w cget -mode]]} {
	return
    }
    set val [$w get]
    if {[string is double $val]} {
	$w delete 0 end
	set res [$w cget -resolution]
	if {[string equal "up" $which]} {
	    set val [expr {$val + $res}]
	    if {$val > [$w cget -to]} {
		set val [$w cget -from]
	    }
	} else {
	    set val [expr {$val - $res}]
	    if {$val < [$w cget -from]} {
		set val [$w cget -to]
	    }
	}
	$w insert 0 $val
    }
}

# tkComboboxDropCmd --
#
# Returns the selected text of the combobox with respect to the -show option.
#
# Arguments:
# w -         The combobox window from which the text to get

proc tkComboboxDropCmd {w} {
    if {[string equal "" [$w cget -mode]]} {
	return
    }
    puts "drop combobox box"
}
