#!/bin/sh
#\
exec /opt/tcl803/bin/tcl "$0"
# ----------------------------------------------------------------------
#  PURPOSE: Processes incoming mail messages to receive reviews
#    for papers, panels, demos and posters.  Registers each review
#    in the database and sends e-mail to the reviewer.
# ----------------------------------------------------------------------
#
# where are the incoming files spooled?
#
set spooldir /lopt/apache/u/de/Tcl2K/rcv
set copydir /lopt/apache/u/de/Tcl2K/pro
set logdir /lopt/apache/u/de/Tcl2K/log
#
set files2process [glob -nocomplain $spooldir/Incoming.*]
if {$files2process == ""} {exit 0}
#
# if there are any files to process, continue!
#
# go to the central location
cd /lopt/apache/cgi-bin/Tcl2K
source cgi.tcl
source common.tcl
source tclPaperslib.tcl
#
package require Tclgdbm
#
#
proc forward {addr subject explain mesg} {
    set fid [open "| /usr/lib/sendmail -oi -t" "w"]
    puts $fid "To: $addr"
    puts $fid "From: tcl2k <tcl2k@ucolick.org>"
    puts $fid "Subject: $subject"
    puts $fid "Date: [clock format [clock seconds]]"
    puts $fid ""  ;# sendmail terminates header with blank line
    puts $fid $explain
    puts $fid ""
    puts $fid "--> $TclChairs"
    puts $fid "--> $TclConf"
    puts $fid "------------------------------------------------------------"
    foreach line [split $mesg "\n"] {
        puts $fid "> $line"
    }
    close $fid
}

foreach rf $files2process {

set fn [file tail $rf]
global ofp
set err [catch {set ofp [open $logdir/$fn.log w]} res]
if {$err} {
	puts stderr "OUCH cannot open $logdir/$fn.log\n$res"
}

#
# Read the message from stdin and see if it looks like a review.
# If not, forward elsewhere for consideration.
#
set mesg [read_file $rf]

if {![regexp "REVIEWER:.+TITLE:.+SCORE:.+MULTIPLIER" $mesg] &&
    ![regexp "REVIEWER:.+TITLE:.+SCORE:.+WEIGHT" $mesg]} {
    catch {
        forward $WebMaster "please follow up" \
            "The following message was received by tclPaperspc, but it
doesn't match the review format for the program committee.
Please follow up with the sender." $mesg
    }
    exit
}

set reviewer ""
regexp "REVIEWER: *(\[^\n\]+)\n" $mesg match reviewer

puts $ofp "Reviewer for $rf is $reviewer"

set db [gdbm open $DataFile rwc]

if {![gdbm exists $db "pc-$reviewer-email"]} {
    catch {
        forward $WebMaster "please follow up" \
            "The following message was received by tclPaperspc, and
it looks like a review.  But the reviewer name \"$reviewer\"
is not recognized.  Please follow up with the sender." $mesg
    }
    exit
}

set email [gdbm fetch $db "pc-$reviewer-email"]
set email $WebMaster

#
# Scan through the message and process reviews.
#
set processed ""  ;# titles for all reviews processed
set problems ""   ;# no problems yet

set form $mesg
while {[regexp "(.+)(-+\n *TITLE:.+SCORE:.+)" $form match form review]} {
    set review [string trimright $review "-"]

    set title ""
    regexp "TITLE: *(\[^\n\]+)\n.+SCORE:" $review match title

	puts $ofp "Title of reviewed paper:\n\t$title"

    set all [gdbm fetch $db "submissions"]
    set found 0
    foreach paper $all {
        set t [gdbm fetch $db "$paper-title"]
  	puts $ofp "    Compare to $paper $t"
        if {[string compare $t $title] == 0} {
	    puts $ofp "MATCH"
            set found 1
            break
        } else {
		regsub -all "  " $t " " t
		regsub -all "  " $title " " title
		puts $ofp "    OK, singlify all double spaces and try again"
		puts $ofp "    Compare $paper $title (reviewer)"
		puts $ofp "    Compare $paper $t (database)"
		if {[string compare $t $title] == 0} {
	    	puts $ofp "MATCH"
            	set found 1
            	break
		}
	}
    }

    if {!$found} {
        lappend problems "title \"$title\" not recognized"
        continue
    }

    if {![regexp "TITLE:.+\n *SCORE: *(\[^\n\]+)" $review match score]} {
        lappend problems "can't pick out score for \"$title\""
        continue
    }
    set score [string trim $score " _"]

    if {[catch {expr 1*$score}] != 0 || $score < -1 || $score > 4} {
        lappend problems "bad score \"$score\" for \"$title\""
        continue
    }

    if {[regexp "TITLE:.+SCORE:.+\n *MULTIPLIER: *(\[^\n\]+)" \
            $review match multiplier] ||
        [regexp "TITLE:.+SCORE:.+\n *WEIGHT: *(\[^\n\]+)" \
            $review match multiplier]} {
        set multiplier [string trim $multiplier " _"]
    } else {
        lappend problems "can't pick out multiplier weight for \"$title\""
        continue
    }

    if {[catch {expr 1*$multiplier}] != 0 || $multiplier < 0 || $multiplier > 3} {
        lappend problems "bad multiplier weight \"$multiplier\" for \"$title\""
        continue
    }

    set head(strengths) "SIGNIFICANT STRENGTHS OF THIS SUBMISSION:"
    set head(weaknesses) "SIGNIFICANT WEAKNESSES OF THIS SUBMISSION:"
#	This one is a problem
    set head(bottomline) "BOTTOM LINE: \\(why \[^\n\]+"
    set head(comments) "OTHER NOTES TO THE AUTHOR:\n *\\(\[^\n\]+"
    set head(private) "PRIVATE COMMENTS TO OTHER PC MEMBERS:\n\[^\n\]+\n\[^\n\]+"

    if {![regexp "$head(strengths)\n(.+)$head(weaknesses)" \
        $review match strengths]} {
        lappend problems "can't pick out STRENGTHS for \"$title\""
        continue
    }


    if {![regexp "$head(weaknesses)\n(.+)$head(bottomline)" \
        $review match weaknesses]} {
        lappend problems "can't pick out WEAKNESSES for \"$title\""
        continue
    }


    if {![regexp "$head(bottomline)\n(.+)$head(comments)" \
        $review match bottomline]} {
        lappend problems "can't pick out BOTTOM LINE for \"$title\""
        continue
    }


    if {![regexp "$head(comments)\n(.+)$head(private)" \
        $review match comments]} {
        lappend problems "can't pick out NOTES TO AUTHOR for \"$title\""
        continue
    }


    if {![regexp "$head(private)\n(.*)" \
        $review match private]} {
        lappend problems "can't pick out PRIVATE COMMENTS for \"$title\""
        continue
    }

    foreach var {strengths weaknesses bottomline comments private} {
        regexp "^ *\\.\\.\\.put your comments here\\.\\.\\.\n(.*)" \
            [set $var] match $var
        set $var [string trim [set $var]]
    }

    gdbm close $db  ;# hack! have to close db before calling these procs

    puts $ofp "Check whether $reviewer has already reviewed $paper"
    if {[TP_reviewed $paper $reviewer]} {
	puts $ofp "    YES, delete it now with review_del call"
        set err [catch {TP_review_del $paper $reviewer} res]

	if {$err} {
		puts $ofp "OUCH couldn't delete $res"
	}

    } else {
	puts $ofp "    NO, $review has not previously reviewed it."
    }

#	reopen db

    set db [gdbm open $DataFile rwc]


    #
    # Add review to the database.
    #
    lappend processed $title

    set id [gdbm fetch $db "identifier"]

    if {"" == $id} {set id 0}
    incr id
    gdbm store $db "identifier" $id
    set token "review$id"

    gdbm store $db "$token-title" $title
    gdbm store $db "$token-reviewer" $reviewer
    gdbm store $db "$token-score" $score
    gdbm store $db "$token-multiplier" $multiplier
    gdbm store $db "$token-strengths" $strengths
    gdbm store $db "$token-weaknesses" $weaknesses
    gdbm store $db "$token-bottomline" $bottomline
    gdbm store $db "$token-comments" $comments
    gdbm store $db "$token-private" $private

    set htmlfile "$HTMLDIR/reviews/$token.html"
    set filepath "/reviews/$token.html"
    gdbm store $db "$token-html" $filepath

    #
    # Register review and compute average score.
    #
    if {"" != $paper} {
        set all [gdbm fetch $db "$paper-reviews"]
        lappend all $token
        gdbm store $db "$paper-reviews" $all

        set total 0
        set mtotal 0
        foreach rev $all {
            set sval [gdbm fetch $db "$rev-score"]
            set mval [gdbm fetch $db "$rev-multiplier"]
            if {"" != $sval} {
                set total [expr $total + $sval*$mval]
                set mtotal [expr $mtotal + $mval]
            }
        }
        if {$mtotal == 0} {
            set avg ""
        } else {
            set avg [format "%.2f" [expr double($total)/$mtotal]]
        }
        gdbm store $db "$paper-score" $avg
    }

    #
    # Save HTML for review.
    #
    set fid [open "reviews/template.html" r]
    set body [read $fid]
    close $fid

    foreach field {strengths weaknesses bottomline comments private} {
        set $field [cgi_quote_html [set $field]]
        regsub -all "\n\[ \t\]*\n" [set $field] "\n<p>" $field
    }

    foreach field {
        title reviewer score multiplier
        strengths weaknesses bottomline comments private
    } {
        set replace [set $field]
        regsub -all "&" $replace "\\\\&" replace
        regsub -all "@[string toupper $field]@" $body $replace body
    }

    set fid [open $htmlfile w]
    puts $fid $body
    close $fid
}
gdbm close $db

if {"" != $email} {
    set body "
To $reviewer

Thanks for sending reviews of material for the $TclConf!
This note is your record that the reviews have been
received and processed.
"
    append body "
We have processed reviews for the following titles:
"
    foreach title $processed {
        append body "  - $title\n"
    }

    if {[llength $problems] == 0} {
        append body "
You should be able to see your reviews on the web site for the
program committee:  $pcPage
"
    } else {
        append body "
We encountered some problems while processing these reviews:
"
        foreach err $problems {
            append body "  >> $err\n"
        }
        append body "
Please fix your review form and submit it again.

If we had trouble picking out scores or comments, you may have
accidentally edited areas of the form that should have been left
intact.  In that case, you should download a new review form that
covers the papers that we couldn't process and submit your comments
on that form."
    }

    catch {forward $email "$TclNick :  received your reviews" $body $mesg}
}

# When all done, move the incoming review out of the way
# to an archive directory

lassign [split [file tail $rf] .] junk uniq
system "mv $rf $copydir/Processed.$uniq"

close $ofp
}
#
